<?php

namespace Bongo\Image\Services;

use Bongo\Framework\Helpers\File;
use Bongo\Image\Models\Image;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Http\UploadedFile;

class AvatarImage
{
    protected UploadedFile $file;

    protected Authenticatable $user;

    public function __construct(Authenticatable $user, UploadedFile $file)
    {
        $this->user = $user;
        $this->file = $file;
    }

    public function save(): Authenticatable
    {
        // Delete the existing image and file as only need one
        $this->delete();

        // Create the filename
        $ext = $this->file->getClientOriginalExtension();
        $name = rtrim($this->file->getClientOriginalName(), $ext);
        $size = $this->file->getSize();
        $fileName = File::generateName($name, $ext);

        // Upload the image to the directory
        $this->file->storePubliclyAs($this->getAvatarPath(), $fileName);

        // Create a new database image
        $dbImage = new Image();
        $dbImage->name = $fileName;
        $dbImage->title = $this->user->name;
        $dbImage->path = $this->getAvatarPath();
        $dbImage->size = $size;
        $dbImage->ext = $ext;
        $dbImage->type = Image::AVATAR;
        $dbImage->created_by = user() ? user()->id : null;
        $dbImage->updated_by = user() ? user()->id : null;

        // Set the dimensions
        [$width, $height] = getimagesize($dbImage->getFileSrc());
        $dbImage->width = $width;
        $dbImage->height = $height;
        $dbImage->orientation = ($width > $height) ? Image::LANDSCAPE : Image::PORTRAIT;

        // Save the new image
        $this->user->images()->save($dbImage);

        return $this->user;
    }

    protected function delete(): void
    {
        $images = $this->user->images()->where('type', Image::AVATAR)->get();

        if ($images) {
            foreach ($images as $image) {
                if ($image->fileExists()) {
                    $image->deleteFile();
                }
                $image->forceDelete();
            }
        }
    }

    private function getAvatarPath(): string
    {
        return config('image.public_path').'avatars/';
    }
}
