<?php

namespace Bongo\Image\Traits;

use Bongo\Image\Models\Image;
use Bongo\Image\Services\ImagePlaceholder;
use Illuminate\Database\Eloquent\Relations\MorphToMany;

trait HasImages
{
    public function images(): MorphToMany
    {
        return $this
            ->morphToMany(Image::class, 'imageable')
            ->withPivot(['sort_order'])
            ->orderBy('sort_order', 'ASC');
    }

    public function hasImages(): bool
    {
        $this->loadMissing('images');

        return ! is_null($this->images) && count($this->images) > 0;
    }

    public function primaryImage(): ?Image
    {
        // Check if the relation has already been eager loaded before doing another database call
        if ($this->relationLoaded('images')) {
            return $this->images
                ->sortBy('type')
                ->sortBy('sort_order')
                ->first();
        }

        return $this->images()
            ->orderBy('type')
            ->orderBy('sort_order')
            ->first();
    }

    public function hasPrimaryImage(): bool
    {
        return ! empty($this->primaryImage());
    }

    public function getPrimaryImage(array $args): string
    {
        if ($this->hasImages()) {
            return route('frontend.image.show', array_merge(
                ['image' => $this->primaryImage()], $args
            ));
        }

        return (new ImagePlaceholder($args))->url();
    }

    public function secondaryImage(): ?Image
    {
        // Check if the relation has already been eager loaded before doing another database call
        if ($this->relationLoaded('images')) {
            return $this->images
                ->skip(1)
                ->first();
        }

        return $this->images()
            ->offset(1)
            ->first();
    }

    public function hasSecondaryImage(): bool
    {
        return ! empty($this->secondaryImage());
    }

    public function getSecondaryImage(array $args): string
    {
        if ($this->images && count($this->images) > 1) {
            return route('frontend.image.show', array_merge(
                ['image' => $this->secondaryImage()], $args
            ));
        }

        return (new ImagePlaceholder($args))->url();
    }

    public function thumbnails(int $limit = 12)
    {
        // Check if the relation has already been eager loaded before doing another database call
        if ($this->relationLoaded('images')) {
            return $this->images->skip(1)->take($limit)->all();
        }

        return $this->images()->offset(1)->limit($limit)->get();
    }

    public function getThumbnail(int $offset): string
    {
        // Check if the relation has already been eager loaded before doing another database call
        if ($this->relationLoaded('images') && $image = $this->images->skip($offset)->first()) {
            return route('frontend.image.show', array_merge(
                ['image' => $image], ['preset' => 'thumb']
            ));
        }

        // Otherwise try and get it from the database
        if ($image = $this->images()->offset($offset)->first()) {
            return route('frontend.image.show', array_merge(
                ['image' => $image], ['preset' => 'thumb']
            ));
        }

        // Else just return the placeholder
        return (new ImagePlaceholder(['preset' => 'thumb']))->url();
    }

    public function getPlaceholder(array $args): string
    {
        return (new ImagePlaceholder($args))->url();
    }
}
