# Bongo Image Package - Claude Code Guidance

## Overview

Laravel package providing comprehensive image management including uploads, on-the-fly manipulation with caching, and polymorphic relationships. Built on Intervention Image and extends `bongo/framework`.

**For detailed documentation, see:**
- [ARCHITECTURE.md](ARCHITECTURE.md) - Complete architecture, diagrams, and data flow
- [.cursorrules](.cursorrules) - Cursor AI development guide
- [.github/copilot-instructions.md](.github/copilot-instructions.md) - GitHub Copilot templates

## Requirements

- PHP 8.2+
- Laravel 10+
- `bongo/framework` ^3.0
- `intervention/image` ^2.6

## Commands

### Development

```bash
# Install dependencies
composer install

# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint

# Update dependencies
rm -f composer.lock && composer update -W
```

### Artisan Commands

```bash
# Update missing image dimensions
php artisan image:update_dimensions
```

## Quick Architecture Reference

### Service Provider

Extends `Bongo\Framework\Providers\AbstractServiceProvider` which automatically:
- Loads config from `src/Config/image.php`
- Registers routes from `src/Routes/` with appropriate middleware
- Registers views under `image::` namespace
- Loads migrations and translations
- Registers commands

```php
class ImageServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'image';

    protected array $commands = [
        UpdateDimensionsCommand::class,
    ];
}
```

### Polymorphic Relationships

Models use `HasImages` trait for polymorphic image relationships:

```php
use Bongo\Image\Traits\HasImages;

class Post extends Model
{
    use HasImages;
}

// Usage:
$post->images();                              // MorphToMany relationship
$post->primaryImage();                        // First image
$post->getPrimaryImage(['preset' => 'large']); // URL with preset
$post->getPlaceholder(['w' => 800]);          // Placeholder URL
```

### Image Types

Four predefined types in `Image` model:

| Type | Constant | Usage | Behaviour |
|------|----------|-------|-----------|
| Avatar | `Image::AVATAR` | User avatars | Single (replaces existing) |
| Cover Image | `Image::COVER_IMAGE` | Cover images | Single (replaces existing) |
| Upload | `Image::UPLOAD` | Standard uploads | Multiple allowed |
| WYSIWYG | `Image::WYSIWYG` | Editor images | Multiple allowed |

### Service Classes

| Service | Purpose | Pattern |
|---------|---------|---------|
| `ImageManipulator` | Resize/crop/fit images | Fluent interface |
| `ImagePlaceholder` | Generate default placeholders | Constructor args |
| `AvatarImage` | Single avatar upload | Replaces existing |
| `CoverImage` | Single cover upload | Replaces existing |
| `WysiwygImage` | Multi-image upload | Appends to collection |
| `BuilderService` | Extract base64 from HTML | Process and replace |

### Route Patterns

```
Frontend (Public Image Delivery):
  GET /photos/{image}?preset=large&q=80&mode=resize
  Route name: frontend.image.show

Backend (Admin Panel):
  GET    /admin/images                → backend.image.index
  POST   /admin/images/store          → backend.image.store
  POST   /admin/images/copy           → backend.image.copy (base64)
  POST   /admin/images/{image}/update → backend.image.update
  DELETE /admin/images/{image}/delete → backend.image.destroy

API (Authenticated):
  GET    /api/images                  → api.image.index
  POST   /api/images/store            → api.image.store
  POST   /api/images/update           → api.image.update
  POST   /api/images/delete           → api.image.destroy
```

## Key Files

| File | Purpose | Key Methods/Features |
|------|---------|---------------------|
| `Models/Image.php` | Core image model | File operations, cache path generation, dimensions |
| `Traits/HasImages.php` | Polymorphic relationships | primaryImage(), getPrimaryImage(), thumbnails() |
| `Services/ImageManipulator.php` | Image manipulation | resize(), crop(), fit(), stream() |
| `Controllers/Frontend/ImageController.php` | Public delivery | On-demand manipulation with caching |
| `Services/BuilderService.php` | Base64 extraction | Extract and store base64 images from HTML |
| `Config/image.php` | Configuration | Paths, presets, quality settings |

## Configuration

```php
// config/image.php
return [
    'prefix' => 'photos',              // URL prefix (/photos/{image})
    'public_path' => 'public/',        // Base storage path
    'tmp_path' => 'public/tmp/',       // Temporary uploads
    'cache_path' => 'public/cache/',   // Cached manipulations
    'quality' => 100,                  // JPEG quality (1-100)
    'presets' => [
        'thumb' => 150,
        'small' => 480,
        'medium' => 720,
        'large' => 1600,
        'full' => 1920,
    ],
    'browser_cache' => 86400,          // Cache-Control max-age (seconds)
    'suffix' => '__',                  // Filename suffix separator
];
```

## Common Tasks

### Upload an Avatar

```php
use Bongo\Image\Services\AvatarImage;

$avatar = new AvatarImage(auth()->user(), $request->file('avatar'));
$user = $avatar->save(); // Deletes old avatar, saves new one
```

### Upload WYSIWYG Images

```php
use Bongo\Image\Services\WysiwygImage;

$wysiwyg = new WysiwygImage($post, $request->file('file'));
$image = $wysiwyg->save(); // Appends to collection
```

### Extract Base64 from HTML Content

```php
use Bongo\Image\Services\BuilderService;

$post = Post::create($request->validated());

$builder = new BuilderService($post, 'posts/images');
$post->content = $builder->process(); // Extracts and updates content
$post->save();
```

### Display Images in Blade

```php
{{-- Primary image with preset --}}
<img src="{{ $post->getPrimaryImage(['preset' => 'large']) }}" alt="{{ $post->title }}">

{{-- Custom dimensions with crop --}}
<img src="{{ $post->getPrimaryImage(['w' => 800, 'h' => 600, 'mode' => 'crop']) }}">

{{-- Direct route with quality --}}
<img src="{{ route('frontend.image.show', ['image' => $post->primaryImage(), 'preset' => 'large', 'q' => 80]) }}">

{{-- Placeholder fallback --}}
<img src="{{ $post->getPlaceholder(['preset' => 'medium']) }}">
```

### Eager Load Images

```php
// Avoid N+1 queries
$posts = Post::with('images')->get();

// With specific order/limit
$posts = Post::with(['images' => function($query) {
    $query->where('type', Image::UPLOAD)
          ->orderBy('sort_order')
          ->limit(5);
}])->get();
```

### Add Custom Image Methods

```php
class Post extends Model
{
    use HasImages;

    public function featuredImage(): ?Image
    {
        return $this->images()
            ->where('type', Image::COVER_IMAGE)
            ->first();
    }

    public function galleryImages()
    {
        return $this->images()
            ->where('type', Image::UPLOAD)
            ->orderBy('sort_order');
    }
}
```

## Code Style Summary

### Return Types

Always declare return types:

```php
public function getFileName(): string
public function hasImages(): bool
public function primaryImage(): ?Image
public function images(): MorphToMany
```

### Path Handling

Use config and clean paths:

```php
$path = config('image.public_path') . $relativePath;
$path = ltrim($path, '/');
$path = rtrim($path, '/');
```

### Storage Operations

Use Storage facade exclusively:

```php
Storage::exists($path)
Storage::put($path, $content)
Storage::get($path)
Storage::url($path)
Storage::path($path) // Absolute filesystem path
```

### Constants Over Strings

```php
// Good
$image->type = Image::AVATAR;
$image->orientation = Image::LANDSCAPE;

// Bad
$image->type = 'avatar';
$image->orientation = 'landscape';
```

### Dimension Setting

Always set width, height, and orientation:

```php
[$width, $height] = getimagesize($dbImage->getFileSrc());
$dbImage->width = $width;
$dbImage->height = $height;
$dbImage->orientation = ($width > $height) ? Image::LANDSCAPE : Image::PORTRAIT;
```

## Image Manipulation Flow

1. **Request**: `GET /photos/{image}?preset=large&q=80&mode=resize`
2. **Controller**: Validates params (preset exists, dimensions in range)
3. **Cache Check**: Looks for `cache_path/large_r_q80_image.jpg`
4. **Generate** (if not cached):
   - Load original with `ImageManipulator`
   - Apply manipulation (resize/crop/fit)
   - Stream to cache with quality setting
5. **Response**: Stream file with Cache-Control headers

## Database Schema

### images Table

Primary table storing image metadata:

```
- id, uuid
- name, title
- width, height, orientation
- path, ext, size
- type (avatar/cover_image/upload/wysiwyg)
- created_by, updated_by
- timestamps, deleted_at
```

### imageables Table

Polymorphic pivot table:

```
- id
- image_id (FK to images)
- imageable_id, imageable_type
- sort_order
- timestamps
```

## Testing

```bash
# Run all tests
vendor/bin/phpunit

# Run specific test
vendor/bin/phpunit --filter test_image_upload
```

Key test scenarios:
- Image upload and storage
- Polymorphic relationships
- Cached file generation
- Placeholder fallback
- BuilderService HTML processing

## Common Pitfalls

1. **Forget eager loading** → N+1 queries
2. **Hardcode paths** → Use config('image.*')
3. **Skip orientation** → Always set when creating images
4. **Use strings for types** → Use Image::AVATAR constants
5. **Don't clean paths** → Use ltrim/rtrim
6. **Ignore placeholder fallbacks** → Always provide fallback

## Extension Points

### Add New Image Type

1. Add constant to `Image` model
2. Create service class following `AvatarImage` pattern
3. Add controller endpoint (if needed)

### Add New Manipulation Mode

1. Add method to `ImageManipulator`
2. Update `Frontend\ImageController@show` mode handling
3. Update `Image::getCachedFileName()` for cache naming

### Add New Preset

Update `config/image.php`:

```php
'presets' => [
    'thumb' => 150,
    'xlarge' => 2400,  // New preset
],
```

## Related Packages

This package integrates with:
- `bongo/framework` - Base classes and service provider
- `cms/builder` - May use BuilderService for content processing
- `cms/page`, `cms/post` - Likely use HasImages trait

## Support

For issues or questions:
- Check [ARCHITECTURE.md](ARCHITECTURE.md) for detailed flow diagrams
- Review [.cursorrules](.cursorrules) for development patterns
- See Bongo Framework documentation for base class features
