<?php

namespace Bongo\Image\Models;

use Bongo\Framework\Helpers\File;
use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasUUID;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Storage;

class Image extends AbstractModel
{
    // Orientations
    public const LANDSCAPE = 'landscape';
    public const PORTRAIT = 'portrait';

    // Types
    public const AVATAR = 'avatar';
    public const COVER_IMAGE = 'cover_image';
    public const UPLOAD = 'upload';
    public const WYSIWYG = 'wysiwyg';

    use HasUUID;
    use SoftDeletes;

    protected $appends = ['name_formatted'];

    /** @var array */
    protected $fillable = [
        'name',
        'title',
        'width',
        'height',
        'orientation',
        'path',
        'ext',
        'size',
        'type',
        'created_by',
        'updated_by',
    ];

    public function imageables(): HasMany
    {
        return $this->hasMany(Imageable::class, 'image_id');
    }

    public function scopeOfType($query, $type)
    {
        return $query->whereType($type);
    }

    public function scopeForImageableId($query, $imagable_id)
    {
        return $query->where('imageable_id', $imagable_id);
    }

    public function getNameFormattedAttribute(): ?string
    {
        return File::nameWithoutSuffix($this->name);
    }

    public function getFileName(): string
    {
        // Clean up the file name
        $fileName = ltrim($this->name, '/');
        $fileName = rtrim($fileName, '/');

        return $fileName;
    }

    public function getFilePath(): string
    {
        // Clean up the file path
        $filePath = str_replace(config('image.public_path'), '', $this->path);
        $filePath = ltrim($filePath, '/');
        $filePath = rtrim($filePath, '/');

        return $filePath;
    }

    public function getPublicFilePath(): string
    {
        return config('image.public_path').$this->getFilePath();
    }

    public function getFileSrc(): string
    {
        return Storage::path("{$this->getPublicFilePath()}/{$this->getFileName()}");
    }

    public function getFileUrl(): string
    {
        return Storage::url("{$this->getPublicFilePath()}/{$this->getFileName()}");
    }

    public function getFileData(): string
    {
        return Storage::get("{$this->getPublicFilePath()}/{$this->getFileName()}");
    }

    public function fileExists(): string
    {
        return Storage::exists("{$this->getPublicFilePath()}/{$this->getFileName()}");
    }

    public function deleteFile(): string
    {
        return Storage::delete("{$this->getPublicFilePath()}/{$this->getFileName()}");
    }

    public function renameFile(string $newFileName): void
    {
        Storage::move(
            "{$this->getPublicFilePath()}/{$this->getFileName()}",
            "{$this->getPublicFilePath()}/{$newFileName}"
        );
    }

    public function getFileExtension(): string
    {
        return pathinfo("{$this->getPublicFilePath()}/{$this->getFileName()}", PATHINFO_EXTENSION);
    }

    public function getCachedFilePath(): string
    {
        $cachedFilePath = config('image.cache_path').$this->getFilePath();

        // If the cached path does not exist then create it
        if (! Storage::exists($cachedFilePath)) {
            Storage::makeDirectory($cachedFilePath);
        }

        return $cachedFilePath;
    }

    public function getCachedFileName(array $args): string
    {
        // Set the quality
        $q = $args['q'] ?? config('image.quality');

        // Set the mode
        $m = 'r';
        if (isset($args['mode']) && $args['mode'] == 'crop') {
            $m = 'c';
        } else {
            if (isset($args['mode']) && $args['mode'] == 'fit') {
                $m = 'f';
            } else {
                if (isset($args['mode']) && $args['mode'] == 'resizeCrop') {
                    $m = 'rc';
                } else {
                    if (isset($args['mode']) && $args['mode'] == 'resizeFit') {
                        $m = 'rf';
                    }
                }
            }
        }

        // Do we have a preset?
        if (isset($args['preset']) && ! empty($args['preset'])) {
            return "{$args['preset']}_{$m}_q{$q}_{$this->getFileName()}";
        }

        // If not do we have both width & height
        if ((isset($args['w']) && ! empty($args['w'])) && (isset($args['h']) && ! empty($args['h']))) {
            return "w{$args['w']}_h{$args['h']}_{$m}_q{$q}_{$this->getFileName()}";
        }

        // If not do we just have the width
        if (isset($args['w']) && ! empty($args['w'])) {
            return "w{$args['w']}_{$m}_q{$q}_{$this->getFileName()}";
        }

        // Or just the height
        if (isset($args['h']) && ! empty($args['h'])) {
            return "h{$args['h']}_{$m}_q{$q}_{$this->getFileName()}";
        }

        // else just return a large version
        return "_{$m}_q{$q}_{$this->getFileName()}";
    }

    public function getCachedFile(array $args): string
    {
        return "{$this->getCachedFilePath()}/{$this->getCachedFileName($args)}";
    }

    public function getMimeType(): string
    {
        $mimeType = 'image/jpeg';
        switch ($this->ext) {
            case 'png':
                $mimeType = 'image/png';
                break;
        }

        return $mimeType;
    }

    public function getPresetByKey(string $key): array
    {
        return config("image.presets.{$key}");
    }

    public function isPortrait(): bool
    {
        return $this->orientation === self::PORTRAIT;
    }

    public function isLandscape(): bool
    {
        return $this->orientation === self::LANDSCAPE;
    }
}
