<?php

namespace Bongo\Image\Services;

use Bongo\Framework\Helpers\File;
use Bongo\Image\Exceptions\BuilderException;
use Bongo\Image\Models\Image;
use DOMDocument;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class BuilderService
{
    /**
     * @var
     */
    private $domDocument;

    private ?string $imagePath;

    /**
     * @var
     */
    private $images;

    /**
     * @var
     */
    private $model;

    private ?string $relativePath;

    /**
     * BuilderService constructor.
     *
     * @param $model
     * @param $imagePath
     */
    public function __construct($model, $imagePath)
    {
        $this->model = $model;
        $imagePath = ltrim($imagePath, '/');
        $imagePath = rtrim($imagePath, '/').'/';
        $this->imagePath = config('image.public_path').$imagePath;
    }

    /**
     * @throws BuilderException
     * @throws Exception
     */
    public function process(): ?string
    {
        $this->validateModel();
        $this->convertHtmlToDomObjects();
        $this->extractImagesFromHtml();
        $this->setDirectoryPaths();
        $this->processImages();

        return $this->model->content;
    }

    /**
     * Validate that the model passed in has content and can store images.
     *
     * @throws Exception
     */
    protected function validateModel(): void
    {
        $class = get_class($this->model);

        // Make sure the model has the content field
        if (! Schema::hasColumn($this->model->getTable(), 'content')) {
            throw new Exception("Class[{$class}] does not have the content field");
        }

        // Make sure the model has the imageable trait
        if (! method_exists($this->model, 'images')) {
            throw new Exception("Class[{$class}] does not have the imagable trait");
        }
    }

    /**
     * Convert html to dom objects.
     *
     * @throws BuilderException
     */
    protected function convertHtmlToDomObjects(): void
    {
        // Make sure there is html
        if (empty($this->model->content)) {
            throw new BuilderException('Model content cannot be empty');
        }

        // Turn off DOMDocument HTML warnings
        libxml_use_internal_errors(true);

        // Parse the html into dom objects
        $this->domDocument = new DOMDocument();
        $this->domDocument->loadHTML('<html>'.$this->model->content.'</html>');
    }

    /**
     * Extract images from the html.
     *
     * @throws BuilderException
     */
    protected function extractImagesFromHtml(): void
    {
        // Get the images from the html
        $this->images = $this->domDocument->getElementsByTagName('img');
        if (empty($this->images)) {
            throw new BuilderException('No image tags found within the html content');
        }
    }

    /**
     * Set the directory paths which will house the images.
     */
    protected function setDirectoryPaths(): void
    {
        $this->relativePath = $this->imagePath.$this->model->id.'/';

        // Make sure we have a directory to store the file in
        if (! Storage::exists($this->relativePath)) {
            Storage::makeDirectory($this->relativePath);
        }
    }

    /**
     * Process each of the image nodes found.
     */
    protected function processImages(): void
    {
        foreach ($this->images as $image) {
            try {
                $this->processImage($image);
            } catch (BuilderException $e) {
                Log::error($e->getMessage());
                continue;
            }
        }
    }

    /**
     * Process the individual image.
     *
     * @param $htmlImage
     */
    protected function processImage($htmlImage): void
    {
        // Retrieve the src attribute and make sure it needs moving
        $srcAttribute = $this->getImageSrcAttribute($htmlImage);

        if (preg_match('/^data:image\/(\w+);base64,/', $srcAttribute)) {
            $imageData = substr($srcAttribute, strpos($srcAttribute, ',') + 1);
            $imageData = base64_decode($imageData);

            // Create a new name for the file
            $fileExt = 'png';
            $fileName = $this->createANewUniqueFileName($fileExt);

            // Move the file from temp to a permanent directory
            Storage::put($this->relativePath.$fileName, $imageData);

            // Save the image to the database
            $image = $this->storeImageDetailsInDatabase($fileName, $fileExt);

            // Update the old html paths within the model content
            $this->updateImageSrcAttribute($srcAttribute, $image);
        }
    }

    /**
     * @param  string  $fileName
     * @param  string  $fileExt
     *
     * @return mixed
     */
    protected function storeImageDetailsInDatabase(string $fileName, string $fileExt)
    {
        [$width, $height] = getimagesize(Storage::path($this->relativePath.$fileName));

        $dbImage = new Image();
        $dbImage->name = $fileName;
        $dbImage->title = isset($this->model->name) ? Str::limit($this->model->name, 60) : $fileName;
        $dbImage->width = $width;
        $dbImage->height = $height;
        $dbImage->orientation = ($width > $height) ? Image::LANDSCAPE : Image::PORTRAIT;
        $dbImage->path = $this->relativePath;
        $dbImage->ext = $fileExt;
        $dbImage->type = Image::WYSIWYG;
        $dbImage->created_by = user() ? user()->id : null;

        // Attach the image relationship to the model
        return $this->model->images()->save($dbImage);
    }

    /**
     * Get the image src attribute from the extracted image tags.
     *
     * @param $image
     *
     * @return mixed
     */
    private function getImageSrcAttribute($image)
    {
        return $image->getAttribute('src');
    }

    private function createANewUniqueFileName($fileExt = 'png'): ?string
    {
        do {
            $fileName = Str::random(8);
            if (isset($this->model->name) && ! empty($this->model->name)) {
                $fileName = $this->model->name;
            }

            $fileName = File::generateName($fileName, $fileExt);
        } while (Storage::exists($this->relativePath.$fileName));

        return $fileName;
    }

    /**
     * @param $srcAttribute
     * @param $image
     */
    private function updateImageSrcAttribute($srcAttribute, $image): void
    {
        $this->model->content = str_replace(
            $srcAttribute,
            '/'.config('image.prefix')."/{$image->name}",
            $this->model->content
        );
    }
}
