<?php

namespace Bongo\Image\Services;

use Bongo\Framework\Helpers\File;
use Bongo\Image\Interfaces\Imageable;
use Bongo\Image\Models\Image;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Str;

class CoverImage
{
    protected UploadedFile $file;

    protected Imageable $entity;

    public function __construct(Imageable $entity, UploadedFile $file)
    {
        $this->entity = $entity;
        $this->file = $file;
    }

    public function save(): Imageable
    {
        // Delete the existing image and file as only need one
        $this->delete();

        // Create the filename
        $ext = $this->file->getClientOriginalExtension();
        $name = rtrim($this->file->getClientOriginalName(), $ext);
        $size = $this->file->getSize();
        $fileName = File::generateName($name, $ext);

        // Upload the image to the directory
        $this->file->storePubliclyAs($this->getEntityPath(), $fileName);

        // Create a new database image
        $dbImage = new Image();
        $dbImage->name = $fileName;
        $dbImage->title = $this->entity->name ?? null;
        $dbImage->path = $this->getEntityPath();
        $dbImage->size = $size;
        $dbImage->ext = $ext;
        $dbImage->type = Image::COVER_IMAGE;
        $dbImage->created_by = user() ? user()->id : null;
        $dbImage->updated_by = user() ? user()->id : null;

        // Set the dimensions
        [$width, $height] = getimagesize($dbImage->getFileSrc());
        $dbImage->width = $width;
        $dbImage->height = $height;
        $dbImage->orientation = ($width > $height) ? Image::LANDSCAPE : Image::PORTRAIT;

        // Save the new image
        $this->entity->images()->save($dbImage);

        return $this->entity;
    }

    protected function delete(): void
    {
        $images = $this->entity->images()->where('type', Image::COVER_IMAGE)->get();

        if ($images) {
            foreach ($images as $image) {
                if ($image->fileExists()) {
                    $image->deleteFile();
                }
                $image->forceDelete();
            }
        }
    }

    private function getEntityPath(): string
    {
        $entityName = explode('\\', get_class($this->entity));
        $entityName = array_pop($entityName);
        $entityName = Str::snake($entityName);
        $entityName = Str::plural($entityName);

        return config('image.public_path')."{$entityName}/{$this->entity->id}/";
    }
}
