<?php

namespace Bongo\Image\Services;

use Intervention\Image\Image;
use Intervention\Image\ImageManager;
use Psr\Http\Message\StreamInterface;

class ImageManipulator
{
    protected Image $image;

    protected int $quality;

    protected ?int $width = null;

    protected ?int $height = null;

    public function __construct()
    {
        if (config('settings.memory_limit') && ! empty(config('settings.memory_limit'))) {
            ini_set('memory_limit', config('settings.memory_limit'));
        }

        $this->setQuality(config('image.quality'));
    }

    public function resize(): ImageManipulator
    {
        $this->image->resize($this->width, $this->height, function ($constraint) {
            $constraint->aspectRatio();
            $constraint->upsize();
        });

        return $this;
    }

    public function crop(): ImageManipulator
    {
        // If we do not have a height
        if (empty($this->height)) {

            // Try and calculate the height by the image ration
            if ($this->image->width() > $this->width) {
                $ratio = $this->width / $this->image->width();
                $this->height = $this->image->height() * $ratio;
            } else {
                $this->height = $this->width;
            }
        }

        $this->image->crop($this->width, $this->height);

        return $this;
    }

    public function fit(): ImageManipulator
    {
        $this->image->fit($this->width, $this->height);

        return $this;
    }

    public function stream(): StreamInterface
    {
        return $this->image->stream(null, $this->quality);
    }

    public function setImage(string $fileData): ImageManipulator
    {
        $this->image = (new ImageManager())->make($fileData);

        return $this;
    }

    public function setMode(string $mode): ImageManipulator
    {
        $this->mode = $mode;

        return $this;
    }

    public function setOrientation(string $orientation): ImageManipulator
    {
        $this->orientation = $orientation;

        return $this;
    }

    public function setQuality(int $quality): ImageManipulator
    {
        $this->quality = $quality;

        return $this;
    }

    public function setPreset(string $preset): ImageManipulator
    {
        $this->preset = $preset;

        return $this;
    }

    public function setWidth(int $width): ImageManipulator
    {
        $this->width = $width;

        return $this;
    }

    public function setHeight(int $height): ImageManipulator
    {
        $this->height = $height;

        return $this;
    }
}
