<?php

namespace Bongo\Image\Services;

use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager;

class ImagePlaceholder
{
    protected ?int $width = null;

    protected ?int $height = null;

    protected string $fileName = 'default_image.png';

    protected ?string $cachedFile = null;

    public function __construct(array $args)
    {
        $this->setDimensions($args);
        $this->setCachedFile();
    }

    protected function setDimensions(array $args): void
    {
        // Do we have a preset?
        if (isset($args['preset']) && ! empty($args['preset'])) {
            $this->width = config("image.presets.{$args['preset']}");
            $this->cachedFile = config('image.cache_path')."{$args['preset']}_{$this->fileName}";

        } elseif ((isset($args['w']) && ! empty($args['w'])) && (isset($args['h']) && ! empty($args['h']))) {
            $this->width = $args['w'];
            $this->height = $args['h'];
            $this->cachedFile = config('image.cache_path')."w{$args['w']}_h{$args['h']}_{$this->fileName}";

        } elseif (isset($args['w']) && ! empty($args['w'])) {
            $this->width = $args['w'];
            $this->cachedFile = config('image.cache_path')."w{$args['w']}_{$this->fileName}";

        } elseif (isset($args['h']) && ! empty($args['h'])) {
            $this->height = $args['h'];
            $this->cachedFile = config('image.cache_path')."h{$args['h']}_{$this->fileName}";

        } else {
            $this->cachedFile = config('image.cache_path').$this->fileName;
        }
    }

    /**
     * If the cached file does not exist in the cached folder then create it
     */
    protected function setCachedFile(): void
    {
        if (! Storage::exists($this->cachedFile)) {
            $imageManager = (new ImageManager())
                ->make(file_get_contents(public_path("images/{$this->fileName}")))
                ->resize($this->width, $this->height, function ($constraint) {
                    $constraint->aspectRatio();
                    $constraint->upsize();
                });

            // Save the modified image to cache
            Storage::put($this->cachedFile, $imageManager->stream(), 'public');
        }
    }

    public function get(): string
    {
        return Storage::get($this->cachedFile);
    }

    public function url(): string
    {
        return Storage::url($this->cachedFile);
    }
}
