# GitHub Copilot Instructions - Bongo Import

## Project Overview

Laravel package for importing legacy Bongo CMS content (pages, posts, reviews) from a remote MySQL database into Bongo v3.0. Handles HTML cleaning, URL transformation, and image migration.

**Namespace**: `Bongo\Import`
**Laravel**: 10+ | **PHP**: 8.2+
**Key Dependencies**: bongo/framework, bongo/image, bongo/page, bongo/post, bongo/review

## Architecture

### Service Provider
Extends `Bongo\Framework\Providers\AbstractServiceProvider` with automatic config/view/command registration:

```php
class ImportServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'import';
    protected array $commands = [ImportCommand::class, ResetCommand::class];
}
```

### Class Relationships

```
ImportCommand
├── ImportPageFromBongo → Page (Bongo\Page\Models\Page)
│   ├── BongoPage (chunks → texts)
│   └── ImageHelper
├── ImportPostFromBongo → Post (Bongo\Post\Models\Post)
│   ├── BongoPost
│   └── ImageHelper
└── ImportReviewFromBongo → Review (Bongo\Review\Models\Review)
    └── BongoReview

BongoImport (abstract)
├── cleanHtml()
├── updateLinks()
└── updateImageSrc()
```

### Data Flow

1. **Import Command** prompts for credentials → creates `import` database connection
2. **Service Classes** fetch legacy models → transform data → create new models
3. **HTML Processing** via `BongoImport` base class
4. **Image Download** via `ImageHelper` → polymorphic attachments

## Code Style Templates

### Service Class Pattern

```php
class Import{Type}FromBongo extends BongoImport
{
    public function __construct(Bongo{Type} $bongoModel, string $siteUrl)
    {
        $model = $this->findOrCreate($bongoModel);
        $this->saveContent($bongoModel, $model, $siteUrl);
        $this->saveImages($model, $siteUrl);
    }

    private function findOrCreate(Bongo{Type} $bongoModel): {Type}
    {
        $model = {Type}::firstOrNew(['slug' => Str::slug($bongoModel->title)]);
        $model->name = $bongoModel->title;
        $model->status = {Type}::ACTIVE;
        $model->save();
        return $model;
    }

    private function saveContent(Bongo{Type} $bongoModel, {Type} $model, string $siteUrl): void
    {
        $html = $this->cleanHtml($bongoModel->content);
        $html = $this->updateLinks($html, $siteUrl);
        $model->content = $html;
        $model->save();
    }

    private function saveImages({Type} $model, string $siteUrl): void
    {
        libxml_use_internal_errors(true);
        $domDocument = new DOMDocument();
        $domDocument->loadHTML('<html>'.$model->content.'</html>');
        $images = $domDocument->getElementsByTagName('img');

        if ($images && count($images)) {
            $html = $model->content;
            foreach ($images as $key => $image) {
                $imageSrc = $image->getAttribute('src');
                $imageSrc = $this->updateImageSrc($imageSrc, $siteUrl);

                if (ImageHelper::remoteFileExists($siteUrl.$imageSrc)) {
                    $dbImage = ImageHelper::store($model, $siteUrl.$imageSrc, "{type}s", Image::WYSIWYG, $key + 1);
                    $html = str_replace($imageSrc, "/photos/{$dbImage->getFileName()}", $html);
                }
            }
            $model->content = $html;
            $model->save();
        }
    }
}
```

### Source Database Model

```php
namespace Bongo\Import\Models;

use Illuminate\Database\Eloquent\Model;

class {Type} extends Model
{
    protected $connection = 'import';
    protected $table = '{legacy_table_name}';

    // Relationships to legacy tables
    public function relatedRecords(): HasMany
    {
        return $this->hasMany(RelatedModel::class);
    }
}
```

### Command Method Pattern

```php
private function import{Types}(): void
{
    // Check if destination package is enabled
    if (!package()->isEnabled('{type}')) {
        $this->error('{Type} module is not enabled');
        return;
    }

    $bongoRecords = Bongo{Type}::query()
        ->where('unpublish', 0)
        ->get();

    if (!$bongoRecords || count($bongoRecords) < 1) {
        $this->error("No {types} found");
        return;
    }

    foreach ($bongoRecords as $bongoRecord) {
        $this->comment("- Importing {$bongoRecord->title} {type}");

        try {
            new Import{Type}FromBongo($bongoRecord, $this->siteUrl);
        } catch (Throwable|Exception $e) {
            $this->error($e->getMessage());
            log_exception($e);
            continue;
        }
    }

    $this->comment("{Types} have been imported");
}
```

## Common Patterns

### HTML Cleaning Pipeline
Always process HTML in this order:
```php
$html = $this->cleanHtml($text->raw_text);           // Remove styles, classes, empty tags
$html = $this->updateLinks($html, $siteUrl);         // Transform legacy URLs
$model->content = $html;
```

### Image Download and Storage
```php
// Check existence
if (!ImageHelper::remoteFileExists($siteUrl.$imageSrc)) {
    console_print("!! Remote image {$siteUrl}{$imageSrc} not found !!");
    continue;
}

// Download and store
console_print("-- Downloading image {$siteUrl}{$imageSrc}");
$dbImage = ImageHelper::store($model, $siteUrl.$imageSrc, "directory", Image::TYPE, $sortOrder);

// Update references
$html = str_replace($oldSrc, "/photos/{$dbImage->getFileName()}", $html);
```

### Dynamic Database Connection Setup
```php
config()->set('database.connections.import', [
    'driver' => 'mysql',
    'host' => $this->dbHost,
    'database' => $this->dbName,
    'username' => $this->dbUsername,
    'password' => $this->dbPassword,
    'charset' => 'utf8mb4',
]);

DB::purge('import'); // Refresh connection pool
```

### Error Handling in Import Loops
```php
foreach ($records as $record) {
    try {
        new ImportServiceClass($record, $siteUrl);
    } catch (Throwable|Exception $e) {
        $this->error($e->getMessage());
        log_exception($e);
        continue; // Don't break entire import
    }
}
```

## Key Classes Reference

### Commands
- **ImportCommand** (`src/Commands/ImportCommand.php:16`) - Interactive import orchestrator (`import:run`)
- **ResetCommand** (`src/Commands/ResetCommand.php:10`) - Cleanup utility (`import:reset`)

### Services
- **BongoImport** (`src/Services/BongoImport.php:8`) - Abstract base with HTML/URL processing
- **ImportPageFromBongo** (`src/Services/ImportPageFromBongo.php:13`) - Page migration with chunks/texts
- **ImportPostFromBongo** (`src/Services/ImportPostFromBongo.php:13`) - Blog post migration
- **ImportReviewFromBongo** (`src/Services/ImportReviewFromBongo.php:8`) - Review migration

### Helpers
- **ImageHelper** (`src/Helpers/ImageHelper.php:9`)
  - `store($model, string $imageUrl, string $directory, string $type, int $key = 1): Image`
  - `remoteFileExists(string $fileUrl): bool`

### Source Models (connection: 'import')
- **Page** (`src/Models/Page.php:8`) - `pages` table, `hasMany(Chunk::class)`
- **Chunk** (`src/Models/Chunk.php:9`) - `chunks` table, `hasMany(Text::class)`
- **Text** (`src/Models/Text.php:8`) - `text` table, contains `raw_text`
- **Post** (`src/Models/Post.php:7`) - `blogposts` table
- **Review** (`src/Models/Review.php:7`) - `reviewposts` table

## Method Signatures

### BongoImport
```php
protected function cleanHtml(string $html): string
protected function updateLinks(string $html, string $siteUrl): string
protected function updateImageSrc(string $imageSrc, string $siteUrl): string
```

### ImageHelper
```php
public static function store($model, string $imageUrl, string $directory, string $type, int $key = 1): Image
public static function remoteFileExists(string $fileUrl): bool
```

### ImportCommand
```php
private function setDatabaseConnection(): void
private function checkDatabaseConnection(): bool
private function importPages(): void
private function importPosts(): void
private function importReviews(): void
```

## Configuration

**Config file**: `src/Config/import.php`

```php
return [
    'site_url' => 'https://designtec.co.uk',    // Remote site for image downloads
    'db_host' => '127.0.0.1',                    // Legacy database host
    'db_name' => 'designtec',                    // Legacy database name
    'db_username' => 'designtec',                // Legacy database username
    'db_password' => 'designtec',                // Legacy database password
];
```

## Console Commands

```bash
php artisan import:run    # Interactive import wizard
php artisan import:reset  # Truncate destination tables and delete images
```

## Important Notes

### URL Transformations (in updateLinks)
- `?p=page.name` → `/page-name`
- `?review=reviews/slug` → `/reviews/slug`
- `?blog=blogs/archive/YYYY/MM/DD/name.aspx` → `/posts/name`

### Data Transformations
- Review ratings: 10-point scale → 5-star (divide by 2, round)
- Post meta description: Generated from content if `pagemeta` is null (150 char limit)
- Slugs: Generated via `Str::slug()` from titles/pagenames

### Content Filters (Source Queries)
- Pages: `where('hidden', 0)`
- Posts: `where('unpublish', 0)`
- Reviews: `where('unpublish', 0)->where('reviewstate', 1)`

### Image Storage Paths
- Storage: `storage/app/public/{type}/{model_id}/{filename}`
- Reference: `/photos/{filename}` (assumes symlink or routing)

### HTML Cleaning Steps (cleanHtml)
1. Remove `<br>` tags
2. Parse HTML via `Html::parse()`
3. Strip inline styles, classes
4. Remove empty tags
5. Wrap icons and shortcodes
6. Keep only: `h1-h6`, `p`, `b`, `i`, `ul`, `li`, `a`, `img`
7. Fix double-wrapped paragraphs

### Section Wrapper HTML
```html
<div class="is-section is-box">
    <div class="is-boxes">
        <div class="is-box-centered">
            <div class="is-container container-fluid" style="max-width:800px">
                <div class="row">
                    <div class="col">
                        <!-- Content here -->
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
```
