# Bongo Import - Claude Code Guidance

## Overview

Laravel package for importing legacy Bongo CMS content (pages, posts, reviews) from a remote MySQL database into Bongo v3.0. Handles nested content structures, HTML transformation, URL rewriting, and image migration.

**Package**: bongo/import
**Namespace**: Bongo\Import
**Repository**: https://bitbucket.org/designtec/import
**Laravel**: 10+ | **PHP**: 8.2+

## Documentation

- **ARCHITECTURE.md** - Detailed architecture, class diagrams, flow diagrams, extension points
- **.cursorrules** - Cursor AI instructions with coding conventions and common tasks
- **.github/copilot-instructions.md** - GitHub Copilot code templates and patterns

## Console Commands

```bash
# Interactive import wizard
php artisan import:run

# Reset destination data (truncate tables, delete images)
php artisan import:reset

# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint
```

## Quick Architecture Reference

### Service Provider

Extends `Bongo\Framework\Providers\AbstractServiceProvider` (no manual route/view registration needed):

```php
class ImportServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'import';
    protected array $commands = [ImportCommand::class, ResetCommand::class];
}
```

**Auto-bootstrapped**: Config from `src/Config/import.php`, Views from `src/Views/import/`

### Import Flow

```
ImportCommand (import:run)
├─▶ Prompts for database credentials (host, name, username, password, site URL)
├─▶ Creates dynamic 'import' database connection
├─▶ Tests connection
└─▶ Delegates to service classes:
    ├─▶ ImportPageFromBongo (pages with chunks/texts)
    ├─▶ ImportPostFromBongo (blog posts)
    └─▶ ImportReviewFromBongo (customer reviews)

Each service:
1. findOrCreate() - Create destination model
2. saveContent() - Clean HTML, update links
3. saveImages() - Download and store images
```

### Data Transformation Pipeline

```
Legacy HTML → cleanHtml() → updateLinks() → saveImages() → Destination Model
                 │              │                │
                 │              │                └─▶ Download remote images
                 │              │                    Store in /public/{type}/{id}/
                 │              │                    Update src to /photos/{filename}
                 │              │
                 │              └─▶ ?p=foo → /foo
                 │                  ?blog=blogs/.../name.aspx → /posts/name
                 │                  ?review=reviews/slug → /reviews/slug
                 │
                 └─▶ Strip styles, classes, empty tags
                     Wrap icons/shortcodes
                     Keep only: h1-h6, p, b, i, ul, li, a, img
```

## Key Files

| File | Purpose | Key Methods/Properties |
|------|---------|------------------------|
| `src/ImportServiceProvider.php` | Package registration | `$module = 'import'`, `$commands` |
| `src/Commands/ImportCommand.php` | Interactive import orchestrator | `handle()`, `importPages()`, `importPosts()`, `importReviews()` |
| `src/Commands/ResetCommand.php` | Database/storage cleanup | `handle()` (truncates tables, deletes images) |
| `src/Services/BongoImport.php` | Abstract base for HTML/URL processing | `cleanHtml()`, `updateLinks()`, `updateImageSrc()` |
| `src/Services/ImportPageFromBongo.php` | Page import (chunks/texts) | `findOrCreate()`, `saveContent()`, `saveImages()`, `saveOpenGraphImages()` |
| `src/Services/ImportPostFromBongo.php` | Post import | `findOrCreate()`, `saveContent()`, `saveImages()` |
| `src/Services/ImportReviewFromBongo.php` | Review import | Constructor (direct field mapping) |
| `src/Helpers/ImageHelper.php` | Image download/storage | `store()`, `remoteFileExists()` |
| `src/Models/Page.php` | Source page model (import connection) | `chunks()` relationship |
| `src/Models/Chunk.php` | Source chunk model | `texts()` relationship |
| `src/Models/Text.php` | Source text model | `raw_text` column |
| `src/Models/Post.php` | Source post model (blogposts table) | - |
| `src/Models/Review.php` | Source review model (reviewposts table) | - |
| `src/Config/import.php` | Default database credentials | `site_url`, `db_host`, `db_name`, `db_username`, `db_password` |

## Code Style Summary

### Namespace and Naming
- **Namespace**: `Bongo\Import`
- **Commands**: `*Command` suffix, signature `import:*`
- **Services**: `Import*FromBongo` pattern
- **Models**: Singular class, plural table (e.g., `Page` → `pages`)

### Return Types
- Always declare return types: `:void`, `:string`, `:bool`, `: Model`
- Use nullable types when appropriate: `: ?Image`

### Constructor Pattern for Services
Services perform all logic in constructor (no separate `execute()` method):

```php
public function __construct(BongoModel $bongoModel, string $siteUrl)
{
    $model = $this->findOrCreate($bongoModel);
    $this->saveContent($bongoModel, $model, $siteUrl);
    $this->saveImages($model, $siteUrl);
}
```

### Error Handling
Import command uses try-catch per record (continues on failure):

```php
foreach ($records as $record) {
    try {
        new ImportServiceClass($record, $siteUrl);
    } catch (Throwable|Exception $e) {
        $this->error($e->getMessage());
        log_exception($e);
        continue; // Don't break entire import
    }
}
```

### HTML Processing
Always process in this order:

```php
$html = $this->cleanHtml($text->raw_text);
$html = $this->updateLinks($html, $siteUrl);
// ... use $html for content
```

Use `libxml_use_internal_errors(true)` before DOMDocument parsing:

```php
libxml_use_internal_errors(true);
$domDocument = new DOMDocument();
$domDocument->loadHTML('<html>'.$content.'</html>');
```

## Configuration

### Default Settings
Edit `src/Config/import.php` to change default connection settings:

```php
return [
    'site_url' => 'https://designtec.co.uk',
    'db_host' => '127.0.0.1',
    'db_name' => 'designtec',
    'db_username' => 'designtec',
    'db_password' => 'designtec',
];
```

**Access**: `config('import.site_url')`

### Dynamic Connection
The `import` database connection is created at runtime in `ImportCommand::setDatabaseConnection()`:

```php
config()->set('database.connections.import', [...]);
DB::purge('import'); // Refresh connection pool
```

## Dependencies

### Composer
- **bongo/framework** (^3.0) - `AbstractServiceProvider`, `Html` helper, `File` helper
- **Laravel 10+** - `illuminate/contracts` (^10.0)
- **PHP 8.2+**

### Package Integration
- **bongo/image** - `Image` model, polymorphic relationships
- **bongo/page** - Destination `Page` model
- **bongo/post** - Destination `Post` model (optional, checked via `package()->isEnabled('post')`)
- **bongo/review** - Destination `Review` model (optional, checked via `package()->isEnabled('review')`)

## Common Tasks

### Running an Import

1. Ensure destination packages are enabled (page, post, review)
2. Have legacy database credentials ready
3. Run: `php artisan import:run`
4. Answer prompts for connection details
5. Confirm which content types to import
6. Monitor console output for errors

### Resetting Before Re-import

```bash
php artisan import:reset  # Truncates: pages, posts, reviews, images, imageables
                           # Deletes: /public/pages, /public/posts, /public/cache
php artisan import:run    # Run fresh import
```

### Adding Support for New Content Type

See **ARCHITECTURE.md** > Extension Points > Adding a New Content Type

Quick steps:
1. Create source model in `src/Models/` with `protected $connection = 'import'`
2. Create service in `src/Services/` extending `BongoImport`
3. Add import method to `ImportCommand`
4. Update `ResetCommand` to truncate new tables

### Customising HTML Cleaning

Override `cleanHtml()` in service class:

```php
class ImportPageFromBongo extends BongoImport
{
    protected function cleanHtml(string $html): string
    {
        $html = parent::cleanHtml($html);
        // Add custom transformations
        return $html;
    }
}
```

### Adding Custom URL Transformations

Edit `BongoImport::updateLinks()` in `src/Services/BongoImport.php`:

```php
// Add new pattern matching
if (Str::contains($url, '?product=')) {
    $url = str_replace('?product=', '/products/', $url);
}
```

## Data Transformations

### URL Rewrites (updateLinks)
- `?p=page.name` → `/page-name`
- `?review=reviews/slug` → `/reviews/slug`
- `?blog=blogs/archive/YYYY/MM/DD/name.aspx` → `/posts/name`
- Internal links: domain removed

### Content Transformations
- **Review ratings**: 10-point scale → 5-star (`round($rank / 2)`)
- **Post slugs**: Dots replaced with dashes, then slugified
- **Post meta**: Generated from content if `pagemeta` is null (150 char limit)

### HTML Cleaning (cleanHtml)
1. Remove `<br>` tags
2. Strip inline styles and classes
3. Remove empty tags
4. Wrap icons and shortcodes
5. Keep only allowed tags: `h1-h6`, `p`, `b`, `i`, `ul`, `li`, `a`, `img`
6. Fix double-wrapped `<p>` tags

### Image Storage
- **Download**: via `file_get_contents()` from remote URL
- **Storage**: `storage/app/public/{type}/{model_id}/{filename}`
- **Reference**: Updated to `/photos/{filename}` in content
- **Metadata**: Dimensions extracted via `getimagesize()`, orientation calculated

## Source Database Tables

All source models use `protected $connection = 'import'`:

| Model | Table | Key Columns | Relationships |
|-------|-------|-------------|---------------|
| Page | `pages` | `title`, `titleexten`, `description`, `hidden`, `ogimage` | `hasMany(Chunk::class)` |
| Chunk | `chunks` | `page_id`, `order` | `belongsTo(Page::class)`, `hasMany(Text::class)` |
| Text | `text` | `chunk_id`, `text_index`, `raw_text` | `belongsTo(Chunk::class)` |
| Post | `blogposts` | `title`, `pagename`, `blog`, `pagemeta`, `postdate`, `unpublish` | - |
| Review | `reviewposts` | `title`, `contactname`, `contactemail`, `review`, `rank`, `reviewstate`, `unpublish`, `modwhen` | - |

## Content Filters (Source Queries)

- **Pages**: `where('hidden', 0)`
- **Posts**: `where('unpublish', 0)`
- **Reviews**: `where('unpublish', 0)->where('reviewstate', 1)`

## Important Notes

### Security
- Database credentials prompted interactively (not committed)
- Image downloads use `CURLOPT_SSL_VERIFYPEER = 0` (development only!)
- No XSS sanitisation (assumes trusted source database)

### Limitations
- No rollback mechanism (use `import:reset` to start over)
- `firstOrNew` doesn't prevent all duplicates (may create duplicate records on re-run)
- Image downloads are synchronous (slow for large content)
- No progress bar or percentage indicator

### Content Wrappers
Imported content wrapped in section divs:

```html
<div class="is-section is-box">
    <div class="is-boxes">
        <div class="is-box-centered">
            <div class="is-container container-fluid" style="max-width:800px">
                <div class="row">
                    <div class="col">
                        <!-- Content here -->
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
```

## Testing

```bash
# Run all tests
vendor/bin/phpunit

# Test with coverage
vendor/bin/phpunit --coverage-html coverage/
```

**Base class**: `Bongo\Import\Tests\TestCase` extends Orchestra Testbench

## Troubleshooting

| Issue | Solution |
|-------|----------|
| Connection error | Verify host, database, credentials; check firewall/network |
| Missing images | Verify site URL, check remote paths, confirm network access |
| Duplicate content | Use `import:reset` before re-importing |
| Memory exhausted | Chunk queries, increase PHP memory_limit |
| Broken HTML | Check cleaning rules, verify allowed tags, inspect legacy markup |

## Framework Helpers Used

- `console_print(string $message, string $level = 'info')` - CLI output
- `log_exception(Throwable $e)` - Exception logging
- `package()->isEnabled(string $name): bool` - Package availability check
- `user()` - Current user (for image created_by/updated_by)

## Version Information

**Current**: Migrated to Laravel 10+ and PHP 8.2+
**Legacy**: Originally PHP 7.0, Laravel 5/6 (see outdated README.md)
