<?php

namespace Bongo\Import\Commands;

use Bongo\Import\Models\Page as BongoPage;
use Bongo\Import\Models\Post as BongoPost;
use Bongo\Import\Models\Review as BongoReview;
use Bongo\Import\Services\ImportPageFromBongo;
use Bongo\Import\Services\ImportPostFromBongo;
use Bongo\Import\Services\ImportReviewFromBongo;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Throwable;

class ImportCommand extends Command
{
    /** @var string */
    protected $signature = 'import:run';

    /** @var string */
    protected $description = 'Import pages, posts and reviews from old bongo.';

    private string $siteUrl;

    private string $dbHost;

    private string $dbName;

    private string $dbUsername;

    private string $dbPassword;

    public function handle(): void
    {
        $this->info('Welcome, please complete the import details below to continue.');

        $this->siteUrl =  rtrim($this->ask('Site Url?', config('import.site_url')), '/');
        $this->dbHost = $this->ask('Database Host?', config('import.db_host'));
        $this->dbName = $this->ask('Database Name?', config('import.db_name'));
        $this->dbUsername = $this->ask('Database Username?', config('import.db_username'));
        $this->dbPassword = $this->ask('Database Password?', config('import.db_password'));

        $this->comment('Connecting to database, please wait...');
        $this->setDatabaseConnection();

        if (! $this->checkDatabaseConnection()) {
            $this->error("Could not connect to the database with the below credentials!");
            $this->info("-- host: {$this->dbHost}");
            $this->info("-- database: {$this->dbName}");
            $this->info("-- username: {$this->dbUsername}");
            $this->info("-- password: {$this->dbPassword}");

            return;
        }
        $this->comment('Connection successful');

        if ($this->confirm('Import pages?')) {
            $this->importPages();
        }

        if ($this->confirm('Import posts?')) {
            $this->importPosts();
        }

        if ($this->confirm('Import reviews?')) {
            $this->importReviews();
        }

        $this->comment('Import successful, goodbye :)');
    }

    private function setDatabaseConnection(): void
    {
        config()->set('database.connections.import', [
            'driver' => 'mysql',
            'port' => '3306',
            'charset' => 'utf8mb4',
            'collation' => 'utf8mb4_unicode_ci',
            'engine' => 'InnoDB',
            'host' => $this->dbHost,
            'database' => $this->dbName,
            'username' => $this->dbUsername,
            'password' => $this->dbPassword,
        ]);

        // Refresh the import connection
        DB::purge('import');
    }

    private function checkDatabaseConnection(): bool
    {
        try {
            DB::connection('import')->getPdo();

            return true;

        } catch (Exception $e) {
            return false;
        }
    }

    private function importPages(): void
    {
        $bongoPages = BongoPage::query()
            ->with('chunks.texts')
            ->where('hidden', 0)
            ->get();

        if (! $bongoPages || count($bongoPages) < 1) {
            $this->error("No pages found");

            return;
        }

        foreach ($bongoPages as $bongoPage) {
            $this->comment("- Importing {$bongoPage->title} page");

            try {
                (new ImportPageFromBongo($bongoPage, $this->siteUrl));

            } catch (Throwable|Exception $e) {
                $this->error($e->getMessage());
                log_exception($e);
                continue;
            }
        }

        $this->comment("Pages have been imported");
    }

    private function importPosts(): void
    {
        if (! package()->isEnabled('post')) {
            $this->error('Posts module is not enabled');

            return;
        }

        $bongoPosts = BongoPost::query()
            ->where('unpublish', 0)
            ->get();

        if (! $bongoPosts || count($bongoPosts) < 1) {
            $this->error("No posts found");

            return;
        }

        foreach ($bongoPosts as $bongoPost) {
            $this->comment("- Importing {$bongoPost->title} post");

            try {
                (new ImportPostFromBongo($bongoPost, $this->siteUrl));

            } catch (Throwable|Exception $e) {
                $this->error($e->getMessage());
                log_exception($e);
                continue;
            }
        }

        $this->comment("Posts have been imported");
    }

    private function importReviews(): void
    {
        if (! package()->isEnabled('review')) {
            $this->error('Reviews module is not enabled');

            return;
        }

        $bongoReviews = BongoReview::query()
            ->where('unpublish', 0)
            ->where('reviewstate', 1)
            ->get();

        if (! $bongoReviews || count($bongoReviews) < 1) {
            $this->error("No reviews found");

            return;
        }

        foreach ($bongoReviews as $bongoReview) {
            $this->comment("- Importing {$bongoReview->title} review");

            try {
                (new ImportReviewFromBongo($bongoReview));

            } catch (Throwable|Exception $e) {
                $this->error($e->getMessage());
                log_exception($e);
                continue;
            }
        }

        $this->comment("Reviews have been imported");
    }
}
