# Architecture Documentation - Bongo Import

## Overview

The Bongo Import package is a Laravel migration tool for transferring legacy Bongo CMS content from an old database structure to the new Bongo v3.0 platform. It handles three content types (pages, posts, reviews) with complex nested structures, HTML transformation, URL rewriting, and remote image downloading.

**Package**: bongo/import
**Namespace**: Bongo\Import
**Laravel**: 10+
**PHP**: 8.2+

## Directory Structure

```
bongo/import/
├── src/
│   ├── Commands/
│   │   ├── ImportCommand.php           # Interactive import orchestrator (import:run)
│   │   └── ResetCommand.php            # Database/storage cleanup (import:reset)
│   │
│   ├── Config/
│   │   └── import.php                  # Default database credentials
│   │
│   ├── Helpers/
│   │   └── ImageHelper.php             # Image download/storage utilities
│   │
│   ├── Models/                         # Source database models (all use 'import' connection)
│   │   ├── Page.php                    # Maps to 'pages' table
│   │   ├── Chunk.php                   # Maps to 'chunks' table (page sections)
│   │   ├── Text.php                    # Maps to 'text' table (chunk content)
│   │   ├── Post.php                    # Maps to 'blogposts' table
│   │   └── Review.php                  # Maps to 'reviewposts' table
│   │
│   ├── Services/
│   │   ├── BongoImport.php             # Abstract base: HTML cleaning, URL transformation
│   │   ├── ImportPageFromBongo.php     # Page import logic
│   │   ├── ImportPostFromBongo.php     # Post import logic
│   │   └── ImportReviewFromBongo.php   # Review import logic
│   │
│   ├── Views/backend/
│   │   ├── open_section.blade.php      # Opening HTML wrapper for content sections
│   │   ├── close_section.blade.php     # Closing HTML wrapper
│   │   └── section_title.blade.php     # Title heading template
│   │
│   └── ImportServiceProvider.php       # Package service provider
│
├── tests/
│   └── TestCase.php                    # Orchestra Testbench base class
│
├── composer.json                       # Package dependencies and autoloading
├── phpunit.xml                         # PHPUnit configuration
└── README.md                           # Package documentation
```

## Class Diagram

```
┌─────────────────────────────────────────────────────────────────┐
│                    AbstractServiceProvider                      │
│                   (bongo/framework package)                     │
└────────────────────────────────┬────────────────────────────────┘
                                 │ extends
                                 │
                    ┌────────────▼───────────────┐
                    │ ImportServiceProvider      │
                    │ - $module = 'import'       │
                    │ - $commands = [...]        │
                    └────────────────────────────┘
                                 │
                                 │ registers
                    ┌────────────┴────────────┐
                    │                         │
         ┌──────────▼──────────┐   ┌─────────▼──────────┐
         │  ImportCommand      │   │  ResetCommand      │
         │  (import:run)       │   │  (import:reset)    │
         └──────────┬──────────┘   └────────────────────┘
                    │
                    │ delegates to
         ┌──────────┼──────────┬──────────────────┐
         │          │          │                  │
    ┌────▼─────┐ ┌─▼──────┐ ┌─▼─────────┐       │
    │  Import  │ │ Import │ │  Import   │       │
    │  Page    │ │  Post  │ │  Review   │       │
    │  From    │ │  From  │ │  From     │       │
    │  Bongo   │ │ Bongo  │ │  Bongo    │       │
    └────┬─────┘ └────┬───┘ └─────┬─────┘       │
         │            │            │             │
         │ extends    │ extends    │ (no base)   │
         │            │            │             │
    ┌────▼────────────▼────────────┘             │
    │      BongoImport (abstract)                │
    │  - cleanHtml()                             │
    │  - updateLinks()                           │
    │  - updateImageSrc()                        │
    └────────────────────────────────────────────┘
                                                  │
                                                  │ uses
                                         ┌────────▼────────┐
                                         │  ImageHelper    │
                                         │  - store()      │
                                         │  - exists()     │
                                         └─────────────────┘

Source Models (connection: 'import')         Destination Models (other packages)
┌──────────────────────────┐                ┌──────────────────────────┐
│  Page                    │                │  Bongo\Page\Models\Page  │
│  - chunks: HasMany       │   ──────────▶  │                          │
│    └─▶ Chunk             │                └──────────────────────────┘
│        - texts: HasMany  │
│          └─▶ Text        │                ┌──────────────────────────┐
└──────────────────────────┘                │  Bongo\Post\Models\Post  │
                                            │                          │
┌──────────────────────────┐   ──────────▶  └──────────────────────────┘
│  Post (blogposts table)  │
└──────────────────────────┘                ┌────────────────────────────┐
                                            │  Bongo\Review\Models\Review│
┌──────────────────────────┐   ──────────▶  │                            │
│  Review (reviewposts)    │                └────────────────────────────┘
└──────────────────────────┘
```

## Import Flow Diagram

```
┌──────────────────────────────────────────────────────────────────┐
│                       User runs: import:run                      │
└───────────────────────────────┬──────────────────────────────────┘
                                │
                    ┌───────────▼──────────┐
                    │  Prompt for:         │
                    │  - Site URL          │
                    │  - DB Host           │
                    │  - DB Name           │
                    │  - DB Username       │
                    │  - DB Password       │
                    └───────────┬──────────┘
                                │
                    ┌───────────▼──────────────────────┐
                    │  Create dynamic DB connection:   │
                    │  config()->set('database.        │
                    │    connections.import', [...])   │
                    │  DB::purge('import')             │
                    └───────────┬──────────────────────┘
                                │
                    ┌───────────▼──────────┐
                    │  Test connection     │
                    │  via getPdo()        │
                    └───────────┬──────────┘
                                │
                         ┌──────┴───────┐
                         │              │
                    Success          Failure
                         │              │
                         │         Show error & exit
                         │
              ┌──────────┴──────────┬──────────────────┬──────────────┐
              │                     │                  │              │
    ┌─────────▼─────────┐  ┌────────▼────────┐  ┌─────▼──────┐      │
    │ Confirm import    │  │ Confirm import  │  │ Confirm    │      │
    │ pages?            │  │ posts?          │  │ reviews?   │      │
    └─────────┬─────────┘  └────────┬────────┘  └─────┬──────┘      │
              │                     │                  │              │
          ┌───▼───┐             ┌───▼───┐          ┌───▼───┐         │
          │  Yes  │             │  Yes  │          │  Yes  │         │
          └───┬───┘             └───┬───┘          └───┬───┘         │
              │                     │                  │              │
    ┌─────────▼──────────┐  ┌───────▼─────────┐  ┌────▼──────────┐  │
    │ importPages()      │  │ importPosts()   │  │ importReviews()│ │
    │                    │  │                 │  │                │  │
    │ Query BongoPage    │  │ Query BongoPost │  │ Query Bongo    │  │
    │ ::where('hidden',0)│  │ ::where('unpub  │  │ Review::where( │  │
    │ ->with('chunks.    │  │ lish', 0)->get()│  │ 'unpublish',0) │  │
    │  texts')->get()    │  │                 │  │ ->where('review│  │
    │                    │  │                 │  │ state',1)->get()│ │
    └─────────┬──────────┘  └───────┬─────────┘  └────┬──────────┘  │
              │                     │                  │              │
    ┌─────────▼──────────┐  ┌───────▼─────────┐  ┌────▼──────────┐  │
    │ Foreach page:      │  │ Foreach post:   │  │ Foreach review:│  │
    │  new ImportPage    │  │  new ImportPost │  │  new ImportRev│  │
    │  FromBongo()       │  │  FromBongo()    │  │  iewFromBongo()│  │
    │                    │  │                 │  │                │  │
    │ (on exception:     │  │ (on exception:  │  │ (on exception: │  │
    │  log & continue)   │  │  log & continue)│  │  log&continue) │  │
    └────────────────────┘  └─────────────────┘  └────────────────┘  │
                                                                      │
                            ┌─────────────────────────────────────────┘
                            │
                ┌───────────▼──────────┐
                │ Import successful,   │
                │ goodbye :)           │
                └──────────────────────┘
```

## Page Import Lifecycle

```
ImportPageFromBongo::__construct(BongoPage $bongoPage, string $siteUrl)
│
├─▶ findOrCreate()
│   │
│   ├─▶ Page::firstOrNew(['slug' => Str::slug($bongoPage->title)])
│   ├─▶ Set: name, meta_title, meta_description, status
│   └─▶ save()
│
├─▶ saveContent()
│   │
│   ├─▶ Foreach $bongoPage->chunks
│   │   │
│   │   ├─▶ Append view('import::backend.open_section')
│   │   │
│   │   ├─▶ Foreach $chunk->texts
│   │   │   │
│   │   │   ├─▶ cleanHtml($text->raw_text)
│   │   │   │   ├─▶ Remove <br> tags
│   │   │   │   ├─▶ Html::parse(), removeStyles(), removeClasses()
│   │   │   │   ├─▶ Html::removeEmptyTags(), wrapIcons(), wrapShortCodes()
│   │   │   │   ├─▶ Html::removeTags() (keep h1-h6, p, b, i, ul, li, a, img)
│   │   │   │   └─▶ Fix double-wrapped <p> tags
│   │   │   │
│   │   │   ├─▶ updateLinks($html, $siteUrl)
│   │   │   │   ├─▶ Transform ?p=foo → /foo
│   │   │   │   ├─▶ Transform ?review=reviews/slug → /reviews/slug
│   │   │   │   ├─▶ Transform ?blog=blogs/archive/YYYY/MM/DD/name.aspx → /posts/name
│   │   │   │   └─▶ Remove domain from internal links
│   │   │   │
│   │   │   └─▶ Append to $page->content
│   │   │
│   │   └─▶ Append view('import::backend.close_section')
│   │
│   └─▶ save()
│
├─▶ saveImages()
│   │
│   ├─▶ DOMDocument::loadHTML($page->content)
│   ├─▶ getElementsByTagName('img')
│   │
│   ├─▶ Foreach <img>
│   │   │
│   │   ├─▶ updateImageSrc($imageSrc, $siteUrl)
│   │   │   ├─▶ Remove domain
│   │   │   ├─▶ Trim slashes
│   │   │   └─▶ Encode spaces (%20)
│   │   │
│   │   ├─▶ ImageHelper::remoteFileExists($siteUrl.$imageSrc)
│   │   │   └─▶ cURL HEAD request (CURLOPT_NOBODY)
│   │   │
│   │   ├─▶ ImageHelper::store($page, $fullUrl, "pages", Image::WYSIWYG, $key+1)
│   │   │   ├─▶ Download via file_get_contents()
│   │   │   ├─▶ Generate unique filename
│   │   │   ├─▶ Storage::put("/public/pages/{$page->id}/{filename}", $file)
│   │   │   ├─▶ Get image dimensions (getimagesize)
│   │   │   ├─▶ Create Image model record
│   │   │   └─▶ $page->images()->save($dbImage, ['sort_order' => $key])
│   │   │
│   │   └─▶ Replace <img src="..."> with <img src="/photos/{filename}">
│   │
│   └─▶ save()
│
└─▶ saveOpenGraphImages()
    │
    ├─▶ Check if $bongoPage->ogimage exists
    ├─▶ ImageHelper::remoteFileExists("{$siteUrl}/Storage/general/ogimage/originals/{ogimage}")
    ├─▶ ImageHelper::store($page, $fullUrl, "pages", Image::COVER_IMAGE)
    └─▶ (Image relationship auto-saved by helper)
```

## Post Import Lifecycle

```
ImportPostFromBongo::__construct(BongoPost $bongoPost, string $siteUrl)
│
├─▶ findOrCreate()
│   │
│   ├─▶ Post::firstOrNew(['slug' => Str::slug(str_replace('.', '-', $bongoPost->pagename))])
│   ├─▶ Set: name, meta_title, date
│   ├─▶ Set meta_description:
│   │   └─▶ If $bongoPost->pagemeta: trim + limit to 150 chars
│   │       Else: null
│   └─▶ save()
│
├─▶ saveContent()
│   │
│   ├─▶ Append view('import::backend.open_section')
│   ├─▶ Append view('import::backend.section_title', ['title' => $bongoPost->title])
│   ├─▶ cleanHtml($bongoPost->blog)
│   ├─▶ updateLinks($html, $siteUrl)
│   ├─▶ Append to content
│   ├─▶ Append view('import::backend.close_section')
│   └─▶ save()
│
└─▶ saveImages()
    └─▶ (Same logic as ImportPageFromBongo::saveImages)
```

## Review Import Lifecycle

```
ImportReviewFromBongo::__construct(BongoReview $bongoReview)
│
└─▶ Direct mapping (no HTML processing)
    │
    ├─▶ Review::firstOrNew(['title' => $bongoReview->title, 'name' => $bongoReview->contactname])
    ├─▶ Set fields:
    │   - date: $bongoReview->modwhen
    │   - title: $bongoReview->title
    │   - content: $bongoReview->review (raw, no cleaning)
    │   - email: $bongoReview->contactemail
    │   - status: Review::ACTIVE
    │   - rating: round($bongoReview->rank / 2)  # 10-point → 5-star
    └─▶ save()
```

## Database Schema Mapping

### Source Database (Legacy Bongo)

**Connection**: `import` (dynamic, configured at runtime)

#### pages table
```
┌──────────────┬──────────┬────────────────────────────┐
│ Column       │ Type     │ Description                │
├──────────────┼──────────┼────────────────────────────┤
│ id           │ int      │ Primary key                │
│ title        │ string   │ Page title                 │
│ titleexten   │ string   │ Meta title extension       │
│ description  │ text     │ Meta description           │
│ hidden       │ boolean  │ 0=visible, 1=hidden        │
│ ogimage      │ string   │ Open graph image filename  │
└──────────────┴──────────┴────────────────────────────┘
    │
    │ hasMany
    ▼
┌──────────────┬──────────┬────────────────────────────┐
│ chunks table │          │                            │
├──────────────┼──────────┼────────────────────────────┤
│ id           │ int      │ Primary key                │
│ page_id      │ int      │ Foreign key to pages       │
│ order        │ int      │ Sort order                 │
└──────────────┴──────────┴────────────────────────────┘
    │
    │ hasMany
    ▼
┌──────────────┬──────────┬────────────────────────────┐
│ text table   │          │                            │
├──────────────┼──────────┼────────────────────────────┤
│ id           │ int      │ Primary key                │
│ chunk_id     │ int      │ Foreign key to chunks      │
│ text_index   │ int      │ Sort order                 │
│ raw_text     │ text     │ HTML content               │
└──────────────┴──────────┴────────────────────────────┘
```

#### blogposts table
```
┌──────────────┬──────────┬────────────────────────────┐
│ Column       │ Type     │ Description                │
├──────────────┼──────────┼────────────────────────────┤
│ id           │ int      │ Primary key                │
│ title        │ string   │ Post title                 │
│ pagename     │ string   │ URL slug (with dots)       │
│ blog         │ text     │ HTML content               │
│ pagemeta     │ text     │ Meta description (optional)│
│ postdate     │ datetime │ Publication date           │
│ unpublish    │ boolean  │ 0=published, 1=draft       │
└──────────────┴──────────┴────────────────────────────┘
```

#### reviewposts table
```
┌──────────────┬──────────┬────────────────────────────┐
│ Column       │ Type     │ Description                │
├──────────────┼──────────┼────────────────────────────┤
│ id           │ int      │ Primary key                │
│ title        │ string   │ Review title               │
│ contactname  │ string   │ Reviewer name              │
│ contactemail │ string   │ Reviewer email             │
│ review       │ text     │ Review content             │
│ rank         │ int      │ Rating (0-10 scale)        │
│ reviewstate  │ int      │ 1=approved, 0=pending      │
│ unpublish    │ boolean  │ 0=published, 1=draft       │
│ modwhen      │ datetime │ Modified date              │
└──────────────┴──────────┴────────────────────────────┘
```

### Destination Database (Bongo v3.0)

Uses default Laravel connection, models from other packages:

- `Bongo\Page\Models\Page` (pages table)
- `Bongo\Post\Models\Post` (posts table)
- `Bongo\Review\Models\Review` (reviews table)
- `Bongo\Image\Models\Image` (images table)
- Polymorphic: `imageables` pivot table

## HTML Processing Pipeline

### cleanHtml() Method

```
Input: <p style="color:red" class="foo">Hello<br>World</p><h1>Title</h1>

Step 1: Remove <br> variants
  → <p style="color:red" class="foo">HelloWorld</p><h1>Title</h1>

Step 2: Html::parse()
  → Parse into DOM structure

Step 3: Html::removeStyles()
  → <p class="foo">HelloWorld</p><h1>Title</h1>

Step 4: Html::removeClasses()
  → <p>HelloWorld</p><h1>Title</h1>

Step 5: Html::removeEmptyTags()
  → (no change, no empty tags)

Step 6: Html::wrapIcons()
  → Wraps icon markup in special containers

Step 7: Html::wrapShortCodes()
  → Identifies [shortcode] patterns and wraps

Step 8: Html::removeTags() - keep only: h1-h6, p, b, i, ul, li, a, img
  → <p>HelloWorld</p><h1>Title</h1>

Step 9: Fix double-wrapped paragraphs
  → <p>HelloWorld</p><h1>Title</h1>

Output: <p>HelloWorld</p><h1>Title</h1>
```

### updateLinks() Method

```
Input: <a href="https://designtec.co.uk?p=about.us">Link</a>

Step 1: Extract href via regex
  → href="https://designtec.co.uk?p=about.us"

Step 2: Remove domain
  → href="?p=about.us"

Step 3: Check for ?p= pattern
  → Match found: "about.us"

Step 4: Replace dots with dashes, convert to slug
  → "/about-us"

Output: <a href="/about-us">Link</a>

------

Input: <a href="?blog=blogs/archive/2020/03/15/my-post.aspx">Post</a>

Step 1-2: Extract and clean
  → href="?blog=blogs/archive/2020/03/15/my-post.aspx"

Step 3: Check for ?blog= pattern
  → Match found

Step 4: Regex match date pattern (\d{4})/(\d{2})/(\d{2})(.*)
  → Groups: 2020, 03, 15, /my-post.aspx

Step 5: Extract last segment, remove .aspx, slugify
  → "/posts/my-post"

Output: <a href="/posts/my-post">Post</a>
```

## Extension Points

### Adding a New Content Type

To add support for importing a new content type (e.g., products):

1. **Create source model** (`src/Models/Product.php`):
   ```php
   namespace Bongo\Import\Models;

   use Illuminate\Database\Eloquent\Model;

   class Product extends Model
   {
       protected $connection = 'import';
       protected $table = 'products';

       // Define relationships to legacy tables if needed
   }
   ```

2. **Create import service** (`src/Services/ImportProductFromBongo.php`):
   ```php
   namespace Bongo\Import\Services;

   use Bongo\Import\Models\Product as BongoProduct;
   use Bongo\Product\Models\Product;

   class ImportProductFromBongo extends BongoImport
   {
       public function __construct(BongoProduct $bongoProduct, string $siteUrl)
       {
           $product = $this->findOrCreate($bongoProduct);
           $this->saveContent($bongoProduct, $product, $siteUrl);
           $this->saveImages($product, $siteUrl);
       }

       // Implement findOrCreate, saveContent, saveImages methods
   }
   ```

3. **Add import method to ImportCommand** (`src/Commands/ImportCommand.php`):
   ```php
   if ($this->confirm('Import products?')) {
       $this->importProducts();
   }

   private function importProducts(): void
   {
       if (!package()->isEnabled('product')) {
           $this->error('Product module is not enabled');
           return;
       }

       $bongoProducts = BongoProduct::query()->get();

       foreach ($bongoProducts as $bongoProduct) {
           $this->comment("- Importing {$bongoProduct->name} product");

           try {
               new ImportProductFromBongo($bongoProduct, $this->siteUrl);
           } catch (Throwable|Exception $e) {
               $this->error($e->getMessage());
               log_exception($e);
               continue;
           }
       }

       $this->comment("Products have been imported");
   }
   ```

4. **Update ResetCommand** to truncate new tables (`src/Commands/ResetCommand.php`):
   ```php
   if (Schema::hasTable('products')) {
       DB::table('products')->truncate();
   }
   if (Storage::exists('/public/products')) {
       Storage::deleteDirectory('/public/products');
   }
   ```

### Customising HTML Cleaning

Override `cleanHtml()` in specific service class:

```php
class ImportPageFromBongo extends BongoImport
{
    protected function cleanHtml(string $html): string
    {
        // Call parent cleaning first
        $html = parent::cleanHtml($html);

        // Add page-specific transformations
        $html = str_replace('[LEGACY_TAG]', '<div class="special"></div>', $html);

        return $html;
    }
}
```

### Adding Custom URL Transformations

Extend `updateLinks()` in `BongoImport.php`:

```php
protected function updateLinks(string $html, string $siteUrl): string
{
    $siteUrl = str_replace(['https://', 'http://', 'www.'], '', $siteUrl);

    return preg_replace_callback('/href=[\'"](.*?)[\'"]/i', function ($matches) use ($siteUrl) {
        $url = $matches[1];

        // ... existing transformations ...

        // Add new transformation
        if (Str::contains($url, '?product=')) {
            $url = str_replace('?product=', '/products/', $url);
            $url = Str::slug($url);
        }

        return 'href="'.$url.'"';
    }, $html);
}
```

## Configuration

### Config File (`src/Config/import.php`)

```php
return [
    'site_url' => 'https://designtec.co.uk',  // Default remote site
    'db_host' => '127.0.0.1',                  // Default database host
    'db_name' => 'designtec',                  // Default database name
    'db_username' => 'designtec',              // Default username
    'db_password' => 'designtec',              // Default password
];
```

**Usage**: Values are used as defaults in `ImportCommand::handle()` prompts. User can override during interactive session.

### Dynamic Connection Setup

The `import` database connection is created at runtime:

```php
// src/Commands/ImportCommand.php:73-89
private function setDatabaseConnection(): void
{
    config()->set('database.connections.import', [
        'driver' => 'mysql',
        'port' => '3306',
        'charset' => 'utf8mb4',
        'collation' => 'utf8mb4_unicode_ci',
        'engine' => 'InnoDB',
        'host' => $this->dbHost,
        'database' => $this->dbName,
        'username' => $this->dbUsername,
        'password' => $this->dbPassword,
    ]);

    // Refresh the import connection
    DB::purge('import');
}
```

## Dependencies

### Composer Dependencies

```json
{
    "require": {
        "php": ">=8.2",
        "illuminate/contracts": "^10.0",
        "bongo/framework": "^3.0"
    }
}
```

### Package Relationships

- **bongo/framework**: Provides `AbstractServiceProvider`, `Html` helper, `File` helper
- **bongo/image**: Provides `Image` model and polymorphic image relationships
- **bongo/page**: Provides destination `Page` model (optional check: not enforced)
- **bongo/post**: Provides destination `Post` model (checked via `package()->isEnabled('post')`)
- **bongo/review**: Provides destination `Review` model (checked via `package()->isEnabled('review')`)

### Framework Helpers Used

- `console_print(string $message, string $level = 'info')` - CLI output helper
- `log_exception(Throwable $e)` - Exception logging helper
- `package()->isEnabled(string $name): bool` - Package availability checker
- `user()` - Current user helper (for image created_by/updated_by)

## Testing

### Test Setup (`tests/TestCase.php`)

```php
class TestCase extends \Orchestra\Testbench\TestCase
{
    protected function getPackageProviders($app)
    {
        return [ImportServiceProvider::class];
    }

    protected function getEnvironmentSetUp($app)
    {
        // Configure test environment
        // Set up test database connections
    }
}
```

### Running Tests

```bash
# Run all tests
vendor/bin/phpunit

# Run with coverage
vendor/bin/phpunit --coverage-html coverage/
```

## Security Considerations

### Database Credentials
- Prompted interactively (not stored in version control)
- Config defaults should be changed for production
- No encryption/hashing of stored credentials

### Image Downloads
- Uses `CURLOPT_SSL_VERIFYPEER = 0` (development only!)
- Downloads arbitrary remote files (validate source)
- No file type validation (relies on getimagesize)

### HTML Content
- No XSS sanitisation (assumes trusted source database)
- HTML cleaning removes styles/classes but preserves structure
- Links rewritten but not validated for safety

## Troubleshooting

### Connection Issues
**Problem**: "Could not connect to the database"
**Solution**: Verify host, database name, credentials; check firewall/network access

### Missing Images
**Problem**: Images not importing, "Remote image not found"
**Solution**: Verify site URL is correct, check remote image paths, confirm network access

### Duplicate Content
**Problem**: Running import multiple times creates duplicates
**Solution**: Use `import:reset` command before re-importing, or modify `firstOrNew` logic to use unique constraints

### Memory Issues
**Problem**: PHP memory exhausted on large imports
**Solution**: Chunk queries, increase PHP memory_limit, process in batches

### HTML Corruption
**Problem**: Content looks broken after import
**Solution**: Check HTML cleaning rules, verify allowed tags list, inspect legacy markup structure
