<?php

namespace Bongo\Import\Services;

use Bongo\Image\Models\Image;
use Bongo\Import\Helpers\ImageHelper;
use Bongo\Import\Models\Page as BongoPage;
use Bongo\Page\Models\Page;
use DOMDocument;
use Illuminate\Support\Str;
use Throwable;

class ImportPageFromBongo extends BongoImport
{
    /**
     * @throws Throwable
     */
    public function __construct(BongoPage $bongoPage, string $siteUrl)
    {
        $page = $this->findOrCreate($bongoPage);
        $this->saveContent($bongoPage, $page, $siteUrl);
        $this->saveImages($page, $siteUrl);
        $this->saveOpenGraphImages($bongoPage, $page, $siteUrl);
    }

    private function findOrCreate(BongoPage $bongoPage): Page
    {
        $page = Page::firstOrNew([
            'slug' => Str::slug($bongoPage->title),
        ]);
        $page->name = $bongoPage->title;
        $page->meta_title = $bongoPage->title.' '.$bongoPage->titleexten;
        $page->meta_description = $bongoPage->description;
        $page->status = Page::ACTIVE;
        $page->save();

        return $page;
    }

    /**
     * @throws Throwable
     */
    private function saveContent(BongoPage $bongoPage, Page $page, string $siteUrl): void
    {
        $page->content = '';

        if (! is_null($bongoPage->chunks) && count($bongoPage->chunks)) {
            console_print("-- Saving page chunks");

            foreach ($bongoPage->chunks as $chunk) {
                if (! is_null($chunk->texts) && count($chunk->texts)) {

                    $page->content .= view('import::backend.open_section')->render();

                    foreach ($chunk->texts as $text) {
                        if (! $text || empty($text->raw_text)) {
                            continue;
                        }
                        $html = $this->cleanHtml($text->raw_text);
                        $html = $this->updateLinks($html, $siteUrl);
                        $page->content .= $html;
                    }

                    $page->content .= view('import::backend.close_section')->render();
                }
            }

            $page->save();
        }
    }

    private function saveImages(Page $page, string $siteUrl): void
    {
        libxml_use_internal_errors(true);

        $domDocument = new DOMDocument();
        $domDocument->loadHTML('<html>'.$page->content.'</html>');
        $images = $domDocument->getElementsByTagName('img');

        if ($images && count($images)) {
            $html = $page->content;

            foreach ($images as $key => $image) {
                if (! $imageSrc = $image->getAttribute('src')) {
                    console_print("!! Image does not contain a valid src tag !!");
                    continue;
                }

                // Remove the domain name etc. from the image src
                $imageSrc = $this->updateImageSrc($imageSrc, $siteUrl);

                // Make sure the remote image exists
                console_print("-- Checking for remote image {$siteUrl}{$imageSrc}");
                if (! ImageHelper::remoteFileExists($siteUrl.$imageSrc)) {
                    console_print("!! Remote image {$siteUrl}{$imageSrc} not found !!");
                    continue;
                }

                console_print("-- Downloading image {$siteUrl}{$imageSrc}");
                $dbImage = ImageHelper::store($page, $siteUrl.$imageSrc, "pages", Image::WYSIWYG, $key + 1);
                $html = str_replace($imageSrc, "/photos/{$dbImage->getFileName()}", $html);
            }

            $page->content = $html;
            $page->save();
        }
    }

    private function saveOpenGraphImages(BongoPage $bongoPage, Page $page, string $siteUrl): void
    {
        $imagePath = 'Storage/general/ogimage/originals';

        if ($siteUrl
            && isset($bongoPage->ogimage)
            && ! empty($bongoPage->ogimage)
            && ImageHelper::remoteFileExists("{$siteUrl}/{$imagePath}/$bongoPage->ogimage")
        ) {
            console_print("-- Downloading image {$bongoPage->ogimage}");
            ImageHelper::store($page, "{$siteUrl}/{$imagePath}/$bongoPage->ogimage", "pages", Image::COVER_IMAGE);
        }
    }
}
