<?php

namespace Bongo\Import\Services;

use Bongo\Image\Models\Image;
use Bongo\Import\Helpers\ImageHelper;
use Bongo\Import\Models\Post as BongoPost;
use Bongo\Post\Models\Post;
use DOMDocument;
use Illuminate\Support\Str;
use Throwable;

class ImportPostFromBongo extends BongoImport
{
    /**
     * @throws Throwable
     */
    public function __construct(BongoPost $bongoPost, string $siteUrl)
    {
        $post = $this->findOrCreate($bongoPost);
        $this->saveContent($bongoPost, $post, $siteUrl);
        $this->saveImages($post, $siteUrl);
    }

    private function findOrCreate(BongoPost $bongoPost): Post
    {
        $post = Post::firstOrNew([
            'slug' => Str::slug(str_replace('.', '-', $bongoPost->pagename)),
        ]);
        $post->name = $bongoPost->title;
        $post->meta_title = $bongoPost->title;

        $post->meta_description = $bongoPost->pagemeta
            ? trim(Str::limit(trim(preg_replace('/\s\s+/', ' ', strip_tags($bongoPost->blog))), 150))
            : null;

        $post->status = Post::ACTIVE;
        $post->date = $bongoPost->postdate ?? null;
        $post->save();

        return $post;
    }

    /**
     * @throws Throwable
     */
    private function saveContent(BongoPost $bongoPost, Post $post, string $siteUrl): void
    {
        $post->content = '';
        $post->content .= view('import::backend.open_section')->render();
        $post->content .= view('import::backend.section_title', ['title' => $bongoPost->title])->render();

        if (! empty($bongoPost->blog)) {
            $html = $this->cleanHtml($bongoPost->blog);
            $html = $this->updateLinks($html, $siteUrl);
            $post->content .= $html;
        }

        $post->content .= view('import::backend.close_section')->render();

        $post->save();
    }

    private function saveImages(Post $post, string $siteUrl): void
    {
        libxml_use_internal_errors(true);

        $domDocument = new DOMDocument();
        $domDocument->loadHTML('<html>'.$post->content.'</html>');
        $images = $domDocument->getElementsByTagName('img');

        if ($images && count($images)) {
            $html = $post->content;

            foreach ($images as $key => $image) {
                if (! $imageSrc = $image->getAttribute('src')) {
                    console_print("!! Image does not contain a valid src tag !!");
                    continue;
                }

                // Remove the domain name etc. from the image src
                $imageSrc = $this->updateImageSrc($imageSrc, $siteUrl);

                // Make sure the remote image exists
                console_print("-- Checking for remote image {$siteUrl}{$imageSrc}");
                if (! ImageHelper::remoteFileExists($siteUrl.$imageSrc)) {
                    console_print("!! Remote image {$siteUrl}{$imageSrc} not found !!", 'error');
                    continue;
                }

                console_print("-- Downloading image {$siteUrl}{$imageSrc}");
                $dbImage = ImageHelper::store($post, $siteUrl.$imageSrc, "posts", Image::WYSIWYG, $key + 1);
                $html = str_replace($image->getAttribute('src'), "/photos/{$dbImage->getFileName()}", $html);
            }

            $post->content = $html;
            $post->save();
        }
    }
}
