# GitHub Copilot Instructions - Bongo Install Package

## Project Overview

The `bongo/install` package is a Laravel installation wizard that guides users through setting up a new Bongo application. It provides a web-based interface for checking system requirements, configuring environment variables, setting up the database, and creating an initial developer account.

**Namespace**: `Bongo\Install`
**Framework**: Laravel 10+ / Bongo Framework 3.0
**PHP**: 8.2+

---

## Key Classes and Relationships

### Controllers (src/Http/Controllers/)

#### WelcomeController
**Purpose**: Initial landing page, publishes config and assets
**Key Methods**:
```php
public function index() // Publishes vendor assets, sets session step to 1
```

#### RequirementsController
**Purpose**: Validates PHP version and required extensions
**Dependencies**: `RequirementsChecker`
**Key Methods**:
```php
public function index() // Checks requirements, renders results
```

#### PermissionsController
**Purpose**: Validates directory permissions
**Dependencies**: `PermissionsChecker`
**Key Methods**:
```php
public function index() // Checks folder permissions, renders results
```

#### EnvironmentController
**Purpose**: Configures .env and robots.txt files
**Dependencies**: `EnvironmentManager`, `RobotsManager`
**Key Methods**:
```php
public function index()                                      // Display environment form
public function store(StoreEnvironmentRequest $request)     // Save .env and robots.txt
private function checkDatabaseConnection(Request $request)  // Validate DB credentials
```

#### SummaryController
**Purpose**: Runs migrations, seeds data, creates developer account
**Dependencies**: `DatabaseManager`, `EnvironmentManager`
**Key Methods**:
```php
public function index()            // Run migrations, seed, create developer account
public function store(User $user)  // Mark installation complete, log in developer
```

---

### Helpers (src/Helpers/)

#### RequirementsChecker
**Purpose**: Check PHP version and extensions
**Key Methods**:
```php
public function check(array $requirements): array              // Check php/apache extensions
public function checkPhpVersion(string $minPhpVersion): array  // Validate PHP version
```
**Implementation Pattern**:
```php
if (!extension_loaded($requirement)) {
    $results['requirements']['php'][$requirement] = false;
}
```

#### PermissionsChecker
**Purpose**: Validate directory permissions
**Key Methods**:
```php
public function check(array $folders): array          // Check folder permissions
private function getPermission(string $folder): string // Get octal permission string
```
**Implementation Pattern**:
```php
$permissions = fileperms($folderPath);
$permissionsAsString = sprintf('%o', $permissions);
return substr($permissionsAsString, -3);
```

#### EnvironmentManager
**Purpose**: Read/write .env file
**Key Methods**:
```php
public function get(): string                   // Read .env file contents
public function getAsArray(): array             // Parse .env into associative array
public function set(Request $request): mixed    // Generate and write .env file
public function prepend($value): mixed          // Append value to .env
```
**Implementation Pattern**:
```php
$envFileData = "APP_NAME=Bongo\n";
$envFileData .= "APP_ENV={$request->environment}\n";
$envFileData .= 'APP_KEY=base64:'.base64_encode(Str::random(32))."\n";
file_put_contents($this->envPath, $envFileData);
```

#### DatabaseManager
**Purpose**: Database setup and seeding
**Key Methods**:
```php
public function reset(): self                          // Drop non-default tables
public function migrateAndSeed(): string               // Run migrations and seeders
private function migrate(): void                       // Run package migrations
private function seed(): void                          // Run package seeders
public function createDeveloperAccount(): array        // Create developer user
private function generatePassword(): string            // Generate random password
```
**Implementation Pattern**:
```php
// Reset database
Schema::disableForeignKeyConstraints();
foreach ($currentTables as $table) {
    if (!in_array($tableName, $defaultTables)) {
        Schema::drop($tableName);
    }
}
Schema::enableForeignKeyConstraints();

// Create developer account
$randomPassword = $this->generatePassword();
$user = new User();
$user->type = User::DEVELOPER;
$user->password = Hash::make($randomPassword);
$user->save();
return [$user, $randomPassword];
```

#### RobotsManager
**Purpose**: Generate robots.txt file
**Key Methods**:
```php
public function get(): false|string                         // Read robots.txt
public function set(StoreEnvironmentRequest $request): void // Write robots.txt
```

#### ConsoleManager
**Purpose**: Execute shell commands
**Key Methods**:
```php
public function runCommand(string $command): string          // Execute shell command
public function runPhpCommand(string $command): string       // Execute PHP command
public function runComposerCommand(string $command): string  // Execute Composer command
```

---

### Form Requests (src/Http/Requests/)

#### StoreEnvironmentRequest
**Purpose**: Validate environment configuration form
**Key Methods**:
```php
public function rules(): array    // Validation rules
public function messages(): array // Custom error messages
```
**Validation Rules**:
```php
'app_url' => ['required', 'url'],
'database_host' => ['required', 'string', 'max:50', 'regex:/^[A-Za-z0-9:\/\/._-]+$/i'],
'database_password' => ['nullable', 'string', 'max:50', 'regex:/^[A-Za-z0-9#?!@$%^&*~._-]+$/i'],
'mail_username' => ['required', 'string', 'max:50', 'email'],
```

---

### View Composers (src/Http/ViewComposers/)

#### StepComposer
**Purpose**: Provide step navigation data to views
**Key Methods**:
```php
public function compose(View $view): void // Add steps array to view
```
**Implementation**:
```php
$view->with('steps', [
    'welcome',
    'requirements',
    'permissions',
    'environment',
    'summary',
]);
```

---

### Service Provider

#### InstallServiceProvider
**Purpose**: Bootstrap package components
**Extends**: `Bongo\Framework\Providers\AbstractServiceProvider`
**Key Properties**:
```php
protected string $module = 'install';

protected array $composers = [
    StepComposer::class => ['install::layouts.partials.steps'],
];
```
**Key Methods**:
```php
public function boot(): void // Call parent boot, load helpers
```

---

## Code Style Templates

### Controller Method Pattern
```php
public function index()
{
    // Inject dependencies via constructor
    $data = $this->helper->check(config('install.key'));

    // Update session step
    session()->put('step', X);

    // Return view with compact data
    return view('install::template', compact('data'));
}
```

### Store/Update Pattern
```php
public function store(Request $request, Redirector $redirect)
{
    // Validate early
    if (!$this->validate($request)) {
        return $redirect
            ->route('install.step')
            ->withInput()
            ->withErrors(['key' => trans('install::frontend.error')]);
    }

    // Process data
    $results = $this->manager->set($request);

    // Redirect with flash data
    return $redirect
        ->route('install.next')
        ->with(['results' => $results]);
}
```

### Helper Check Pattern
```php
public function check(array $config): array
{
    $results = [];

    foreach ($config as $key => $value) {
        $results[$key] = true;
        if (!$this->validate($key, $value)) {
            $results[$key] = false;
            $results['errors'] = true;
        }
    }

    return $results;
}
```

### Environment File Writing
```php
public function set(Request $request): mixed
{
    $envFileData = '';
    $envFileData .= "KEY={$request->value}\n";
    $envFileData .= "OTHER_KEY='{$request->other}'\n\n";

    try {
        file_put_contents($this->envPath, $envFileData);
        return trans('install::frontend.success');
    } catch (Exception $e) {
        Log::error($e->getMessage());
        return trans('install::frontend.errors');
    }
}
```

### Database Migration Pattern
```php
private function migrate(): void
{
    Artisan::call('migrate', ['--force' => true], $this->outputLog);
    Artisan::call('migrate', ['--path' => '/vendor/bongo/package/src/Migrations'], $this->outputLog);
}
```

### Database Seeding Pattern
```php
private function seed(): void
{
    Artisan::call('db:seed', ['--force' => true]);

    if (Schema::hasTable('table_name')) {
        Artisan::call('db:seed', ['--class' => DataSeeder::class, '--force' => true]);
    }
}
```

---

## Common Patterns

### Configuration Access
```php
$minPhpVersion = config('install.core.minPhpVersion');
$requirements = config('install.requirements');
$permissions = config('install.permissions');
```

### Session State Management
```php
session()->put('step', 1);
$currentStep = session('step');
session()->forget('step');
```

### Translation Usage
```php
trans('install::frontend.environment.success')
trans('install::frontend.environment.wizard.form.db_connection_failed')
```

### View Rendering
```php
return view('install::welcome');
return view('install::environment', compact('env'));
```

### Route Naming
```php
route('install.welcome')
route('install.environment.store')
route('install.summary.store', ['user' => $user])
```

### File Path Construction
```php
$this->envPath = base_path('.env');
$this->envExamplePath = base_path('.env.example');
$this->robotsPath = public_path('robots.txt');
```

### Database Connection Testing
```php
config()->set('database.connections.mysql.host', $request->input('database_host'));
DB::purge('mysql');

try {
    DB::connection()->getPdo();
    return true;
} catch (Exception $e) {
    return false;
}
```

---

## Installation Flow

1. **Welcome** → Publish assets, set step 1
2. **Requirements** → Check PHP version and extensions
3. **Permissions** → Validate directory permissions
4. **Environment** → Configure .env and robots.txt
5. **Summary** → Migrate, seed, create developer account, mark complete

Each step increments `session('step')` to track progress.

---

## Important Implementation Details

### Default Tables (Not Dropped on Reset)
```php
$defaultTables = [
    'cache', 'failed_jobs', 'jobs', 'packages',
    'password_reset_tokens', 'sessions', 'settings', 'users',
];
```

### Package Migrations Order
```php
Artisan::call('migrate', ['--path' => '/vendor/bongo/package/src/Migrations']);
Artisan::call('migrate', ['--path' => '/vendor/bongo/setting/src/Migrations']);
Artisan::call('migrate', ['--path' => '/vendor/bongo/redirect/src/Migrations']);
// ... etc
```

### Password Generation
```php
$random = str_shuffle('abcdefghjklmnopqrstuvwxyzABCDEFGHJKLMNOPQRSTUVWXYZ234567890!$%^&!$%^&');
return substr($random, 0, 32);
```

### Installation Complete Flag
```php
$this->environmentManager->prepend('INSTALL_COMPLETE=true');
```

### Post-Installation Cleanup
```php
session()->forget('step');
session()->invalidate();
session()->regenerate();
Artisan::call('cache:clear');
Artisan::call('view:clear');
Auth::loginUsingId($user->id);
```

---

## Type Hints

Always use strict type declarations:

```php
protected string $module = 'install';
protected array $results = [];

public function check(array $folders): array
public function get(): string
public function set(Request $request): mixed
public function createDeveloperAccount(): array
```

---

## Dependencies

### External Packages
- `bongo/framework` - AbstractServiceProvider, base functionality
- `bongo/asset` - Asset publishing
- `bongo/user` - User model
- `bongo/setting` - Settings seeder
- `bongo/menu` - Menu seeder
- `bongo/page` - Page seeder

### Laravel Components
- `Illuminate\Routing\Controller`
- `Illuminate\Foundation\Http\FormRequest`
- `Illuminate\Support\Facades\Artisan`
- `Illuminate\Support\Facades\DB`
- `Illuminate\Support\Facades\Schema`
- `Symfony\Component\Console\Output\BufferedOutput`
- `Symfony\Component\Process\Process`

---

## Testing

Base test case extends Orchestra Testbench:

```php
class TestCase extends \Orchestra\Testbench\TestCase
{
    protected function getPackageProviders(Application $app): array
    {
        return [InstallServiceProvider::class];
    }
}
```

Run tests with: `vendor/bin/phpunit`

---

## Routes Reference

| Route Name | Method | URI | Controller Method |
|------------|--------|-----|-------------------|
| `install.welcome` | GET | `/install` | `WelcomeController@index` |
| `install.requirements` | GET | `/install/requirements` | `RequirementsController@index` |
| `install.permissions` | GET | `/install/permissions` | `PermissionsController@index` |
| `install.environment` | GET | `/install/environment` | `EnvironmentController@index` |
| `install.environment.store` | POST | `/install/environment/store` | `EnvironmentController@store` |
| `install.summary` | GET | `/install/summary` | `SummaryController@index` |
| `install.summary.store` | POST | `/install/summary/store/{user}` | `SummaryController@store` |

---

## File Locations

- **Config**: `src/Config/install.php`
- **Routes**: `src/Routes/web.php`
- **Views**: `src/Views/*.blade.php`
- **Translations**: `src/Translations/en/frontend.php`
- **Controllers**: `src/Http/Controllers/*Controller.php`
- **Helpers**: `src/Helpers/*Manager.php`, `src/Helpers/*Checker.php`
- **Service Provider**: `src/InstallServiceProvider.php`
