# Bongo Install Package - Architecture Documentation

## Overview

The `bongo/install` package provides a web-based installation wizard for Bongo framework applications. It orchestrates a multi-step installation process that validates system requirements, configures the application environment, sets up the database, and creates an initial developer account.

**Package Name**: `bongo/install`
**Namespace**: `Bongo\Install`
**PHP Version**: 8.2+
**Laravel Version**: 10+
**Dependencies**: `bongo/framework ^3.0`

---

## Directory Structure

```
bongo/install/
├── src/
│   ├── Config/
│   │   └── install.php                      # Configuration for requirements, permissions
│   │
│   ├── Http/
│   │   ├── Controllers/
│   │   │   ├── WelcomeController.php        # Step 1: Welcome screen
│   │   │   ├── RequirementsController.php   # Step 2: PHP/extension checks
│   │   │   ├── PermissionsController.php    # Step 3: Directory permissions
│   │   │   ├── EnvironmentController.php    # Step 4: .env configuration
│   │   │   └── SummaryController.php        # Step 5: Migration and account creation
│   │   │
│   │   ├── Requests/
│   │   │   └── StoreEnvironmentRequest.php  # Form validation for environment settings
│   │   │
│   │   └── ViewComposers/
│   │       └── StepComposer.php             # Provides step navigation data
│   │
│   ├── Helpers/
│   │   ├── RequirementsChecker.php          # Validates PHP version and extensions
│   │   ├── PermissionsChecker.php           # Validates directory permissions
│   │   ├── EnvironmentManager.php           # Reads/writes .env file
│   │   ├── DatabaseManager.php              # Runs migrations, seeders, creates users
│   │   ├── RobotsManager.php                # Creates robots.txt file
│   │   └── ConsoleManager.php               # Executes shell commands
│   │
│   ├── Routes/
│   │   └── web.php                          # Web routes for installation wizard
│   │
│   ├── Views/
│   │   ├── layouts/
│   │   │   ├── app.blade.php                # Main layout template
│   │   │   └── partials/
│   │   │       ├── steps.blade.php          # Step navigation wrapper
│   │   │       ├── step_complete.blade.php  # Completed step indicator
│   │   │       ├── step_current.blade.php   # Current step indicator
│   │   │       └── step_upcoming.blade.php  # Upcoming step indicator
│   │   │
│   │   ├── welcome.blade.php                # Welcome page view
│   │   ├── requirements.blade.php           # Requirements check view
│   │   ├── permissions.blade.php            # Permissions check view
│   │   ├── environment.blade.php            # Environment configuration form
│   │   └── summary.blade.php                # Summary and completion view
│   │
│   ├── Translations/
│   │   └── en/
│   │       └── frontend.php                 # English translations
│   │
│   ├── helpers.php                          # Global helper functions
│   └── InstallServiceProvider.php           # Package service provider
│
├── tests/
│   ├── Feature/                             # Feature tests
│   ├── Unit/                                # Unit tests
│   └── TestCase.php                         # Base test case
│
├── .cursorrules                             # Cursor AI instructions
├── .github/
│   └── copilot-instructions.md              # GitHub Copilot instructions
├── ARCHITECTURE.md                          # This file
├── CLAUDE.md                                # Claude Code guidance
├── README.md                                # Package overview
├── composer.json                            # Composer configuration
└── phpunit.xml                              # PHPUnit configuration
```

---

## Class Diagrams

### Controller Hierarchy

```
Illuminate\Routing\Controller
    │
    ├── WelcomeController
    │   └── index() → Publishes vendor assets, sets session step
    │
    ├── RequirementsController
    │   ├── __construct(RequirementsChecker)
    │   └── index() → Validates PHP and extensions
    │
    ├── PermissionsController
    │   ├── __construct(PermissionsChecker)
    │   └── index() → Validates directory permissions
    │
    ├── EnvironmentController
    │   ├── __construct(EnvironmentManager, RobotsManager)
    │   ├── index() → Display environment form
    │   ├── store(StoreEnvironmentRequest) → Save .env file
    │   └── checkDatabaseConnection(Request) → Validate DB credentials
    │
    └── SummaryController
        ├── __construct(DatabaseManager, EnvironmentManager)
        ├── index() → Run migrations, seed, create developer
        └── store(User) → Mark complete, log in developer
```

### Helper Classes

```
RequirementsChecker
├── check(array $requirements): array
│   └── Validates PHP extensions using extension_loaded()
├── checkPhpVersion(string $minPhpVersion): array
│   └── Validates PHP version using version_compare()
└── getPhpVersionInfo(): array (private/static)
    └── Parses PHP_VERSION constant

PermissionsChecker
├── check(array $folders): array
│   └── Validates directory permissions
├── getPermission(string $folder): string (private)
│   └── Returns octal permission string
├── addFile(string, string, bool): void (private)
└── addFileAndSetErrors(string, string, bool): void (private)

EnvironmentManager
├── __construct() → Sets .env and .env.example paths
├── get(): string → Reads .env file contents
├── getAsArray(): array → Parses .env into associative array
├── set(Request $request): mixed → Generates and writes .env file
└── prepend($value): mixed → Appends value to .env

DatabaseManager
├── reset(): self → Drops non-default tables
├── migrateAndSeed(): string → Runs migrations and seeders
├── migrate(): void (private) → Calls Artisan migrate
├── seed(): void (private) → Calls Artisan db:seed
├── createDeveloperAccount(): array → Creates User with random password
└── generatePassword(): string (private) → Generates 32-char password

RobotsManager
├── __construct() → Sets robots.txt path
├── get(): false|string → Reads robots.txt
└── set(StoreEnvironmentRequest): void → Writes robots.txt

ConsoleManager
├── __construct() → Sets PHP and Composer executables
├── runCommand(string): string → Executes shell command
├── runPhpCommand(string): string → Executes PHP command
├── runComposerCommand(string): string → Executes Composer command
├── setPhpExecutable(): void (private)
└── setComposerExecutable(): void (private)
```

### Service Provider

```
Bongo\Framework\Providers\AbstractServiceProvider
    │
    └── InstallServiceProvider
        ├── $module = 'install'
        ├── $composers = [StepComposer::class => [...]]
        └── boot(): void
            ├── parent::boot() → Auto-loads config, routes, views, translations
            └── app->booted(fn => include helpers.php)
```

---

## Installation Lifecycle

### Flow Diagram

```
┌─────────────────────────────────────────────────────────────────┐
│                     Installation Wizard Flow                    │
└─────────────────────────────────────────────────────────────────┘

┌──────────────────┐
│  User visits     │
│  /install        │
└────────┬─────────┘
         │
         ▼
┌──────────────────────────────────────────────────────────────────┐
│ Step 1: Welcome                                                  │
│ ┌──────────────────────────────────────────────────────────────┐ │
│ │ WelcomeController@index                                      │ │
│ │  • Artisan::call('vendor:publish', FrameworkServiceProvider) │ │
│ │  • Artisan::call('vendor:publish', AssetServiceProvider)     │ │
│ │  • session()->put('step', 1)                                 │ │
│ │  • return view('install::welcome')                           │ │
│ └──────────────────────────────────────────────────────────────┘ │
└────────┬─────────────────────────────────────────────────────────┘
         │
         ▼
┌──────────────────────────────────────────────────────────────────┐
│ Step 2: Requirements                                             │
│ ┌──────────────────────────────────────────────────────────────┐ │
│ │ RequirementsController@index                                 │ │
│ │  • RequirementsChecker->checkPhpVersion()                    │ │
│ │  • RequirementsChecker->check(config('install.requirements'))│ │
│ │  • session()->put('step', 2)                                 │ │
│ │  • return view('install::requirements')                      │ │
│ └──────────────────────────────────────────────────────────────┘ │
└────────┬─────────────────────────────────────────────────────────┘
         │
         ▼
┌──────────────────────────────────────────────────────────────────┐
│ Step 3: Permissions                                              │
│ ┌──────────────────────────────────────────────────────────────┐ │
│ │ PermissionsController@index                                  │ │
│ │  • PermissionsChecker->check(config('install.permissions'))  │ │
│ │  • session()->put('step', 3)                                 │ │
│ │  • return view('install::permissions')                       │ │
│ └──────────────────────────────────────────────────────────────┘ │
└────────┬─────────────────────────────────────────────────────────┘
         │
         ▼
┌──────────────────────────────────────────────────────────────────┐
│ Step 4: Environment                                              │
│ ┌──────────────────────────────────────────────────────────────┐ │
│ │ EnvironmentController@index                                  │ │
│ │  • EnvironmentManager->getAsArray()                          │ │
│ │  • session()->put('step', 4)                                 │ │
│ │  • return view('install::environment')                       │ │
│ └──────────────────────────────────────────────────────────────┘ │
│ ┌──────────────────────────────────────────────────────────────┐ │
│ │ EnvironmentController@store (POST)                           │ │
│ │  • checkDatabaseConnection($request)                         │ │
│ │  • RobotsManager->set($request)                              │ │
│ │  • EnvironmentManager->set($request)                         │ │
│ │  • redirect()->route('install.summary')                      │ │
│ └──────────────────────────────────────────────────────────────┘ │
└────────┬─────────────────────────────────────────────────────────┘
         │
         ▼
┌──────────────────────────────────────────────────────────────────┐
│ Step 5: Summary                                                  │
│ ┌──────────────────────────────────────────────────────────────┐ │
│ │ SummaryController@index                                      │ │
│ │  • app()->register(SettingServiceProvider::class)            │ │
│ │  • DatabaseManager->reset()                                  │ │
│ │  • DatabaseManager->migrateAndSeed()                         │ │
│ │  • DatabaseManager->createDeveloperAccount()                 │ │
│ │  • session()->put('step', 5)                                 │ │
│ │  • return view('install::summary')                           │ │
│ └──────────────────────────────────────────────────────────────┘ │
│ ┌──────────────────────────────────────────────────────────────┐ │
│ │ SummaryController@store (POST)                               │ │
│ │  • EnvironmentManager->prepend('INSTALL_COMPLETE=true')      │ │
│ │  • session()->forget('step'), invalidate(), regenerate()     │ │
│ │  • Artisan::call('cache:clear')                              │ │
│ │  • Artisan::call('view:clear')                               │ │
│ │  • Auth::loginUsingId($user->id)                             │ │
│ │  • redirect()->to('/admin/settings')                         │ │
│ └──────────────────────────────────────────────────────────────┘ │
└────────┬─────────────────────────────────────────────────────────┘
         │
         ▼
┌──────────────────┐
│  Installation    │
│  Complete        │
└──────────────────┘
```

---

## Database Migration Flow

### Migration Execution Order

```
DatabaseManager->migrateAndSeed()
│
├── migrate()
│   ├── Artisan::call('migrate', ['--force' => true])
│   ├── Artisan::call('migrate', ['--path' => '/vendor/bongo/package/src/Migrations'])
│   ├── Artisan::call('migrate', ['--path' => '/vendor/bongo/setting/src/Migrations'])
│   ├── Artisan::call('migrate', ['--path' => '/vendor/bongo/redirect/src/Migrations'])
│   ├── Artisan::call('migrate', ['--path' => '/vendor/bongo/document/src/Migrations'])
│   ├── Artisan::call('migrate', ['--path' => '/vendor/bongo/form/src/Migrations'])
│   ├── Artisan::call('migrate', ['--path' => '/vendor/bongo/image/src/Migrations'])
│   ├── Artisan::call('migrate', ['--path' => '/vendor/bongo/menu/src/Migrations'])
│   ├── Artisan::call('migrate', ['--path' => '/vendor/bongo/page/src/Migrations'])
│   └── Artisan::call('migrate', ['--path' => '/vendor/bongo/user/src/Migrations'])
│
└── seed()
    ├── Artisan::call('db:seed', ['--force' => true])
    ├── IF Schema::hasTable('settings')
    │   └── Artisan::call('db:seed', ['--class' => SettingDataSeeder::class])
    ├── IF Schema::hasTable('menus')
    │   └── Artisan::call('db:seed', ['--class' => MenuDataSeeder::class])
    └── IF Schema::hasTable('pages')
        └── Artisan::call('db:seed', ['--class' => PageDataSeeder::class])
```

### Database Reset Logic

```
DatabaseManager->reset()
│
├── Schema::disableForeignKeyConstraints()
│
├── $defaultTables = [
│   'cache', 'failed_jobs', 'jobs', 'packages',
│   'password_reset_tokens', 'sessions', 'settings', 'users'
│   ]
│
├── DB::select('SHOW TABLES')
│   └── foreach table
│       └── IF NOT in $defaultTables
│           └── Schema::drop($tableName)
│
└── Schema::enableForeignKeyConstraints()
```

---

## Component Relationships

### Dependency Graph

```
InstallServiceProvider
    │
    ├──> StepComposer
    │       └──> View Partials (steps navigation)
    │
    └──> Controllers
            │
            ├──> WelcomeController
            │       └──> Artisan (vendor:publish)
            │
            ├──> RequirementsController ──> RequirementsChecker
            │                                    └──> config('install.requirements')
            │
            ├──> PermissionsController ──> PermissionsChecker
            │                                   └──> config('install.permissions')
            │
            ├──> EnvironmentController ──> EnvironmentManager
            │                           └──> RobotsManager
            │                           └──> StoreEnvironmentRequest
            │                           └──> DB (connection testing)
            │
            └──> SummaryController ──> DatabaseManager
                                    └──> EnvironmentManager
                                    └──> User Model (bongo/user)
                                    └──> Seeders (setting, menu, page)
```

---

## Traits and Interfaces Reference

This package does not define custom traits or interfaces. It uses standard Laravel components:

| Component | Type | Purpose |
|-----------|------|---------|
| `Illuminate\Routing\Controller` | Base Class | All controllers extend this |
| `Illuminate\Foundation\Http\FormRequest` | Base Class | `StoreEnvironmentRequest` extends this |
| `Bongo\Framework\Providers\AbstractServiceProvider` | Base Class | `InstallServiceProvider` extends this |
| `Orchestra\Testbench\TestCase` | Base Class | `TestCase` extends this for package testing |

---

## Extension Points

### 1. Adding New Installation Steps

**Location**: `src/Http/Controllers/`, `src/Routes/web.php`, `src/Views/`, `src/Http/ViewComposers/StepComposer.php`

**Steps**:
1. Create new controller extending `Illuminate\Routing\Controller`
2. Add route to `src/Routes/web.php` with `install.*` naming
3. Create view in `src/Views/`
4. Update `StepComposer` to include new step
5. Set session step counter in controller

**Example**:
```php
// src/Http/Controllers/CustomStepController.php
class CustomStepController extends Controller
{
    public function index()
    {
        session()->put('step', 6);
        return view('install::custom_step');
    }
}

// src/Routes/web.php
Route::get('custom-step', [CustomStepController::class, 'index'])
    ->name('custom-step');

// src/Http/ViewComposers/StepComposer.php
$view->with('steps', [
    'welcome', 'requirements', 'permissions',
    'environment', 'summary', 'custom-step'
]);
```

### 2. Customising Requirements

**Location**: `src/Config/install.php`

**Modify**:
```php
'requirements' => [
    'php' => [
        'pdo',
        'mbstring',
        'gd',        // Add new extension
        'imagick',   // Add new extension
    ],
    'apache' => [
        'mod_rewrite',
        'mod_ssl',   // Add new Apache module
    ],
],
```

### 3. Customising Permissions

**Location**: `src/Config/install.php`

**Modify**:
```php
'permissions' => [
    'storage/framework/' => '775',
    'storage/logs/' => '775',
    'bootstrap/cache/' => '775',
    'public/uploads/' => '755',  // Add new directory
],
```

### 4. Customising Environment Variables

**Location**: `src/Helpers/EnvironmentManager.php`, method `set()`

**Modify**:
```php
public function set(Request $request): mixed
{
    $envFileData = "APP_NAME=Bongo\n";
    // Add custom variables
    $envFileData .= "CUSTOM_VAR={$request->custom_var}\n";
    // ...existing code
}
```

### 5. Adding Custom Migrations

**Location**: `src/Helpers/DatabaseManager.php`, method `migrate()`

**Modify**:
```php
private function migrate(): void
{
    Artisan::call('migrate', ['--force' => true], $this->outputLog);
    // Add new package migrations
    Artisan::call('migrate', [
        '--path' => '/vendor/bongo/custom-package/src/Migrations'
    ], $this->outputLog);
}
```

### 6. Adding Custom Seeders

**Location**: `src/Helpers/DatabaseManager.php`, method `seed()`

**Modify**:
```php
private function seed(): void
{
    Artisan::call('db:seed', ['--force' => true]);

    if (Schema::hasTable('custom_table')) {
        Artisan::call('db:seed', [
            '--class' => CustomDataSeeder::class,
            '--force' => true
        ]);
    }
}
```

### 7. Customising Developer Account Creation

**Location**: `src/Helpers/DatabaseManager.php`, method `createDeveloperAccount()`

**Modify**:
```php
public function createDeveloperAccount(): array
{
    $randomPassword = $this->generatePassword();

    $user = new User();
    $user->email = config('developer.email');
    // Add custom fields
    $user->custom_field = config('developer.custom_value');
    $user->save();

    return [$user, $randomPassword];
}
```

---

## How to Add New Features

### Adding a New Helper Class

1. **Create the helper class** in `src/Helpers/`:
```php
namespace Bongo\Install\Helpers;

class CustomHelper
{
    public function check(array $config): array
    {
        // Implementation
    }
}
```

2. **Inject into controller** via constructor:
```php
public function __construct(CustomHelper $helper)
{
    $this->helper = $helper;
}
```

3. **Use in controller method**:
```php
public function index()
{
    $results = $this->helper->check(config('install.custom'));
    return view('install::custom', compact('results'));
}
```

### Adding Form Validation

1. **Create form request** in `src/Http/Requests/`:
```php
namespace Bongo\Install\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StoreCustomRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'field' => ['required', 'string', 'max:50'],
        ];
    }
}
```

2. **Inject into controller**:
```php
public function store(StoreCustomRequest $request)
{
    // Validated data available
    $validated = $request->validated();
}
```

### Adding Configuration Options

1. **Add to config file** `src/Config/install.php`:
```php
return [
    'custom' => [
        'option1' => 'value1',
        'option2' => 'value2',
    ],
];
```

2. **Access in code**:
```php
$option = config('install.custom.option1');
```

### Adding Translations

1. **Add to translation file** `src/Translations/en/frontend.php`:
```php
return [
    'custom' => [
        'title' => 'Custom Step',
        'message' => 'This is a custom message',
    ],
];
```

2. **Use in views or controllers**:
```php
{{ trans('install::frontend.custom.title') }}
```

---

## Route Middleware

The installation routes in `src/Routes/web.php` are registered via `AbstractServiceProvider` without additional middleware. This ensures the installer is accessible before authentication is configured.

**Route Structure**:
```php
Route::as('install.')
    ->prefix('install')
    ->group(function () {
        // All routes prefixed with /install and named install.*
    });
```

**Catch-all Route**:
```php
Route::get('{any}', fn () => redirect('install'))
    ->where('any', '.*');
```

This ensures any unmatched route redirects to the installer, preventing bypassing the installation process.

---

## Session State Management

The wizard uses Laravel sessions to track progress:

| Session Key | Purpose | Values |
|-------------|---------|--------|
| `step` | Current installation step | 1-5 (integer) |

**Usage**:
```php
// Set step
session()->put('step', 1);

// Get step
$currentStep = session('step');

// Clear step (installation complete)
session()->forget('step');
```

---

## Configuration Schema

### install.php

```php
return [
    // Core requirements
    'core' => [
        'minPhpVersion' => '8.2.0',  // Minimum PHP version (string)
    ],

    // Final step options
    'final' => [
        'key' => true,      // Generate application key (boolean)
        'publish' => false, // Publish config files (boolean)
    ],

    // System requirements
    'requirements' => [
        'php' => [
            // Array of PHP extension names (strings)
        ],
        'apache' => [
            // Array of Apache module names (strings)
        ],
    ],

    // Directory permissions
    'permissions' => [
        // 'path/' => 'octal_permission' (key-value pairs)
    ],
];
```

---

## External Package Dependencies

| Package | Purpose | Usage Location |
|---------|---------|----------------|
| `bongo/framework` | AbstractServiceProvider, base functionality | `InstallServiceProvider` |
| `bongo/asset` | Asset management | `WelcomeController` (vendor:publish) |
| `bongo/user` | User model | `DatabaseManager->createDeveloperAccount()` |
| `bongo/setting` | Settings seeder | `DatabaseManager->seed()` |
| `bongo/menu` | Menu seeder | `DatabaseManager->seed()` |
| `bongo/page` | Page seeder | `DatabaseManager->seed()` |

---

## Security Considerations

### Input Validation

- All user input validated via `StoreEnvironmentRequest`
- Regex validation for database and mail credentials
- URL validation for app URL
- Alpha validation for environment type

### Database Connection Testing

- Credentials tested before saving to .env
- Connection purged and retried with new credentials
- Exceptions caught and handled gracefully

### Password Generation

- Random 32-character passwords
- Character set: `a-zA-Z0-9!$%^&`
- Uses `str_shuffle()` for randomisation
- Hashed with `Hash::make()` before storage

### Session Security

- Session invalidated after installation
- Session regenerated to prevent fixation
- Step counter prevents skipping steps

### File Operations

- Try-catch blocks for all file writes
- Errors logged via `Log::error()`
- Graceful fallback to .env.example if .env missing

---

## Testing Architecture

### Base Test Case

```php
namespace Bongo\Install\Tests;

use Orchestra\Testbench\TestCase as OrchestraTestCase;

class TestCase extends OrchestraTestCase
{
    protected function getPackageProviders($app): array
    {
        return [InstallServiceProvider::class];
    }

    protected function getEnvironmentSetUp($app)
    {
        // Environment setup for tests
    }
}
```

### Test Categories

- **Feature Tests** (`tests/Feature/`): Test complete installation workflows
- **Unit Tests** (`tests/Unit/`): Test individual helper classes

### Running Tests

```bash
vendor/bin/phpunit              # Run all tests
vendor/bin/phpunit --filter=RequirementsTest  # Run specific test
```

---

## Related Documentation

- [README.md](README.md) - Package overview and installation
- [CLAUDE.md](CLAUDE.md) - Quick reference for Claude Code
- [.cursorrules](.cursorrules) - Cursor AI instructions
- [.github/copilot-instructions.md](.github/copilot-instructions.md) - GitHub Copilot guidance
