# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with the `bongo/install` package.

---

## Overview

The `bongo/install` package provides a web-based installation wizard for Laravel applications built on the Bongo framework. It guides users through a 5-step process:

1. **Welcome**: Publishes config/assets
2. **Requirements**: Validates PHP version and extensions
3. **Permissions**: Checks directory permissions
4. **Environment**: Configures .env and robots.txt
5. **Summary**: Runs migrations, seeds data, creates developer account

**Namespace**: `Bongo\Install`
**Dependencies**: `bongo/framework ^3.0`
**PHP**: 8.2+ | **Laravel**: 10+

---

## Quick Links

- [ARCHITECTURE.md](ARCHITECTURE.md) - Detailed architecture, class diagrams, lifecycle flows
- [.cursorrules](.cursorrules) - Cursor AI instructions with coding conventions
- [.github/copilot-instructions.md](.github/copilot-instructions.md) - GitHub Copilot code patterns
- [README.md](README.md) - Package overview and installation

---

## Commands

### Testing
```bash
# Run all tests
vendor/bin/phpunit

# Run specific test file
vendor/bin/phpunit tests/Unit/RequirementsCheckerTest.php
```

### Code Quality
```bash
# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint

# Static analysis
vendor/bin/phpstan analyse
```

### Dependencies
```bash
# Install dependencies
composer install

# Update dependencies
rm -f composer.lock && composer update -W
```

---

## Architecture Quick Reference

### Service Provider

Extends `Bongo\Framework\Providers\AbstractServiceProvider`:

```php
class InstallServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'install';  // Config, views, translations namespace

    protected array $composers = [
        StepComposer::class => ['install::layouts.partials.steps'],
    ];

    public function boot(): void
    {
        parent::boot();  // Auto-loads config, routes, views, translations
        $this->app->booted(fn () => include __DIR__.'/helpers.php');
    }
}
```

### Installation Flow

```
Welcome → Requirements → Permissions → Environment → Summary
   ↓           ↓              ↓              ↓           ↓
Step 1      Step 2         Step 3         Step 4      Step 5
Publish     Check PHP      Check dirs     .env file   Migrate
assets      extensions     permissions    robots.txt  Seed
                                                       Create user
```

### Helper Classes

| Class | Purpose | Location |
|-------|---------|----------|
| `RequirementsChecker` | PHP version and extension validation | `src/Helpers/RequirementsChecker.php` |
| `PermissionsChecker` | Directory permission validation | `src/Helpers/PermissionsChecker.php` |
| `EnvironmentManager` | .env file read/write | `src/Helpers/EnvironmentManager.php` |
| `DatabaseManager` | Migrations, seeders, user creation | `src/Helpers/DatabaseManager.php` |
| `RobotsManager` | robots.txt generation | `src/Helpers/RobotsManager.php` |
| `ConsoleManager` | Shell command execution | `src/Helpers/ConsoleManager.php` |

### Controllers

| Controller | Step | Purpose | Methods |
|------------|------|---------|---------|
| `WelcomeController` | 1 | Welcome screen | `index()` |
| `RequirementsController` | 2 | Check requirements | `index()` |
| `PermissionsController` | 3 | Check permissions | `index()` |
| `EnvironmentController` | 4 | Configure environment | `index()`, `store()` |
| `SummaryController` | 5 | Complete installation | `index()`, `store()` |

All controllers located in `src/Http/Controllers/`.

---

## Key Files

| Path | Purpose |
|------|---------|
| `src/InstallServiceProvider.php` | Package service provider |
| `src/Config/install.php` | Requirements, permissions config |
| `src/Routes/web.php` | Web routes (prefix: `/install`) |
| `src/helpers.php` | Global helper functions |
| `src/Http/Requests/StoreEnvironmentRequest.php` | Form validation |
| `src/Http/ViewComposers/StepComposer.php` | Step navigation data |
| `src/Views/*.blade.php` | Blade templates |
| `src/Translations/en/frontend.php` | English translations |
| `tests/TestCase.php` | Base test case |

---

## Code Style Summary

### Type Declarations
Always use typed properties and return types:
```php
protected RequirementsChecker $requirements;

public function check(array $requirements): array
```

### Constructor Injection
Use dependency injection for helpers:
```php
public function __construct(RequirementsChecker $checker)
{
    $this->requirements = $checker;
}
```

### Configuration Access
```php
$minPhpVersion = config('install.core.minPhpVersion');
$requirements = config('install.requirements');
```

### Session Management
Track wizard state using session:
```php
session()->put('step', 1);
$currentStep = session('step');
session()->forget('step');
```

### View Responses
```php
return view('install::welcome');
return view('install::environment', compact('env'));
```

### Route Naming
All routes named with `install.*` prefix:
```php
route('install.welcome')
route('install.environment.store')
route('install.summary.store', ['user' => $user])
```

### Translation Usage
```php
trans('install::frontend.environment.success')
trans('install::frontend.environment.wizard.form.db_connection_failed')
```

### Error Handling
```php
try {
    file_put_contents($this->envPath, $envFileData);
    return trans('install::frontend.environment.success');
} catch (Exception $e) {
    Log::error($e->getMessage());
    return trans('install::frontend.environment.errors');
}
```

---

## Common Tasks

### Adding a New Installation Step

1. Create controller in `src/Http/Controllers/`
2. Add route to `src/Routes/web.php` with `install.*` naming
3. Create view in `src/Views/`
4. Update `StepComposer` to include new step
5. Set session step counter in controller

### Modifying Requirements

Edit `src/Config/install.php`:
```php
'requirements' => [
    'php' => [
        'pdo',
        'mbstring',
        'gd',  // Add new extension
    ],
],
```

### Customising Environment Variables

Modify `EnvironmentManager@set()`:
```php
$envFileData .= "NEW_VAR={$request->new_var}\n";
```

### Adding Database Migrations

Update `DatabaseManager@migrate()`:
```php
Artisan::call('migrate', ['--path' => '/vendor/bongo/newpackage/src/Migrations'], $this->outputLog);
```

---

## How This Package Extends bongo/framework

The package extends `Bongo\Framework\Providers\AbstractServiceProvider` which provides:

- **Auto-loads Config**: `src/Config/install.php` → `config('install')`
- **Auto-loads Routes**: `src/Routes/web.php` → `route('install.*')`
- **Auto-loads Views**: `src/Views/` → `view('install::*')`
- **Auto-loads Translations**: `src/Translations/en/` → `trans('install::frontend.*')`

The `$module` property (`'install'`) determines the namespace for config, views, and translations.

---

## Routes Reference

| Method | URI | Name | Controller |
|--------|-----|------|------------|
| GET | `/install` | `install.welcome` | `WelcomeController@index` |
| GET | `/install/requirements` | `install.requirements` | `RequirementsController@index` |
| GET | `/install/permissions` | `install.permissions` | `PermissionsController@index` |
| GET | `/install/environment` | `install.environment` | `EnvironmentController@index` |
| POST | `/install/environment/store` | `install.environment.store` | `EnvironmentController@store` |
| GET | `/install/summary` | `install.summary` | `SummaryController@index` |
| POST | `/install/summary/store/{user}` | `install.summary.store` | `SummaryController@store` |

**Catch-all**: Any unmatched route redirects to `/install` to prevent bypassing the installation.

---

## Important Implementation Details

### Installation Complete Flag

After installation, `SummaryController@store()` adds to .env:
```php
$this->environmentManager->prepend('INSTALL_COMPLETE=true');
```

Check this flag in middleware to prevent re-running the installer.

### Session Step Management

Each controller increments the session step:
```php
session()->put('step', 1);  // WelcomeController
session()->put('step', 2);  // RequirementsController
// etc.
```

`StepComposer` uses this to render progress navigation.

### Database Connection Testing

`EnvironmentController` validates database credentials before saving:
```php
config()->set('database.connections.mysql.host', $request->input('database_host'));
DB::purge('mysql');
try {
    DB::connection()->getPdo();
    return true;
} catch (Exception $e) {
    return false;
}
```

### Password Generation

Developer accounts use a secure 32-character password:
```php
$random = str_shuffle('abcdefghjklmnopqrstuvwxyzABCDEFGHJKLMNOPQRSTUVWXYZ234567890!$%^&!$%^&');
return substr($random, 0, 32);
```

### Default Tables (Not Dropped on Reset)

```php
$defaultTables = [
    'cache', 'failed_jobs', 'jobs', 'packages',
    'password_reset_tokens', 'sessions', 'settings', 'users',
];
```

Only these tables are preserved when `DatabaseManager->reset()` is called.

---

## Configuration Schema

### src/Config/install.php

```php
return [
    'core' => [
        'minPhpVersion' => '8.2.0',  // Minimum PHP version
    ],

    'final' => [
        'key' => true,      // Generate application key
        'publish' => false, // Publish config files
    ],

    'requirements' => [
        'php' => ['pdo', 'mbstring', 'tokenizer', 'JSON'],
        'apache' => ['mod_rewrite'],
    ],

    'permissions' => [
        'storage/framework/' => '775',
        'storage/logs/' => '775',
        'bootstrap/cache/' => '775',
    ],
];
```

---

## External Dependencies

| Package | Usage |
|---------|-------|
| `bongo/framework` | AbstractServiceProvider |
| `bongo/asset` | Asset publishing |
| `bongo/user` | User model |
| `bongo/setting` | Settings seeder |
| `bongo/menu` | Menu seeder |
| `bongo/page` | Page seeder |

---

## Security Considerations

1. **Database Connection Validation**: Always validate DB credentials before saving
2. **Input Sanitisation**: Use `StoreEnvironmentRequest` for all user input
3. **Regex Validation**: Database/mail credentials validated with regex patterns
4. **Random Password Generation**: Developer accounts use secure random passwords
5. **Session Invalidation**: Session regenerated after installation completes
6. **Catch-all Route**: Prevents bypassing installation steps

---

## Helper Function

### is_active()

Global helper function defined in `src/helpers.php`:

```php
function is_active(string $routeName): bool
{
    return Route::currentRouteName() == strtolower($routeName);
}
```

**Usage**: Check if a route is currently active (for navigation highlighting).

---

## Testing

### Base Test Case

Located at `tests/TestCase.php`:

```php
class TestCase extends \Orchestra\Testbench\TestCase
{
    protected function getPackageProviders(Application $app): array
    {
        return [InstallServiceProvider::class];
    }

    protected function getEnvironmentSetUp(Application $app)
    {
        // Environment setup for tests
    }
}
```

### Test Structure

- **Feature Tests**: `tests/Feature/` - Test complete workflows
- **Unit Tests**: `tests/Unit/` - Test individual helpers

---

## Development Workflow

1. **Make changes** to source code
2. **Run code style fixer**: `vendor/bin/pint`
3. **Run tests**: `vendor/bin/phpunit`
4. **Run static analysis**: `vendor/bin/phpstan analyse`
5. **Commit changes**

---

## Notes

- This package is part of the Bongo monorepo (`/Users/stuart/Packages/bongo/`)
- Each package is a separate git repository
- Packages are published to private Composer repository at `https://designtecpackages.co.uk`
- Use the monorepo scripts for batch operations (see root `CLAUDE.md`)

---

For detailed architecture diagrams, class relationships, and lifecycle flows, see [ARCHITECTURE.md](ARCHITECTURE.md).
