# Bongo Install Package

A Laravel package providing a web-based installation wizard for Bongo framework applications. Guides users through a 5-step process to validate requirements, configure the environment, set up the database, and create an initial developer account.

[![PHP Version](https://img.shields.io/badge/php-%3E%3D8.2-blue)](https://php.net)
[![Laravel Version](https://img.shields.io/badge/laravel-10%2B-red)](https://laravel.com)

---

## Features

- **5-Step Installation Wizard**: Welcome → Requirements → Permissions → Environment → Summary
- **System Requirements Validation**: Checks PHP version and required extensions
- **Directory Permissions Verification**: Validates writable directories
- **Environment Configuration**: Generates .env file with database, mail, and app settings
- **Database Setup**: Runs migrations and seeders automatically
- **Developer Account Creation**: Creates initial user with random secure password
- **robots.txt Generation**: Creates SEO-friendly robots.txt file
- **Session-Based Progress Tracking**: Maintains installation state across steps

---

## Requirements

- PHP 8.2+
- Laravel 10+
- Composer
- MySQL/MariaDB database
- Write permissions on `storage/`, `bootstrap/cache/` directories

---

## Installation

### Via Composer

```bash
composer require bongo/install
```

### Service Provider

The package uses Laravel auto-discovery. If you're using Laravel 5.x, manually add the service provider to `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\Install\InstallServiceProvider::class,
],
```

---

## Usage

### Starting the Installation

Visit `/install` in your browser to begin the installation wizard.

### Installation Steps

1. **Welcome** - Publishes configuration and asset files
2. **Requirements** - Validates PHP version and required extensions
3. **Permissions** - Checks directory write permissions
4. **Environment** - Configures database, mail, and application settings
5. **Summary** - Runs migrations, seeds data, and creates developer account

### After Installation

- The wizard adds `INSTALL_COMPLETE=true` to your `.env` file
- You'll be automatically logged in as the developer
- Redirect to `/admin/settings` for further configuration

---

## Configuration

Publish the configuration file (optional):

```bash
php artisan vendor:publish --provider="Bongo\Install\InstallServiceProvider" --tag="config"
```

### Configuration Options

Edit `config/install.php` to customise:

#### Minimum PHP Version

```php
'core' => [
    'minPhpVersion' => '8.2.0',
],
```

#### Required Extensions

```php
'requirements' => [
    'php' => [
        'pdo',
        'mbstring',
        'tokenizer',
        'JSON',
    ],
    'apache' => [
        'mod_rewrite',
    ],
],
```

#### Directory Permissions

```php
'permissions' => [
    'storage/framework/' => '775',
    'storage/logs/' => '775',
    'bootstrap/cache/' => '775',
],
```

---

## Routes

All routes are prefixed with `/install` and named with `install.*`:

| Method | URI | Name | Description |
|--------|-----|------|-------------|
| GET | `/install` | `install.welcome` | Welcome page |
| GET | `/install/requirements` | `install.requirements` | Requirements check |
| GET | `/install/permissions` | `install.permissions` | Permissions check |
| GET | `/install/environment` | `install.environment` | Environment form |
| POST | `/install/environment/store` | `install.environment.store` | Save environment |
| GET | `/install/summary` | `install.summary` | Installation summary |
| POST | `/install/summary/store/{user}` | `install.summary.store` | Complete installation |

---

## Development

### Running Tests

```bash
vendor/bin/phpunit
```

### Code Style

Check code style:
```bash
vendor/bin/pint --test
```

Fix code style:
```bash
vendor/bin/pint
```

### Static Analysis

```bash
vendor/bin/phpstan analyse
```

---

## Package Structure

```
src/
├── Config/
│   └── install.php                      # Configuration
├── Http/
│   ├── Controllers/
│   │   ├── WelcomeController.php        # Step 1: Welcome
│   │   ├── RequirementsController.php   # Step 2: Requirements
│   │   ├── PermissionsController.php    # Step 3: Permissions
│   │   ├── EnvironmentController.php    # Step 4: Environment
│   │   └── SummaryController.php        # Step 5: Summary
│   ├── Requests/
│   │   └── StoreEnvironmentRequest.php  # Form validation
│   └── ViewComposers/
│       └── StepComposer.php             # Step navigation
├── Helpers/
│   ├── RequirementsChecker.php          # PHP version/extension checker
│   ├── PermissionsChecker.php           # Directory permissions checker
│   ├── EnvironmentManager.php           # .env file manager
│   ├── DatabaseManager.php              # Migration/seeding manager
│   ├── RobotsManager.php                # robots.txt generator
│   └── ConsoleManager.php               # Shell command executor
├── Routes/
│   └── web.php                          # Web routes
├── Views/                               # Blade templates
├── Translations/                        # Translations
├── helpers.php                          # Global helpers
└── InstallServiceProvider.php           # Service provider
```

---

## Dependencies

This package requires:

- **bongo/framework** (^3.0) - Core framework functionality
- **bongo/asset** - Asset publishing
- **bongo/user** - User model for developer account
- **bongo/setting** - Settings seeder
- **bongo/menu** - Menu seeder
- **bongo/page** - Page seeder

---

## Security

### Input Validation

All user input is validated via form requests with strict regex patterns:

- Database credentials: Alphanumeric + special characters
- Mail settings: Email format validation
- URLs: Laravel URL validation

### Database Connection Testing

Database credentials are tested before saving to prevent incomplete installations.

### Password Generation

Developer accounts use securely generated 32-character random passwords with alphanumeric and special characters.

### Session Management

Sessions are invalidated and regenerated after installation to prevent session fixation attacks.

---

## Extending the Package

### Adding a New Step

1. Create controller in `src/Http/Controllers/`
2. Add route to `src/Routes/web.php`
3. Create view in `src/Views/`
4. Update `StepComposer` to include new step
5. Set session step counter in controller

### Customising Requirements

Edit `src/Config/install.php`:

```php
'requirements' => [
    'php' => [
        'gd',      // Add new extension
        'imagick', // Add new extension
    ],
],
```

### Adding Custom Migrations

Update `DatabaseManager@migrate()`:

```php
Artisan::call('migrate', [
    '--path' => '/vendor/your-package/src/Migrations'
], $this->outputLog);
```

---

## Helper Classes

### RequirementsChecker

Validates PHP version and extensions:

```php
$checker = new RequirementsChecker();
$phpInfo = $checker->checkPhpVersion('8.2.0');
$requirements = $checker->check(config('install.requirements'));
```

### PermissionsChecker

Validates directory permissions:

```php
$checker = new PermissionsChecker();
$permissions = $checker->check(config('install.permissions'));
```

### EnvironmentManager

Manages .env file:

```php
$manager = new EnvironmentManager();
$env = $manager->getAsArray();
$result = $manager->set($request);
$manager->prepend('INSTALL_COMPLETE=true');
```

### DatabaseManager

Handles database setup:

```php
$manager = new DatabaseManager();
$manager->reset();
$output = $manager->migrateAndSeed();
[$user, $password] = $manager->createDeveloperAccount();
```

---

## Global Helpers

### is_active()

Check if a route is currently active:

```php
@if(is_active('install.welcome'))
    <span class="active">Welcome</span>
@endif
```

---

## Troubleshooting

### Installation Won't Start

- Ensure `storage/` and `bootstrap/cache/` are writable
- Check PHP version meets minimum requirement (8.2+)
- Verify required PHP extensions are installed

### Database Connection Fails

- Verify database credentials
- Ensure MySQL/MariaDB is running
- Check database user has appropriate permissions

### Session Issues

- Ensure session driver is configured correctly
- Check `storage/framework/sessions/` is writable
- Clear cache: `php artisan cache:clear`

### Installation Incomplete

- Check `storage/logs/laravel.log` for errors
- Verify all required Bongo packages are installed
- Ensure migrations completed successfully

---

## Documentation

- [ARCHITECTURE.md](ARCHITECTURE.md) - Detailed architecture documentation
- [CLAUDE.md](CLAUDE.md) - Quick reference for Claude Code
- [.cursorrules](.cursorrules) - Cursor AI instructions
- [.github/copilot-instructions.md](.github/copilot-instructions.md) - GitHub Copilot guidance

---

## License

MIT License. See [LICENSE](LICENSE) file for details.

---

## Author

**Stuart Elliott**
- Email: stuart.elliott@bespokeuk.com
- Homepage: [https://bespokeuk.com](https://bespokeuk.com)

---

## Contributing

This package is part of the Bongo monorepo. For contribution guidelines, please see the main repository documentation.

---

## Changelog

See [CHANGELOG.md](CHANGELOG.md) for version history and changes.

---

## Support

For issues and questions:
- Create an issue on the repository
- Check the documentation files listed above
- Review the code examples in the `.cursorrules` file
