<?php

namespace Bongo\Install\Helpers;

use Bongo\Menu\Seeders\DataSeeder as MenuDataSeeder;
use Bongo\Page\Seeders\DataSeeder as PageDataSeeder;
use Bongo\Setting\Seeders\DataSeeder as SettingDataSeeder;
use Bongo\User\Models\User;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Schema;
use Symfony\Component\Console\Output\BufferedOutput;

class DatabaseManager
{
    private BufferedOutput $outputLog;

    public function reset(): self
    {
        Schema::disableForeignKeyConstraints();

        $defaultTables = [
            'cache',
            'failed_jobs',
            'jobs',
            'packages',
            'password_reset_tokens',
            'sessions',
            'settings',
            'users',
        ];
        $currentTables = DB::select('SHOW TABLES');

        foreach ($currentTables as $table) {
            $tableArray = get_object_vars($table);
            $tableName = $tableArray[key($tableArray)];

            if (! in_array($tableName, $defaultTables)) {
                Schema::drop($tableName);
            }
        }

        Schema::enableForeignKeyConstraints();

        return $this;
    }

    public function migrateAndSeed(): string
    {
        $this->outputLog = new BufferedOutput();
        $this->migrate();
        $this->seed();

        return $this->outputLog->fetch();
    }

    private function migrate(): void
    {
        Artisan::call('migrate', ['--force' => true], $this->outputLog);

        Artisan::call('migrate', ['--path' => '/vendor/bongo/package/src/Migrations'], $this->outputLog);
        Artisan::call('migrate', ['--path' => '/vendor/bongo/setting/src/Migrations'], $this->outputLog);
        Artisan::call('migrate', ['--path' => '/vendor/bongo/redirect/src/Migrations'], $this->outputLog);
        Artisan::call('migrate', ['--path' => '/vendor/bongo/document/src/Migrations'], $this->outputLog);
        Artisan::call('migrate', ['--path' => '/vendor/bongo/form/src/Migrations'], $this->outputLog);
        Artisan::call('migrate', ['--path' => '/vendor/bongo/image/src/Migrations'], $this->outputLog);
        Artisan::call('migrate', ['--path' => '/vendor/bongo/menu/src/Migrations'], $this->outputLog);
        Artisan::call('migrate', ['--path' => '/vendor/bongo/page/src/Migrations'], $this->outputLog);
        Artisan::call('migrate', ['--path' => '/vendor/bongo/user/src/Migrations'], $this->outputLog);
    }

    private function seed(): void
    {
        Artisan::call('db:seed', ['--force' => true]);

        if (Schema::hasTable('settings')) {
            Artisan::call('db:seed', ['--class' => SettingDataSeeder::class, '--force' => true]);
        }
        if (Schema::hasTable('menus')) {
            Artisan::call('db:seed', ['--class' => MenuDataSeeder::class, '--force' => true]);
        }
        if (Schema::hasTable('pages')) {
            Artisan::call('db:seed', ['--class' => PageDataSeeder::class, '--force' => true], $this->outputLog);
        }
    }

    public function createDeveloperAccount(): array
    {
        // Generate a random password
        $randomPassword = $this->generatePassword();

        // Create the user
        $user = new User();
        $user->email = config('developer.email');
        $user->first_name = config('developer.first_name');
        $user->last_name = config('developer.last_name');
        $user->telephone = config('developer.telephone');
        $user->mobile = config('developer.mobile');
        $user->password = Hash::make($randomPassword);
        $user->type = User::DEVELOPER;
        $user->status = User::ACTIVE;
        $user->save();

        return [$user, $randomPassword];
    }

    private function generatePassword(): string
    {
        $random = str_shuffle('abcdefghjklmnopqrstuvwxyzABCDEFGHJKLMNOPQRSTUVWXYZ234567890!$%^&!$%^&');

        return substr($random, 0, 32);
    }
}
