# Architecture Documentation - Bongo Menu Package

## Table of Contents
1. [System Overview](#system-overview)
2. [Directory Structure](#directory-structure)
3. [Database Schema](#database-schema)
4. [Class Diagrams](#class-diagrams)
5. [Flow Diagrams](#flow-diagrams)
6. [Trait and Interface Reference](#trait-and-interface-reference)
7. [Extension Points](#extension-points)
8. [Adding New Features](#adding-new-features)

---

## System Overview

The Bongo Menu package provides a hierarchical menu management system for Laravel applications. It enables administrators to create menus with nested items that can link to internal entities (pages, posts, projects) or external URLs.

### Key Architectural Patterns

1. **Service Provider Pattern**: Extends `AbstractServiceProvider` from bongo/framework for automatic bootstrapping
2. **Repository Pattern**: Models encapsulate data access logic
3. **Event-Driven**: Uses Laravel's event system for automatic key generation and cache clearing
4. **Polymorphic Relations**: Menu items link to any configured entity type
5. **Hierarchical Data**: Self-referencing parent-child relationships for nested menus
6. **Caching Strategy**: Multi-level caching for performance optimization
7. **View Composer Pattern**: Automatic menu injection into frontend views

### Technology Stack

- **PHP**: 8.2+
- **Laravel**: 10+ or 11+
- **Framework**: bongo/framework ^3.0
- **Testing**: Orchestra Testbench, PHPUnit
- **Code Quality**: Laravel Pint, PHPStan (level 1)

---

## Directory Structure

```
menu/
├── database/
│   ├── factories/
│   │   ├── MenuFactory.php                 # Factory for testing
│   │   └── MenuItemFactory.php             # Factory for testing
│   └── seeders/
│       ├── DataSeeder.php                  # Sample data
│       └── PackageSeeder.php               # Package registration
│
├── src/
│   ├── Config/
│   │   └── menu.php                        # Entity/route mappings
│   │
│   ├── Events/
│   │   └── MenuEventHandler.php            # Eloquent event subscriber
│   │
│   ├── Http/
│   │   ├── Controllers/
│   │   │   ├── Api/
│   │   │   │   ├── MenuController.php      # REST API for menus
│   │   │   │   └── MenuItemController.php  # REST API for menu items
│   │   │   └── Backend/
│   │   │       ├── MenuController.php      # Admin CRUD
│   │   │       └── MenuDatatableController.php  # DataTables
│   │   │
│   │   ├── Requests/
│   │   │   ├── Api/
│   │   │   │   ├── StoreMenu.php
│   │   │   │   ├── UpdateMenu.php
│   │   │   │   ├── StoreMenuItem.php
│   │   │   │   └── UpdateMenuItem.php
│   │   │   └── Backend/
│   │   │       ├── StoreMenuRequest.php
│   │   │       └── UpdateMenuRequest.php
│   │   │
│   │   ├── Resources/
│   │   │   ├── MenuCollection.php          # JSON API collection
│   │   │   ├── MenuItemCollection.php
│   │   │   ├── MenuResource.php            # JSON API resource
│   │   │   └── MenuItemResource.php
│   │   │
│   │   └── ViewComposers/
│   │       └── MenuComposer.php            # Injects menu into views
│   │
│   ├── Listeners/
│   │   └── ClearMenuCache.php              # Cache invalidation listener
│   │
│   ├── Migrations/
│   │   ├── 2019_01_01_000001_create_menus_table.php
│   │   └── 2019_01_01_000002_create_menu_items_table.php
│   │
│   ├── Models/
│   │   ├── Menu.php                        # Menu model
│   │   └── MenuItem.php                    # MenuItem model
│   │
│   ├── Routes/
│   │   ├── api.php                         # /api/menus, /api/menu-items
│   │   └── backend.php                     # /admin/menu
│   │
│   ├── Seeders/
│   │   ├── DataSeeder.php
│   │   └── PackageSeeder.php
│   │
│   ├── Services/
│   │   ├── MenuEntityType.php              # Entity type discovery
│   │   └── MenuItemService.php             # URL generation helpers
│   │
│   ├── Translations/
│   │   └── en/
│   │       └── backend.php                 # Translation strings
│   │
│   ├── Views/
│   │   ├── backend/
│   │   │   ├── index.blade.php             # Menu listing
│   │   │   ├── create.blade.php            # Create menu
│   │   │   ├── edit.blade.php              # Edit menu
│   │   │   ├── show.blade.php              # Menu detail + items
│   │   │   └── partials/
│   │   │       ├── form/
│   │   │       │   └── details.blade.php
│   │   │       ├── menu_item.blade.php
│   │   │       └── sub_menu_item.blade.php
│   │   └── frontend/
│   │       └── partials/
│   │           ├── menu.blade.php          # Generic menu template
│   │           ├── top_menu.blade.php
│   │           ├── main_menu.blade.php
│   │           ├── main_menu_item.blade.php
│   │           ├── main_menu_dropdown_item.blade.php
│   │           ├── sub_menu.blade.php
│   │           ├── footer_menu.blade.php
│   │           ├── footer_menu_2.blade.php
│   │           └── bottom_menu.blade.php
│   │
│   └── MenuServiceProvider.php             # Service provider
│
├── tests/
│   ├── Feature/                            # Integration tests
│   ├── Unit/
│   │   ├── MenuTest.php
│   │   ├── MenuItemTest.php
│   │   └── MenuItemServiceTest.php
│   ├── ExampleTest.php
│   └── TestCase.php                        # Orchestra Testbench setup
│
├── composer.json
├── phpunit.xml
├── phpstan.neon.dist
└── pint.json
```

---

## Database Schema

### Entity Relationship Diagram

```
┌─────────────────────────────────────────┐
│              menus                      │
├─────────────────────────────────────────┤
│ PK  id (unsigned int)                   │
│     uuid (varchar, indexed)             │
│     name (varchar)                      │
│     key (varchar, indexed)              │
│     status (enum: pending/active/       │
│             inactive)                   │
│     created_by (unsigned int, nullable) │
│     updated_by (unsigned int, nullable) │
│     deleted_by (unsigned int, nullable) │
│     created_at (timestamp)              │
│     updated_at (timestamp)              │
│     deleted_at (timestamp, nullable)    │
└─────────────────────────────────────────┘
               │
               │ 1:N (cascade on delete)
               │
               ▼
┌─────────────────────────────────────────┐
│           menu_items                    │
├─────────────────────────────────────────┤
│ PK  id (unsigned int)                   │
│     uuid (varchar, indexed)             │
│ FK  menu_id → menus.id                  │
│ FK  parent_id → menu_items.id (self)    │
│     name (varchar)                      │
│     url (varchar, nullable)             │
│     entity_type (varchar, nullable)     │
│     entity_id (varchar, nullable)       │
│     class (varchar, nullable)           │
│     type (enum: internal/external)      │
│     target (enum: _self/_blank)         │
│     sort_order (unsigned int, indexed)  │
│     created_by (unsigned int, nullable) │
│     updated_by (unsigned int, nullable) │
│     created_at (timestamp)              │
│     updated_at (timestamp)              │
└─────────────────────────────────────────┘
      │                        │
      │ Self-referencing       │ Polymorphic
      │ (parent_id)            │ (entity_type, entity_id)
      │                        │
      ▼                        ▼
┌──────────┐          ┌─────────────────┐
│   Self   │          │  Any Entity     │
│  (child  │          │  (Page, Post,   │
│   items) │          │   Project, etc) │
└──────────┘          └─────────────────┘
```

### Table Details

#### menus

| Column      | Type              | Constraints                  | Purpose                          |
|-------------|-------------------|------------------------------|----------------------------------|
| id          | unsigned int      | PK, auto increment           | Primary key                      |
| uuid        | varchar           | indexed, unique              | Public identifier                |
| name        | varchar           | required                     | Display name                     |
| key         | varchar           | indexed, nullable            | Slug identifier (auto-generated) |
| status      | enum              | pending/active/inactive      | Publication status               |
| created_by  | unsigned int      | nullable, indexed            | User who created                 |
| updated_by  | unsigned int      | nullable, indexed            | User who last updated            |
| deleted_by  | unsigned int      | nullable, indexed            | User who soft deleted            |
| created_at  | timestamp         |                              | Creation timestamp               |
| updated_at  | timestamp         |                              | Last update timestamp            |
| deleted_at  | timestamp         | nullable                     | Soft delete timestamp            |

**Indexes**:
- Primary: `id`
- Index: `uuid`
- Index: `key`
- Index: `created_by`, `updated_by`, `deleted_by`

#### menu_items

| Column      | Type              | Constraints                  | Purpose                          |
|-------------|-------------------|------------------------------|----------------------------------|
| id          | unsigned int      | PK, auto increment           | Primary key                      |
| uuid        | varchar           | indexed, unique              | Public identifier                |
| menu_id     | unsigned int      | FK to menus.id, cascade      | Parent menu                      |
| parent_id   | unsigned int      | nullable, self-referencing   | Parent menu item (for nesting)   |
| name        | varchar           | required                     | Display text                     |
| url         | varchar           | nullable                     | External URL (if type=external)  |
| entity_type | varchar           | nullable                     | Polymorphic type (e.g., "Page")  |
| entity_id   | varchar           | nullable                     | Polymorphic ID                   |
| class       | varchar           | nullable                     | CSS class                        |
| type        | enum              | internal/external            | Link type                        |
| target      | enum              | _self/_blank                 | Link target                      |
| sort_order  | unsigned int      | nullable, indexed            | Display order                    |
| created_by  | unsigned int      | nullable, indexed            | User who created                 |
| updated_by  | unsigned int      | nullable, indexed            | User who last updated            |
| created_at  | timestamp         |                              | Creation timestamp               |
| updated_at  | timestamp         |                              | Last update timestamp            |

**Indexes**:
- Primary: `id`
- Index: `uuid`
- Index: `menu_id`
- Index: `parent_id`
- Index: `sort_order`
- Index: `created_by`, `updated_by`

**Foreign Keys**:
- `menu_id` → `menus.id` (CASCADE on UPDATE and DELETE)

---

## Class Diagrams

### Core Model Hierarchy

```
┌────────────────────────────────────────────┐
│      Bongo\Framework\Models                │
│         AbstractModel                      │
│  (extends Illuminate\Database\Eloquent\    │
│              Model)                        │
└────────────────────────────────────────────┘
                     │
                     │ extends
                     │
      ┌──────────────┴──────────────┐
      │                             │
      ▼                             ▼
┌─────────────┐            ┌────────────────┐
│    Menu     │            │   MenuItem     │
├─────────────┤            ├────────────────┤
│ Traits:     │            │ Traits:        │
│  HasFactory │            │  HasFactory    │
│  HasStatus  │            │  HasRecursive  │
│  HasUUID    │            │  HasUUID       │
│  SoftDeletes│            │                │
├─────────────┤            ├────────────────┤
│ Constants:  │            │ Constants:     │
│  PENDING    │            │  EXTERNAL      │
│  ACTIVE     │            │  INTERNAL      │
│  INACTIVE   │            │  SELF          │
│             │            │  BLANK         │
├─────────────┤            ├────────────────┤
│ Relations:  │            │ Relations:     │
│  items()    │───────────▶│  menu()        │
│  allItems() │            │  parent()      │
│             │            │  children()    │
├─────────────┤            ├────────────────┤
│ Methods:    │            │ Methods:       │
│  hasItems() │            │  isInternal()  │
│  clearCache()            │  isExternal()  │
└─────────────┘            │  isAsset()     │
                           │  getEntity()   │
                           │  getLink()     │
                           │  isActive()    │
                           │  clearCache()  │
                           └────────────────┘
```

### Controller Hierarchy

```
┌──────────────────────────────────────────────┐
│  Bongo\Framework\Http\Controllers            │
│         AbstractController                   │
└──────────────────────────────────────────────┘
                     │
                     │ extends
      ┌──────────────┴──────────────┐
      │                             │
      ▼                             ▼
┌─────────────────────┐   ┌─────────────────────────┐
│  Backend\           │   │  Backend\               │
│  MenuController     │   │  MenuDatatableController│
├─────────────────────┤   ├─────────────────────────┤
│ Methods:            │   │ Methods:                │
│  index()            │   │  index()                │
│  create()           │   │    (returns DataTable   │
│  store()            │   │     JSON)               │
│  show()             │   └─────────────────────────┘
│  edit()             │
│  update()           │
│  destroy()          │
└─────────────────────┘


┌──────────────────────────────────────────────┐
│  Bongo\Framework\Http\Controllers            │
│       AbstractApiController                  │
└──────────────────────────────────────────────┘
                     │
                     │ extends
      ┌──────────────┴──────────────┐
      │                             │
      ▼                             ▼
┌─────────────────────┐   ┌─────────────────────┐
│  Api\               │   │  Api\               │
│  MenuController     │   │  MenuItemController │
├─────────────────────┤   ├─────────────────────┤
│ Methods:            │   │ Methods:            │
│  index()            │   │  index()            │
│  store()            │   │  store()            │
│  show()             │   │  show()             │
│  update()           │   │  update()           │
│  destroy()          │   │  destroy()          │
└─────────────────────┘   └─────────────────────┘
```

### Service Layer

```
┌────────────────────────────┐
│  MenuEntityType            │
│  (Facade)                  │
├────────────────────────────┤
│ Static Methods:            │
│  all(): array              │
│    Returns available       │
│    entity types based on   │
│    enabled packages        │
└────────────────────────────┘

┌────────────────────────────┐
│  MenuItemService           │
├────────────────────────────┤
│ Methods:                   │
│  routeLink($routeName,     │
│    $menuItem): string      │
│    Generates route URL     │
│    with error handling     │
│                            │
│  groupChildrenAsArray(     │
│    $menuItem): array       │
│    Groups children by      │
│    'group' attribute       │
└────────────────────────────┘
```

### Event System

```
┌────────────────────────────────────────────┐
│  Bongo\Framework\Events                    │
│       AbstractEventHandler                 │
└────────────────────────────────────────────┘
                     │
                     │ extends
                     ▼
┌────────────────────────────────────────────┐
│  MenuEventHandler                          │
├────────────────────────────────────────────┤
│ Properties:                                │
│  protected string $model = Menu::class     │
├────────────────────────────────────────────┤
│ Methods:                                   │
│  onCreating($model): void                  │
│    Auto-generates 'key' from 'name'        │
│    using Str::slug($name, '_')             │
│                                            │
│  onSave($model): void                      │
│    Calls $model->clearCache()              │
└────────────────────────────────────────────┘
```

---

## Flow Diagrams

### Menu Link Resolution Flow

```
User clicks menu item
        │
        ▼
┌───────────────────────────────────────┐
│  MenuItem::getLink()                  │
└───────────────────────────────────────┘
        │
        ▼
┌───────────────────────────────────────┐
│  Is type === EXTERNAL?                │
│  AND url is not empty?                │
└───────────────────────────────────────┘
        │
        ├─── YES ──▶ Return $this->url
        │
        ▼ NO
┌───────────────────────────────────────┐
│  Is entity_type === 'document'?       │
│  (isAsset() check)                    │
└───────────────────────────────────────┘
        │
        ├─── YES ──▶ getEntity() ──▶ Return asset($entity->src)
        │
        ▼ NO
┌───────────────────────────────────────┐
│  Does item have entity?               │
│  (hasEntity() check)                  │
└───────────────────────────────────────┘
        │
        ├─── NO ──▶ Return '#'
        │
        ▼ YES
┌───────────────────────────────────────┐
│  Get entity route name                │
│  (getEntityRoute())                   │
└───────────────────────────────────────┘
        │
        ├─── NULL ──▶ Return '#'
        │
        ▼ NOT NULL
┌───────────────────────────────────────┐
│  MenuItemService::routeLink()         │
│  1. getEntity()                       │
│  2. route($routeName, $entity->slug)  │
└───────────────────────────────────────┘
        │
        ├─── Success ──▶ Return route URL
        │
        ├─── RouteNotFoundException ──▶ Log error, Return '#'
        │
        ▼
    Return URL
```

### Entity Configuration Resolution Flow

```
MenuItem stored with:
  entity_type = "Post"
  entity_id = 123
        │
        ▼
┌───────────────────────────────────────┐
│  getEntityType()                      │
│  Converts to config key               │
│  "Post" → "post"                      │
│  (strtolower + slug)                  │
└───────────────────────────────────────┘
        │
        ▼
┌───────────────────────────────────────┐
│  getEntityInstance()                  │
│  config('menu.entities.post')         │
│  Returns: Post::class                 │
└───────────────────────────────────────┘
        │
        ▼
┌───────────────────────────────────────┐
│  getEntity()                          │
│  Post::find(123) with caching         │
│  Cache key: "post_123"                │
└───────────────────────────────────────┘
        │
        ▼
┌───────────────────────────────────────┐
│  getEntityRoute()                     │
│  config('menu.routes.post')           │
│  Returns: 'frontend.post.show'        │
└───────────────────────────────────────┘
        │
        ▼
┌───────────────────────────────────────┐
│  MenuItemService::routeLink()         │
│  route('frontend.post.show',          │
│        $entity->slug)                 │
└───────────────────────────────────────┘
        │
        ▼
   Generated URL
   (e.g., /posts/my-blog-post)
```

### Cache Management Flow

```
Menu saved (create/update)
        │
        ▼
┌───────────────────────────────────────┐
│  MenuEventHandler::onSave()           │
│  Triggered by Laravel events          │
└───────────────────────────────────────┘
        │
        ▼
┌───────────────────────────────────────┐
│  $menu->clearCache()                  │
└───────────────────────────────────────┘
        │
        ├──▶ Cache::forget($menu->key)
        │
        ├──▶ Load $menu->allItems
        │
        └──▶ foreach ($menu->allItems as $item)
                  │
                  ▼
            $item->clearCache()
                  │
                  ▼
            Cache::forget("{entity_type}_{entity_id}")


View rendering with MenuComposer:
        │
        ▼
┌───────────────────────────────────────┐
│  MenuComposer::compose($view)         │
│  Extracts 'key' from view data        │
└───────────────────────────────────────┘
        │
        ▼
┌───────────────────────────────────────┐
│  Cache::remember($key, ...)           │
│  Menu::where('key', $key)             │
│      ->where('status', ACTIVE)        │
│      ->with('items')                  │
│      ->first()                        │
└───────────────────────────────────────┘
        │
        ├─── Cache HIT ──▶ Return cached menu
        │
        └─── Cache MISS ──▶ Query DB, cache result
```

### Menu Item Deletion Flow

```
DELETE /api/menu-items/{menuItem}
        │
        ▼
┌───────────────────────────────────────┐
│  MenuItemController::destroy()        │
└───────────────────────────────────────┘
        │
        ▼
┌───────────────────────────────────────┐
│  Delete children first                │
│  DB::table('menu_items')              │
│    ->where('parent_id', $menuItem->id)│
│    ->where('menu_id', $menuItem->     │
│             menu_id)                  │
│    ->delete()                         │
└───────────────────────────────────────┘
        │
        ▼
┌───────────────────────────────────────┐
│  Delete the menu item                 │
│  $menuItem->delete()                  │
└───────────────────────────────────────┘
        │
        ▼
┌───────────────────────────────────────┐
│  Return 204 No Content                │
└───────────────────────────────────────┘

Note: Menu deletion cascades to items
      via foreign key constraint
```

---

## Trait and Interface Reference

### Traits from bongo/framework

| Trait         | Provider           | Adds Methods                                    | Purpose                              |
|---------------|--------------------|-------------------------------------------------|--------------------------------------|
| HasFactory    | Laravel            | `factory()`, `newFactory()`                     | Model factories for testing          |
| HasStatus     | bongo/framework    | Status constants, scopes                        | Status management (pending/active)   |
| HasUUID       | bongo/framework    | UUID generation on create                       | Public unique identifiers            |
| SoftDeletes   | Laravel            | `delete()`, `restore()`, `forceDelete()`        | Soft delete functionality            |
| HasRecursive  | bongo/framework    | `parent()`, `children()`, `ancestors()`,        | Hierarchical relationships           |
|               |                    | `descendants()`, `nestedChildren()`             |                                      |

### Trait Details

#### HasStatus (on Menu model)

**Constants Added**:
- `PENDING = 'pending'`
- `ACTIVE = 'active'`
- `INACTIVE = 'inactive'`

**Scopes Added**:
- `scopeActive($query)` - Filter by active status
- `scopePending($query)` - Filter by pending status
- `scopeInactive($query)` - Filter by inactive status

**Methods Added**:
- `isActive(): bool`
- `isPending(): bool`
- `isInactive(): bool`

#### HasUUID (on both models)

**Behavior**:
- Automatically generates UUID on model creation
- Stores in `uuid` column
- UUID is indexed for fast lookups

#### HasRecursive (on MenuItem model)

**Relations Added**:
- `parent(): BelongsTo` - Parent menu item
- `children(): HasMany` - Direct children ordered by sort_order
- `ancestors(): HasMany` - All ancestors up the tree
- `descendants(): HasMany` - All descendants down the tree
- `nestedChildren(): HasMany` - Recursive load with 'children' relation

**Example Usage**:
```php
// Load menu with all nested children
$menu->load('items.nestedChildren');

// Get parent item
$parent = $menuItem->parent;

// Get all children
$children = $menuItem->children;

// Check if has children
if ($menuItem->children->count() > 0) { ... }
```

---

## Extension Points

### 1. Adding New Entity Types

**Step 1: Update Configuration**

Edit `src/Config/menu.php`:
```php
return [
    'entities' => [
        // Existing entities...
        'article' => \App\Models\Article::class,
    ],
    'routes' => [
        // Existing routes...
        'article' => 'frontend.article.show',
    ],
];
```

**Step 2: Update MenuEntityType (if package-based)**

Edit `src/Services/MenuEntityType.php`:
```php
public static function all(): array
{
    $categories = ['Page'];

    // Add your package check
    if (package()->isEnabled('article')) {
        array_push($categories, 'Article');
    }

    return collect($categories)->sort()->values()->all();
}
```

**Requirements for Entity**:
- Must have a `slug` attribute (used for routing)
- Must be accessible via Eloquent: `Article::find($id)`

### 2. Custom View Composers

Register additional view composers in `MenuServiceProvider`:

```php
protected array $composers = [
    MenuComposer::class => [
        'menu::frontend.partials.menu',
        // Add more views...
    ],
    CustomMenuComposer::class => [
        'custom::views.menu',
    ],
];
```

### 3. Event Listeners

Add custom event listeners in `MenuServiceProvider`:

```php
protected array $listeners = [
    'menu.created' => [
        CustomMenuCreatedListener::class,
    ],
];
```

### 4. Middleware

Add custom middleware in `MenuServiceProvider`:

```php
protected array $middlewares = [
    'custom' => CustomMiddleware::class,
];
```

### 5. Custom Routes

Add custom routes in `src/Routes/`:

Create `src/Routes/custom.php`:
```php
<?php

use Illuminate\Support\Facades\Route;

Route::get('/custom-menu-endpoint', [CustomController::class, 'index']);
```

This file is auto-registered by `AbstractServiceProvider`.

### 6. Additional Migrations

Add migrations in `src/Migrations/` with appropriate timestamps:

```php
<?php

declare(strict_types=1);

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class AddGroupToMenuItemsTable extends Migration
{
    public function up()
    {
        if (Schema::hasTable('menu_items') &&
            !Schema::hasColumn('menu_items', 'group')) {
            Schema::table('menu_items', function (Blueprint $table) {
                $table->string('group')->nullable()->after('class');
            });
        }
    }

    public function down()
    {
        if (Schema::hasColumn('menu_items', 'group')) {
            Schema::table('menu_items', function (Blueprint $table) {
                $table->dropColumn('group');
            });
        }
    }
}
```

### 7. Custom Form Requests

Create custom validation in `src/Http/Requests/`:

```php
<?php

declare(strict_types=1);

namespace Bongo\Menu\Http\Requests\Backend;

use Illuminate\Foundation\Http\FormRequest;

class CustomMenuRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'name' => 'required|string|max:255',
            'custom_field' => 'nullable|string',
        ];
    }
}
```

---

## Adding New Features

### Feature: Adding Icon Support to Menu Items

**1. Database Migration**

Create `src/Migrations/2024_01_01_000003_add_icon_to_menu_items_table.php`:

```php
<?php

declare(strict_types=1);

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class AddIconToMenuItemsTable extends Migration
{
    public function up()
    {
        if (Schema::hasTable('menu_items') &&
            !Schema::hasColumn('menu_items', 'icon')) {
            Schema::table('menu_items', function (Blueprint $table) {
                $table->string('icon')->nullable()->after('class');
            });
        }
    }

    public function down()
    {
        if (Schema::hasColumn('menu_items', 'icon')) {
            Schema::table('menu_items', function (Blueprint $table) {
                $table->dropColumn('icon');
            });
        }
    }
}
```

**2. Update Model**

Edit `src/Models/MenuItem.php`:

```php
protected $fillable = [
    'menu_id',
    'parent_id',
    'name',
    'url',
    'entity_type',
    'entity_id',
    'class',
    'icon',  // Add this
    'type',
    'target',
    'sort_order',
];
```

**3. Update Form Requests**

Edit `src/Http/Requests/Backend/StoreMenuRequest.php` and `UpdateMenuRequest.php`:

```php
public function rules(): array
{
    return [
        'name' => 'required|string|max:255',
        'icon' => 'nullable|string|max:255',  // Add this
        // ... other rules
    ];
}
```

**4. Update Views**

Edit `src/Views/backend/partials/menu_item.blade.php`:

```blade
<div class="form-group">
    <label for="icon">Icon</label>
    <input type="text" name="icon" class="form-control"
           value="{{ old('icon', $menuItem->icon ?? '') }}">
</div>
```

Edit `src/Views/frontend/partials/menu.blade.php`:

```blade
<a href="{{ $item->getLink() }}">
    @if($item->icon)
        <i class="{{ $item->icon }}"></i>
    @endif
    {{ $item->name }}
</a>
```

**5. Update API Resources**

Edit `src/Http/Resources/MenuItemResource.php`:

```php
public function toArray($request): array
{
    return [
        'type' => 'menu-items',
        'id' => $this->id,
        'attributes' => [
            'name' => $this->name,
            'icon' => $this->icon,  // Add this
            // ... other attributes
        ],
    ];
}
```

**6. Update Tests**

Create `tests/Unit/MenuItemIconTest.php`:

```php
<?php

declare(strict_types=1);

namespace Bongo\Menu\Tests\Unit;

use Bongo\Menu\Models\Menu;
use Bongo\Menu\Models\MenuItem;
use Bongo\Menu\Tests\TestCase;

class MenuItemIconTest extends TestCase
{
    /** @test */
    public function it_can_store_icon_on_menu_item(): void
    {
        $menu = Menu::factory()->create();
        $item = MenuItem::factory()->create([
            'menu_id' => $menu->id,
            'icon' => 'fa fa-home',
        ]);

        $this->assertEquals('fa fa-home', $item->icon);
    }

    /** @test */
    public function icon_is_optional(): void
    {
        $menu = Menu::factory()->create();
        $item = MenuItem::factory()->create([
            'menu_id' => $menu->id,
            'icon' => null,
        ]);

        $this->assertNull($item->icon);
    }
}
```

**7. Update Factory**

Edit `database/factories/MenuItemFactory.php`:

```php
public function definition(): array
{
    return [
        'menu_id' => Menu::factory(),
        'name' => $this->faker->words(3, true),
        'icon' => $this->faker->randomElement(['fa fa-home', 'fa fa-user', null]),
        // ... other fields
    ];
}
```

**8. Run and Test**

```bash
# Run migration
php artisan migrate

# Run tests
composer test

# Format code
composer format

# Static analysis
composer analyse
```

### Feature: Adding Menu Item Visibility Rules

Follow similar pattern:
1. Add `visibility_rule` column via migration
2. Update model fillable
3. Add validation rules
4. Create `MenuItemVisibility` service class
5. Add `isVisible()` method to MenuItem model
6. Update views to check visibility
7. Write tests

This pattern can be extended for any new feature following the package's architecture.

---

## Performance Considerations

### Caching Strategy

1. **Menu-level caching**: Entire menu cached by `key`
2. **Entity-level caching**: Each linked entity cached separately
3. **Cache invalidation**: Automatic on menu save via events
4. **Cache duration**: Configured via `settings.cache_default`

### Optimization Tips

1. **Eager loading**: Always load with relationships:
   ```php
   $menu->load('items.nestedChildren');
   ```

2. **Avoid N+1 queries**: Use `with()` in queries:
   ```php
   Menu::with('items')->get();
   ```

3. **Index usage**: Ensure indexes on:
   - `menus.key` (for fast lookup)
   - `menu_items.menu_id` (for relationship queries)
   - `menu_items.parent_id` (for hierarchical queries)
   - `menu_items.sort_order` (for ordering)

4. **Soft deletes**: Only on menus (items cascade delete)

5. **Cache warming**: Pre-cache frequently accessed menus:
   ```php
   Artisan::command('menu:warm-cache', function () {
       $menus = Menu::where('status', Menu::ACTIVE)->get();
       foreach ($menus as $menu) {
           Cache::remember($menu->key, config('settings.cache_default'),
               fn() => $menu->load('items')
           );
       }
   });
   ```

---

## Security Considerations

1. **Authorization**: Controllers should check permissions
2. **Validation**: All inputs validated via Form Requests
3. **SQL Injection**: Protected by Eloquent ORM
4. **XSS**: Blade auto-escapes output (use `{!! !!}` carefully)
5. **CSRF**: Protected by Laravel middleware (backend routes)
6. **API Authentication**: Uses Sanctum (`auth:sanctum` middleware)

---

## Testing Strategy

### Test Coverage

- **Unit Tests**: Model methods, service classes, helpers
- **Feature Tests**: Controller actions, API endpoints
- **Integration Tests**: Database interactions, relationships

### Testing Best Practices

1. Use factories for model creation
2. Test relationships thoroughly
3. Test cache clearing behavior
4. Test event firing
5. Test validation rules
6. Test API responses (JSON structure)
7. Test edge cases (null values, empty collections)

---

This architecture documentation provides a comprehensive overview of the Bongo Menu package structure, patterns, and extension points. Refer to `.cursorrules` and `CLAUDE.md` for development guidelines and quick reference.
