# Bongo Menu Package

A Laravel package for creating and managing hierarchical frontend menus from within an admin interface. Build dynamic menu structures with items that can link to internal entities (pages, posts, projects, categories) or external URLs.

## Features

- **Hierarchical Menu Structure**: Create unlimited nested menu levels with parent/child relationships
- **Flexible Link Types**: Link to internal entities (pages, posts, projects) or external URLs
- **Entity Polymorphism**: Link menu items to any configured entity type with automatic route resolution
- **Multiple Menus**: Create and manage multiple menus (main menu, footer menu, etc.)
- **Status Management**: Control menu visibility with status states (pending, active, inactive)
- **Caching**: Built-in caching for improved performance
- **API & Backend**: Complete CRUD operations via both backend UI and REST API
- **DataTables Integration**: Backend listing with search and pagination
- **Auto Route Generation**: Automatic URL generation based on entity configuration

## Requirements

- PHP 8.2 or higher
- Laravel 10 or higher
- [bongo/framework](https://designtecpackages.co.uk) ^3.0

## Installation

### 1. Install via Composer

```bash
composer require bongo/menu
```

### 2. Run Migrations

The package will automatically register its service provider. Run the migrations to create the required tables:

```bash
php artisan migrate
```

This creates two tables:
- `menus` - Stores menu definitions
- `menu_items` - Stores individual menu items with hierarchical structure

### 3. Publish Configuration (Optional)

To customize entity configuration:

```bash
php artisan vendor:publish --provider="Bongo\Menu\MenuServiceProvider" --tag="config"
```

This publishes `config/menu.php` where you can configure which entities can be linked in menus.

### 4. Publish Views (Optional)

To customize menu views:

```bash
php artisan vendor:publish --provider="Bongo\Menu\MenuServiceProvider" --tag="views"
```

## Configuration

### Adding Linkable Entity Types

Edit `config/menu.php` to add new entity types that can be linked in menu items:

```php
return [
    'entities' => [
        'page' => \Bongo\Page\Models\Page::class,
        'post' => \Bongo\Post\Models\Post::class,
        'article' => \App\Models\Article::class,  // Add your custom entity
        // ...
    ],
    'routes' => [
        'page' => 'frontend.page.show',
        'post' => 'frontend.post.show',
        'article' => 'frontend.article.show',  // Add corresponding route
        // ...
    ],
];
```

**Requirements for linkable entities:**
- Must have a `slug` attribute (used for route parameters)
- Must have a corresponding named route defined in `routes` array

## Usage

### Backend Administration

Access the menu management interface at:

```
/menu
```

**Create a Menu:**
1. Navigate to `/menu/create`
2. Enter menu name (e.g., "Main Menu")
3. Key is auto-generated as slug (e.g., "main_menu")
4. Set status to "active"

**Add Menu Items:**
1. Edit a menu
2. Add items with type:
   - **Internal**: Link to an entity (page, post, etc.)
   - **External**: Link to external URL

### Frontend Display

Include menu partials in your Blade templates:

```blade
{{-- Main menu --}}
@include('menu::frontend.partials.main_menu')

{{-- Top menu --}}
@include('menu::frontend.partials.top_menu')

{{-- Footer menu --}}
@include('menu::frontend.partials.footer_menu')
```

### Programmatic Access

**Fetch a menu by key:**

```php
use Bongo\Menu\Models\Menu;

$menu = Menu::where('key', 'main_menu')->first();
$rootItems = $menu->items; // Only root-level items
$allItems = $menu->allItems; // All items including nested
```

**Work with menu items:**

```php
foreach ($menu->items as $item) {
    echo $item->name;
    echo $item->getLink();  // Automatically resolves URL

    // Check if item has children
    if ($item->children->count() > 0) {
        foreach ($item->children as $child) {
            echo $child->name;
        }
    }

    // Check if current page
    if ($item->isActive()) {
        // Add active class
    }
}
```

**Using the facade:**

```php
use MenuEntityType;

// Get available entity types based on enabled packages
$types = MenuEntityType::all();
// Returns: ['Document', 'Page', 'Post', 'Post Category', 'Project', 'Project Category']
```

### API Usage

**Get menu with items:**

```http
GET /api/menus/{id}
```

**Menu item CRUD:**

```http
GET    /api/menu-items          # List items
POST   /api/menu-items/store    # Create item
POST   /api/menu-items/update   # Update item
POST   /api/menu-items/delete   # Delete item
```

## Development

### Setup Development Environment

1. Clone the repository
2. Install dependencies:

```bash
composer install
```

3. Set up the workbench environment:

```bash
composer build
```

4. Start the development server:

```bash
composer start
```

### Running Tests

```bash
# Run all tests
composer test

# Run specific test
vendor/bin/phpunit --filter test_method_name

# Run with coverage (requires Xdebug)
composer test:coverage
```

### Code Quality

```bash
# Run static analysis
composer analyse

# Format code
composer format
```

## Architecture Overview

### Database Structure

- **menus**: Stores menu definitions with name, key (slug), and status
- **menu_items**: Hierarchical structure with `parent_id` for nesting
  - Links to entities via `entity_type` and `entity_id` (polymorphic)
  - Supports external URLs via `url` field
  - `sort_order` determines display order

### Key Classes

- `Menu` - Menu model with status management and caching
- `MenuItem` - Menu item model with hierarchical relationships
- `MenuServiceProvider` - Registers routes, views, composers, and events
- `MenuEventHandler` - Auto-generates menu keys and clears cache on save
- `MenuItemService` - Handles URL generation with error handling
- `MenuComposer` - View composer for menu partials

### Caching

- Menu and entity data are cached for performance
- Cache automatically clears when menu is saved
- Manual cache clearing: `$menu->clearCache()`

## Troubleshooting

**Menu items not showing:**
- Check menu status is "active"
- Verify `sort_order` is set on menu items
- Clear cache: `php artisan cache:clear`

**Links returning '#':**
- Ensure entity has a `slug` attribute
- Verify route name is configured in `config/menu.php`
- Check entity exists and is published

**Route not found errors:**
- Ensure named route exists in your application
- Check route name matches configuration
- Verify entity slug is valid

## Stability Notice

This package is actively being developed and we would like to get feedback to improve it. Please feel free to submit feedback via issues or pull requests.

## License

MIT License

## Credits

Developed by [Bespoke UK](https://bespokeuk.com)
