# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Package Overview

**bongo/menu** is a Laravel package for creating and managing hierarchical frontend menus from an admin interface. Menu items can link to internal entities (pages, posts, projects, categories) or external URLs.

**Requirements**: PHP 8.2+, Laravel 10+

**Core Dependency**: Built on `bongo/framework` (^3.0) which provides AbstractServiceProvider, AbstractModel, AbstractController, AbstractEventHandler and traits like HasStatus, HasUUID, HasRecursive.

## Documentation

- **[.cursorrules](.cursorrules)** - Comprehensive development guide with architecture, patterns, and common tasks
- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Detailed architecture with class diagrams, flow charts, and extension points
- **[README.md](README.md)** - Installation, configuration, and usage guide
- **[CHANGELOG.md](CHANGELOG.md)** - Version history and changes

## Development Commands

```bash
# Run tests
composer test                    # All tests
composer test:coverage          # With coverage (requires Xdebug)
composer test:parallel          # Parallel execution
vendor/bin/phpunit --filter test_method_name  # Run specific test

# Code quality
composer analyse                # PHPStan static analysis (level 1)
composer format                 # Laravel Pint formatting

# Package development (Orchestra Testbench)
composer build                  # Build workbench
composer start                  # Start development server
composer clear                  # Clear workbench
composer prepare                # Discover packages
```

## Quick Architecture Reference

### Service Provider Pattern

**MenuServiceProvider** extends `AbstractServiceProvider` from bongo/framework which automatically registers:
- **Config**: From `src/Config/menu.php`
- **Routes**: From `src/Routes/` (api.php → `/api/*`, backend.php → `/admin/menu/*`)
- **Views**: From `src/Views/menu/` with namespace `menu::`
- **Migrations**: From `src/Migrations/`
- **Translations**: From `src/Translations/`

**Key configuration**:
```php
protected string $module = 'menu';  // Defines view namespace

protected array $composers = [
    MenuComposer::class => [
        'menu::frontend.partials.menu',
        'menu::frontend.partials.top_menu',
        'menu::frontend.partials.main_menu',
        'menu::frontend.partials.footer_menu',
        'menu::frontend.partials.footer_menu_2',
        'menu::frontend.partials.bottom_menu',
    ],
];

protected array $subscribers = [
    MenuEventHandler::class,
];
```

**MenuEntityType facade**: Aliased in boot() method, provides `MenuEntityType::all()` which returns available entity types based on enabled packages.

### Database Structure

Two main tables:

**menus** table:
- `id`, `uuid` (indexed)
- `name`, `key` (auto-generated slug from name, indexed)
- `status` enum: `PENDING`, `ACTIVE`, `INACTIVE` (from HasStatus trait)
- Audit fields: `created_by`, `updated_by`, `deleted_by`
- Soft deletes enabled

**menu_items** table:
- `id`, `uuid` (indexed)
- `menu_id` (foreign key to menus, cascade on delete)
- `parent_id` (nullable, for hierarchical structure)
- `name`, `url` (for external links)
- `entity_type`, `entity_id` (polymorphic relationship to linkable entities)
- `class` (CSS class)
- `type` enum: `INTERNAL`, `EXTERNAL`
- `target` enum: `SELF`, `BLANK`
- `sort_order` (indexed, determines display order)


### Model Relationships and Hierarchy

**Menu Model** (`Bongo\Menu\Models\Menu`):
- Traits: `HasStatus`, `HasUUID`, `SoftDeletes`
- Status constants: `PENDING`, `ACTIVE`, `INACTIVE` (used with enum column)
- `items()` - HasMany to MenuItem, returns only root-level items (where `parent_id` is null), ordered by `sort_order`
- `allItems()` - HasMany to all MenuItems including nested children
- `hasItems()` - Boolean check if menu has any items
- `clearCache()` - Clears menu cache AND recursively clears all nested item caches

**MenuItem Model** (`Bongo\Menu\Models\MenuItem`):
- Traits: `HasUUID`, `HasRecursive`
- `HasRecursive` trait provides: `parent()`, `children()`, `ancestors()`, `descendants()` methods
- `parent_id` is nullable; null = root-level item
- Access nested structure via `$item->children` (recursive relationship)
- Type constants: `EXTERNAL`, `INTERNAL` (determines if URL is external or internal entity)
- Target constants: `SELF`, `BLANK` (for link target attribute)

**Key MenuItem methods**:
- `getLink()` - URL resolution chain:
  1. External link: returns `url` field
  2. Document asset: returns `asset($entity->src)`
  3. Internal entity: delegates to `MenuItemService::routeLink()` → generates route with entity slug
  4. Fallback: returns `'#'`
- `getEntity()` - Loads linked entity with caching (key: `{entity_type}_{entity_id}`)
- `getEntityType()` - Converts `entity_type` to lowercase slug (e.g., "Post Category" → "post_category")
- `getEntityInstance()` - Returns model class from config: `config('menu.entities.{entity_type}')`
- `getEntityRoute()` - Returns route name from config: `config('menu.routes.{entity_type}')`
- `isActive()` - Checks if current request URL matches entity slug
- `isInternal()`, `isExternal()`, `isAsset()` - Type checking helpers
- `clearCache()` - Clears entity cache only

### Entity Configuration System

`src/Config/menu.php` maps entity types to models and routes:

```php
'entities' => [
    'page' => Page::class,
    'post' => Post::class,
    'document' => Document::class,  // Special: treated as asset
    // ...
],
'routes' => [
    'page' => 'frontend.page.show',
    'post' => 'frontend.post.show',
    // Note: 'document' has no route (uses asset() instead)
]
```

**Link resolution flow**:
1. MenuItem stores `entity_type` (e.g., "Post") and `entity_id`
2. `getEntityType()` converts to config key: "Post" → "post"
3. `getEntityInstance()` looks up model: `config('menu.entities.post')` → `Post::class`
4. `getEntity()` queries and caches: `Post::find($entity_id)`
5. `getEntityRoute()` gets route name: `config('menu.routes.post')` → `'frontend.post.show'`
6. `MenuItemService::routeLink()` generates URL: `route('frontend.post.show', $entity->slug)`

**Special case - Documents**:
- `isAsset()` returns true when `entity_type === 'document'`
- `getLink()` returns `asset($entity->src)` instead of routing

**To add a new linkable entity type**:
1. Add to `entities` array: `'article' => Article::class`
2. Add to `routes` array: `'article' => 'frontend.article.show'`
3. Ensure entity model has `slug` attribute (used for routing)
4. No code changes needed - system auto-resolves

### Controllers and Routes

**Backend** (`src/Routes/backend.php` + `src/Http/Controllers/Backend/`):
- Route prefix: `/menu`, names: `menu.*`
- `MenuController` - Full CRUD for menus, nested menu item management
- `MenuDatatableController` - DataTables integration for listing
- Request validation: `Backend\StoreMenuRequest`, `Backend\UpdateMenuRequest`

**API** (`src/Routes/api.php` + `src/Http/Controllers/Api/`):
- Route prefixes: `/menus` and `/menu-items`
- Route names: `menu.*` and `menu_item.*`
- `MenuController::show()` - GET `/menus/{id}` - Returns menu with items
- `MenuItemController` - CRUD for menu items (index, store, update, destroy)
- Request validation: Separate `Api\` namespace request classes

**Important**: Request classes are in separate directories (`Backend/` vs `Api/`) based on context.

### Services

**MenuItemService** (`Bongo\Menu\Services\MenuItemService`):
- `routeLink($routeName, $menuItem)` - Generates route URL with error handling
  - Fetches entity, uses `slug` for routing
  - Catches `RouteNotFoundException`, logs and returns `'#'`
- `groupChildrenAsArray($menuItem)` - Groups children by group attribute

**MenuEntityType** (facade):
- `all()` - Returns sorted array of available entity types based on enabled packages
- Dynamically checks: `package()->isEnabled('post')`, `package()->isEnabled('project')`, etc.
- Always includes 'Page', conditionally adds 'Document', 'Post', 'Post Category', 'Project', 'Project Category'

### Event Handling

`MenuEventHandler` extends `AbstractEventHandler`:
- `onCreating()` - Generates menu `key` from `name` using `Str::slug($name, '_')`
- `onSave()` - Calls `$menu->clearCache()` after every save

### View Composers

`MenuComposer` is registered for all menu view partials:
- `menu::frontend.partials.menu`
- `menu::frontend.partials.top_menu`
- `menu::frontend.partials.main_menu`
- `menu::frontend.partials.footer_menu`
- `menu::frontend.partials.footer_menu_2`
- `menu::frontend.partials.bottom_menu`

## Testing

Test files must:
- Extend `Bongo\Menu\Tests\TestCase` (not PHPUnit's TestCase)
- Include `declare(strict_types=1);` at the top
- Use `@test` annotation or `test_` prefix for test methods

TestCase configures Orchestra Testbench with:
- MenuServiceProvider loaded
- In-memory SQLite database
- Testing environment

## Code Style

**Laravel Pint** (`pint.json`):
- Preset: Laravel
- `declare_strict_types` required on all PHP files
- `new_with_parentheses` disabled
- `nullable_type_declaration_for_default_null_value` disabled

**PHPStan** (`phpstan.neon.dist`):
- Level: 1 (currently low)
- Ignores undefined property/method errors in Models, Resources, Feature tests (for Eloquent magic methods)
- Checks `src/` and `config/`

## Cache Management

**Caching strategy**:
- Menu cache key: `$menu->key` (e.g., "main_menu")
- MenuItem entity cache key: `{entity_type}_{entity_id}` (e.g., "post_123")
- Cache duration: `config('settings.cache_default')`

**Cache clearing**:
- Menu saves automatically trigger `clearCache()` via `MenuEventHandler::onSave()`
- `Menu::clearCache()` clears menu cache AND recursively clears all item entity caches
- `MenuItem::clearCache()` clears only that item's entity cache
- Important: Updating a menu item does NOT auto-clear cache - must be done manually

**Cache usage**:
- `MenuItem::getEntity()` caches the loaded entity model
- `MenuComposer` caches menu data for views

## Key Files Reference

| File Path | Purpose | Key Classes/Functions |
|-----------|---------|----------------------|
| `src/MenuServiceProvider.php` | Service provider | MenuServiceProvider extends AbstractServiceProvider |
| `src/Models/Menu.php` | Menu model | Menu (HasStatus, HasUUID, SoftDeletes) |
| `src/Models/MenuItem.php` | Menu item model | MenuItem (HasRecursive, HasUUID) |
| `src/Config/menu.php` | Entity configuration | entities, routes arrays |
| `src/Services/MenuItemService.php` | URL generation | routeLink(), groupChildrenAsArray() |
| `src/Services/MenuEntityType.php` | Entity type discovery | all() |
| `src/Events/MenuEventHandler.php` | Event handling | onCreating(), onSave() |
| `src/Http/ViewComposers/MenuComposer.php` | View composer | compose() |
| `src/Http/Controllers/Backend/MenuController.php` | Backend CRUD | index(), create(), store(), show(), edit(), update(), destroy() |
| `src/Http/Controllers/Api/MenuController.php` | API menu endpoint | index(), store(), show(), update(), destroy() |
| `src/Http/Controllers/Api/MenuItemController.php` | API item endpoint | index(), store(), show(), update(), destroy() |
| `src/Routes/backend.php` | Backend routes | /admin/menu/* routes |
| `src/Routes/api.php` | API routes | /api/menus/*, /api/menu-items/* routes |
| `src/Migrations/2019_01_01_000001_create_menus_table.php` | Menu table | menus table schema |
| `src/Migrations/2019_01_01_000002_create_menu_items_table.php` | Menu items table | menu_items table schema |
| `src/Views/backend/*.blade.php` | Admin views | CRUD interface |
| `src/Views/frontend/partials/*.blade.php` | Frontend templates | Menu rendering |
| `tests/Unit/MenuTest.php` | Menu tests | Menu model tests |
| `tests/Unit/MenuItemTest.php` | MenuItem tests | MenuItem model tests |
| `tests/Unit/MenuItemServiceTest.php` | Service tests | Service method tests |
| `tests/TestCase.php` | Test base class | Orchestra Testbench setup |

## Code Style Summary

**Laravel Pint** (`pint.json`):
- Preset: Laravel
- `declare_strict_types` required on all PHP files
- `new_with_parentheses` disabled
- `nullable_type_declaration_for_default_null_value` disabled

**PHPStan** (`phpstan.neon.dist`):
- Level: 1 (currently low)
- Ignores undefined property/method errors in Models, Resources, Feature tests (for Eloquent magic methods)
- Checks `src/` and `config/`

**Conventions**:
- All PHP files must start with `declare(strict_types=1);`
- Use strict type declarations for all parameters and return types
- Models extend `Bongo\Framework\Models\AbstractModel`
- Controllers extend `AbstractController` (backend) or `AbstractApiController` (api)
- Event handlers extend `AbstractEventHandler`
- Tests extend `Bongo\Menu\Tests\TestCase` (not PHPUnit's TestCase)
- Use `@test` annotation or `test_` prefix for test methods

## Common Development Tasks

### Adding a New Linkable Entity Type

1. **Update config/menu.php**:
```php
'entities' => [
    'article' => \App\Models\Article::class,
],
'routes' => [
    'article' => 'frontend.article.show',
],
```

2. **Update MenuEntityType::all()** (if entity is in a package):
```php
if (package()->isEnabled('article')) {
    array_push($categories, 'Article');
}
```

3. **Ensure entity model has**:
- `slug` attribute (used for routing)
- Named route defined in application

### Displaying Menus in Views

```blade
{{-- Include predefined partial with key --}}
@include('menu::frontend.partials.main_menu')

{{-- Or pass custom key --}}
@include('menu::frontend.partials.menu', ['key' => 'custom_menu'])
```

### Clearing Cache Manually

```php
// Clear specific menu
$menu = Menu::where('key', 'main_menu')->first();
$menu->clearCache();  // Clears menu + all nested item entity caches

// Clear specific menu item entity
$menuItem->clearCache();  // Clears only entity cache
```

## Troubleshooting

**Menu items not showing**:
- Check menu status is `Menu::ACTIVE`
- Verify `sort_order` is set
- Clear cache: `php artisan cache:clear`

**Links returning '#'**:
- Ensure entity has `slug` attribute
- Verify route name in `config/menu.php`
- Check entity exists and is published
- Review logs for `RouteNotFoundException`

**RouteNotFoundException**:
- Ensure named route exists in application
- Check route name matches config
- Verify entity slug is valid

**Cache not clearing**:
- Menu saves auto-clear via `MenuEventHandler::onSave()`
- Menu item updates do NOT auto-clear - must call `$menu->clearCache()` manually
- Check `settings.cache_default` config value

## Dependencies

- **bongo/framework** ^3.0 - Core framework (AbstractServiceProvider, AbstractModel, AbstractController, traits)
- **Laravel** 10+ or 11+
- **PHP** 8.2+

## Additional Resources

- See `ARCHITECTURE.md` for detailed architecture diagrams and flow charts
- See `README.md` for installation and usage guide
- See `.cursorrules` for comprehensive development guide
- See `CHANGELOG.md` for version history
