<?php

declare(strict_types=1);

namespace Bongo\Menu\Http\Controllers\Api;

use Bongo\Framework\Http\Controllers\AbstractApiController;
use Bongo\Menu\Http\Requests\Api\StoreMenuItem;
use Bongo\Menu\Http\Requests\Api\UpdateMenuItem;
use Bongo\Menu\Http\Resources\MenuItemResource;
use Bongo\Menu\Models\MenuItem;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;

class MenuItemController extends AbstractApiController
{
    public function index(Request $request)
    {
        return response()->json([
            'data' => MenuItem::query()
                ->when($request->filled('item_id'), function ($menu) use ($request) {
                    $menu->whereNot('id', $request->get('item_id'));
                })
                ->when($request->filled('entity_type'), function ($menu) use ($request) {
                    $menu->where('entity_type', $request->get('entity_type'));
                })
                ->orderBy('name')
                ->get(),
        ]);
    }

    public function store(StoreMenuItem $request): JsonResponse
    {
        $validated = $request->validated('data.attributes');

        // Create the menuItem
        $menuItem = new MenuItem();
        $menuItem->fill($validated);
        $menuItem->save();

        return (new MenuItemResource($menuItem))
            ->response()
            ->header('Location', route('backend.menu.show', $menuItem->menu));
    }

    public function show(MenuItem $menuItem): MenuItemResource
    {
        return new MenuItemResource($menuItem);
    }

    public function update(UpdateMenuItem $request, MenuItem $menuItem): MenuItemResource
    {
        $validated = $request->validated('data.attributes');

        // Update the menuItem
        $menuItem->fill($validated);
        $menuItem->save();

        return new MenuItemResource($menuItem);
    }

    public function destroy(MenuItem $menuItem): Application|ResponseFactory|Response
    {
        // Delete the children if any
        DB::table('menu_items')
            ->where('parent_id', $menuItem->id)
            ->where('menu_id', $menuItem->menu_id)
            ->delete();

        // The delete the menuItem
        $menuItem->delete();

        return response(null, 204);
    }
}
