<?php

namespace Bongo\Menu\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Menu\Http\Requests\StoreMenuRequest;
use Bongo\Menu\Http\Requests\UpdateMenuRequest;
use Bongo\Menu\Models\Menu;
use Bongo\Menu\Models\MenuItem;
use Illuminate\Http\RedirectResponse;

class MenuController extends AbstractController
{
    protected Menu $menu;

    private MenuItem $menuItem;

    public function __construct(Menu $menu, MenuItem $menuItem)
    {
        $this->menu = $menu;
        $this->menuItem = $menuItem;
    }

    public function index()
    {
        return view('menu::backend.index');
    }

    public function create()
    {
        $menu = $this->menu;

        return view('menu::backend.create', compact('menu'));
    }

    public function store(StoreMenuRequest $request)
    {
        $menu = $this->menu->create($request->all());
        $this->handleMenuItems($menu);

        return redirect()
            ->route('backend.menu.show', $menu->id)
            ->success(trans('menu::backend.store_success'));
    }

    public function show(Menu $menu)
    {
        $menu->loadMissing('items.nestedChildren');

        return view('menu::backend.show', compact('menu'));
    }

    public function edit(Menu $menu)
    {
        $menu->loadMissing('items.nestedChildren');

        return view('menu::backend.edit', compact('menu'));
    }

    public function update(UpdateMenuRequest $request, Menu $menu): RedirectResponse
    {
        $menu->update($request->all());
        $this->handleMenuItems($menu);

        return redirect()
            ->route('backend.menu.show', $menu->id)
            ->success(trans('menu::backend.update_success'));
    }

    public function destroy(Menu $menu)
    {
        if ($menu->delete()) {
            $redirect = back();

            // If the menu was deleted from their detail view,
            // we'll redirect back to the menu list view.
            if (route('backend.menu.show', $menu) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.menu.index');
            }

            return $redirect->success(trans('menu::backend.delete_success'));
        }

        return back()->error(trans('menu::backend.delete_failed'));
    }

    private function handleMenuItems($menu)
    {
        $items = request('items');

        // If the user has not passed any items through then delete all
        if (! isset($items) || empty($items)) {
            $menu->allItems()->delete();

            return;
        }

        // Delete the removed items
        $menu->allItems()->whereNotIn('id', array_keys($items))->delete();

        // Create/Update the items
        foreach ($items as $item) {
            $item = $this->setItemDetails($menu, $item);

            if ($menuItem = $this->menuItem->find($item['id'])) {
                $menuItem->update($item);
            } else {
                $this->menuItem->create($item);
            }
        }
    }

    private function setItemDetails($menu, $item)
    {
        $item['menu_id'] = $menu->id;

        if ($item['type'] === MenuItem::EXTERNAL) {
            $item['entity_id'] = null;
            $item['entity_type'] = null;
        }

        return $item;
    }
}
