<?php

namespace Bongo\Menu\Models;

use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasRecursive;
use Bongo\Framework\Traits\HasUUID;
use Bongo\Menu\Services\MenuItemService;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class MenuItem extends AbstractModel
{
    public const EXTERNAL = 'external';
    public const INTERNAL = 'internal';
    public const SELF = '_self';
    public const BLANK = '_blank';

    use HasUUID;
    use HasRecursive;

    /** @var array */
    protected $fillable = [
        'menu_id',
        'parent_id',
        'name',
        'url',
        'entity_type',
        'entity_id',
        'class',
        'type',
        'target',
        'sort_order',
    ];

    public function menu(): BelongsTo
    {
        return $this->belongsTo(Menu::class, 'menu_id');
    }

    public function isInternal(): bool
    {
        return $this->type === self::INTERNAL;
    }
    public function isExternal(): bool
    {
        return $this->type === self::EXTERNAL;
    }

    public function isAsset(): bool
    {
        return $this->getEntityType() === 'document';
    }

    public function getEntity()
    {
        if ($entityInstance = $this->getEntityInstance()) {
            return Cache::remember($this->getCacheKey(), config('settings.cache_default'),
                function () use ($entityInstance) {
                    return $entityInstance::find($this->entity_id);
                }
            );
        }

        return null;
    }
    public function getEntityInstance()
    {
        return app(config('menu.entities.'.$this->getEntityType()));
    }
    public function getEntityType(): string
    {
        return strtolower(Str::slug($this->entity_type, '_'));
    }
    public function getEntityRoute()
    {
        return config('menu.routes.'.$this->getEntityType());
    }
    public function hasEntity(): bool
    {
        return ! empty($this->entity_type) && ! empty($this->entity_id);
    }

    public function getCacheKey(): string
    {
        return $this->getEntityType().'_'.$this->entity_id;
    }
    public function clearCache(): void
    {
        Cache::forget($this->getCacheKey());
    }

    public function getLink(): string
    {
        // Is this an external link?
        if ($this->isExternal() && ! empty($this->url)) {
            return $this->url;
        }

        // Is this an asset?
        if ($this->isAsset() && $this->hasEntity() && $entity = $this->getEntity()) {
            return asset($entity->src);
        }

        // Is it internal with a route?
        if ($this->hasEntity() && $routeName = $this->getEntityRoute()) {
            return resolve(MenuItemService::class)->routeLink($routeName, $this);
        }

        return '#';
    }
    public function isActive(): bool
    {
        if ($this->hasEntity() && $entity = $this->getEntity()) {
            $url = strtolower($entity->slug);

            return request()->is("{$url}*");
        }

        return false;
    }
}
