# WARP.md

This file provides guidance to WARP (warp.dev) when working with code in this repository.

## Package Overview

This is the **bongo/menu** Laravel package - a CMS module that provides functionality to create and manage frontend menus from within an admin interface. It allows creating hierarchical menu structures with items that can link to internal entities (pages, posts, projects) or external URLs.

**Requirements**: PHP 8.2+, Laravel 10+

**Dependencies**: Built on top of `bongo/framework` (version ^3.0) which provides the base architecture including AbstractServiceProvider, AbstractModel, AbstractController, and AbstractEventHandler.

## Development Commands

### Testing
```bash
# Run tests without coverage
composer test
# or
vendor/bin/phpunit --no-coverage

# Run tests with coverage (requires Xdebug)
composer test:coverage

# Run parallel tests
composer test:parallel
```

### Code Quality
```bash
# Run static analysis (PHPStan level 1)
composer analyse
# or
vendor/bin/phpstan analyse --memory-limit=256M

# Format code with Laravel Pint
composer format
# or
vendor/bin/pint
```

### Package Development (Orchestra Testbench)
```bash
# Clear workbench
composer clear

# Discover packages
composer prepare

# Build workbench
composer build

# Start development server
composer start
```

## Code Architecture

### Service Provider Pattern
The package uses `MenuServiceProvider` which extends `Bongo\Framework\Providers\AbstractServiceProvider`. This abstract provider automatically handles registration of:
- View composers (defined in `$composers` array)
- Event subscribers (defined in `$subscribers` array)
- Routes from `src/Routes/` directory (api.php, backend.php)
- Views from `src/Views/` directory
- Translations from `src/Translations/` directory
- Migrations from `src/Migrations/` directory
- Config from `src/Config/` directory

The `$module` property defines the package namespace ('menu').

### Model Architecture

**Menu Model** (`Bongo\Menu\Models\Menu`):
- Extends `AbstractModel` from bongo/framework
- Uses traits: `HasStatus`, `HasUUID`, `SoftDeletes`
- Status constants: `PENDING`, `ACTIVE`, `INACTIVE`
- Key relationships:
  - `items()` - HasMany to MenuItem (root level only, ordered by sort_order)
  - `allItems()` - HasMany to all MenuItems (including nested)
- Cache management: `clearCache()` method clears both menu and all nested item caches

**MenuItem Model** (`Bongo\Menu\Models\MenuItem`):
- Uses traits: `HasUUID`, `HasRecursive` (for hierarchical structure)
- Type constants: `EXTERNAL`, `INTERNAL`, `SELF`, `BLANK`
- Supports polymorphic relationships via `entity_type` and `entity_id`
- Key methods:
  - `getLink()` - Resolves the actual URL (external, asset, or internal route)
  - `getEntity()` - Fetches linked entity with caching
  - `isActive()` - Determines if current request matches this menu item
  - `clearCache()` - Clears entity cache

### Entity Configuration
Entity types and their routes are configured in `src/Config/menu.php`:
- `entities` - Maps entity types to model classes (Page, Post, Project, etc.)
- `routes` - Maps entity types to named routes for frontend display

To add new entity types, update this config file.

### Controllers

**Backend Controllers**:
- `MenuController` - Full CRUD for menus with nested menu item management
- `MenuDatatableController` - DataTables integration for menu listing

**API Controllers**:
- `MenuController` - API endpoint to show menu with items
- `MenuItemController` - CRUD operations for menu items

### Event Handling
`MenuEventHandler` automatically:
- Generates menu `key` from `name` using slug format on creation
- Clears cache whenever a menu is saved

### Route Structure
- Backend routes: `/menu/*` with names prefixed `menu.*`
- API routes: `/menus/{id}` and `/menu-items/*` with names prefixed `menu.*` and `menu_item.*`

## Testing Guidelines

### Test Structure
- Extend `Bongo\Menu\Tests\TestCase` (not the base PHPUnit TestCase)
- TestCase configures Orchestra Testbench with MenuServiceProvider
- Uses in-memory SQLite database for testing
- All test methods should have `@test` annotation or `test_` prefix
- Include `declare(strict_types=1);` at the top of test files

### Running Specific Tests
```bash
# Run a specific test class
vendor/bin/phpunit tests/Unit/MenuTest.php

# Run a specific test method
vendor/bin/phpunit --filter it_can_test
```

## Code Style

### Formatting Rules (Laravel Pint)
- Preset: Laravel
- Strict types declarations required: `declare(strict_types=1);`
- New without parentheses: disabled
- Nullable type declaration for default null value: disabled

### Static Analysis (PHPStan)
- Level: 1 (currently set low, consider increasing as codebase matures)
- Ignores undefined property/method errors in Models, Resources, and Feature tests (necessary for Eloquent magic methods)
- Paths checked: `src/` and `config/`

## Common Patterns

### Creating a New Menu Item Type
1. Add entity class to `src/Config/menu.php` in `entities` array
2. Add corresponding route to `routes` array
3. The `MenuItem::getLink()` method will automatically resolve the URL

### Cache Management
Menus and their items are cached. When updating:
- Menu save automatically triggers cache clear via MenuEventHandler
- Manual cache clearing: `$menu->clearCache()` or `$menuItem->clearCache()`
- Cache keys based on entity type and ID

### View Composers
Menu views automatically receive menu data via `MenuComposer` registered for:
- `menu::frontend.partials.menu`
- `menu::frontend.partials.top_menu`
- `menu::frontend.partials.main_menu`
- `menu::frontend.partials.footer_menu`
- `menu::frontend.partials.footer_menu_2`
- `menu::frontend.partials.bottom_menu`
