<?php

declare(strict_types=1);

namespace Bongo\Menu\Models;

use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasStatus;
use Bongo\Framework\Traits\HasUUID;
use Bongo\Menu\Database\Factories\MenuFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Cache;

/**
 * @method whenLoaded(string $string)
 */
class Menu extends AbstractModel
{
    public const PENDING = 'pending';

    public const ACTIVE = 'active';

    public const INACTIVE = 'inactive';

    use HasFactory;
    use HasStatus;
    use HasUUID;
    use SoftDeletes;

    protected $fillable = [
        'name',
        'key',
    ];

    public function items(): HasMany
    {
        return $this->hasMany(MenuItem::class, 'menu_id')
            ->whereNull('parent_id')
            ->orderBy('sort_order', 'ASC');
    }

    public function allItems(): HasMany
    {
        return $this->hasMany(MenuItem::class, 'menu_id')
            ->orderBy('sort_order', 'ASC');
    }

    public function hasItems(): bool
    {
        $this->loadMissing('items');

        return ! is_null($this->items) && count($this->items);
    }

    public function clearCache(): void
    {
        // Clear the menu cache
        Cache::forget($this->key);

        $this->loadMissing('allItems');

        // Clear any cached child items
        if (! is_null($this->allItems) && count($this->allItems)) {
            foreach ($this->allItems as $menuItem) {
                $menuItem->clearCache();
            }
        }
    }

    protected static function newFactory(): MenuFactory
    {
        return MenuFactory::new();
    }
}
