# GitHub Copilot Instructions - Bongo OpenAI Package

## Project Overview

Laravel package providing OpenAI integration for the Bongo CMS monorepo. Offers fluent API for AI-powered content generation using OpenAI's Chat API.

**Tech Stack**: PHP 8.2+, Laravel 10-11, openai-php/laravel ^0.11, bongo/framework ^3.0

## Key Classes & Relationships

```
OpenAIServiceProvider (extends AbstractServiceProvider)
    │
    ├─ Auto-loads: src/Config/openai.php
    ├─ Boot hook: Sets API key from settings if env not provided
    │
    └─ Enables:
         │
         AbstractAI (abstract class)
         │   ├─ Properties: $model, $temperature, $maxTokens, $topP, penalties
         │   ├─ Fluent setters: setModel(), setTemperature(), etc.
         │   └─ Abstract: generate(): ?string
         │
         └─── ContentAI (concrete implementation)
              ├─ Properties: $instruction, $prompt
              ├─ Methods: setInstruction(), setPrompt()
              ├─ Implements: generate() - Calls OpenAI::chat()->create()
              ├─ Validation: validateRequest()
              └─ Response: hasValidResponse(), getContentFromResponse()
```

## Code Style Templates

### 1. Service Provider
```php
declare(strict_types=1);

namespace Bongo\OpenAI;

use Bongo\Framework\Providers\AbstractServiceProvider;

class OpenAIServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'openai';

    public function boot(): void
    {
        parent::boot();

        $this->app->booted(function () {
            // Custom boot logic
        });
    }
}
```

### 2. Abstract AI Service
```php
declare(strict_types=1);

namespace Bongo\OpenAI\Services;

use InvalidArgumentException;

abstract class AbstractAI
{
    protected ?string $model = null;
    protected ?int $temperature = null;

    abstract public function generate(): ?string;

    public function setModel(?string $model = null): self
    {
        $this->model = $model ?? config('openai.model');
        return $this;
    }

    public function setTemperature(?int $temperature = null): self
    {
        $this->temperature = $temperature ?? config('openai.temperature');
        return $this;
    }
}
```

### 3. Concrete AI Service
```php
declare(strict_types=1);

namespace Bongo\OpenAI\Services;

use InvalidArgumentException;
use OpenAI\Laravel\Facades\OpenAI;

class ContentAI extends AbstractAI
{
    protected string $instruction = '';
    protected string $prompt = '';

    public function setInstruction(string $instruction): self
    {
        $this->instruction = $instruction;
        return $this;
    }

    public function setPrompt(string $prompt): self
    {
        $this->prompt = $prompt;
        return $this;
    }

    public function generate(): ?string
    {
        $this->validateRequest();

        $response = OpenAI::chat()->create([
            'model' => $this->model,
            'messages' => [
                ['role' => 'system', 'content' => $this->instruction],
                ['role' => 'user', 'content' => $this->prompt],
            ],
            'temperature' => $this->temperature,
            'max_tokens' => $this->maxTokens,
            'top_p' => $this->topP,
            'frequency_penalty' => $this->frequencyPenalty,
            'presence_penalty' => $this->presencePenalty,
        ]);

        return $this->hasValidResponse($response->toArray())
            ? $this->getContentFromResponse($response->toArray())
            : null;
    }

    protected function validateRequest(): void
    {
        if (empty($this->instruction)) {
            throw new InvalidArgumentException('Instruction cannot be empty.');
        }
        if (empty($this->prompt)) {
            throw new InvalidArgumentException('Prompt cannot be empty.');
        }
    }

    protected function hasValidResponse(array $response): bool
    {
        return isset($response['choices'][0]['message']['content']);
    }

    protected function getContentFromResponse(array $response): ?string
    {
        return nl2br($response['choices'][0]['message']['content'] ?? '');
    }
}
```

### 4. Configuration File
```php
declare(strict_types=1);

return [
    'api_key' => env('OPENAI_API_KEY'),
    'organization' => env('OPENAI_ORGANIZATION'),
    'request_timeout' => env('OPENAI_REQUEST_TIMEOUT', 30),

    'model' => env('OPENAI_MODEL', 'gpt-4.1-mini'),
    'max_tokens' => env('OPENAI_MAX_TOKENS', 500),
    'temperature' => env('OPENAI_TEMPERATURE', 1),
];
```

### 5. Package Seeder
```php
declare(strict_types=1);

namespace Bongo\OpenAI\Seeders;

use Bongo\Package\Models\Package;
use Bongo\Package\Traits\SeedsPackage;
use Illuminate\Database\Seeder;

class PackageSeeder extends Seeder
{
    use SeedsPackage;

    public function run(): void
    {
        $this->package([
            'name' => 'OpenAI',
            'key' => 'openai',
            'route' => 'backend.page',
            'icon' => 'openai',
            'status' => Package::ACTIVE,
            'is_visible' => Package::HIDDEN,
        ]);
    }
}
```

### 6. Test Case
```php
declare(strict_types=1);

namespace Bongo\OpenAI\Tests;

use Bongo\OpenAI\OpenAIServiceProvider;
use Orchestra\Testbench\TestCase as Orchestra;

class TestCase extends Orchestra
{
    protected function setUp(): void
    {
        parent::setUp();
    }

    protected function getPackageProviders($app): array
    {
        return [
            OpenAIServiceProvider::class,
        ];
    }

    protected function getEnvironmentSetUp($app): void
    {
        config()->set('database.default', 'testing');
        config()->set('openai.api_key', 'test-api-key');
    }
}
```

### 7. Service Test
```php
declare(strict_types=1);

namespace Bongo\OpenAI\Tests\Services;

use Bongo\OpenAI\Services\ContentAI;
use Bongo\OpenAI\Tests\TestCase;
use InvalidArgumentException;

class ContentAITest extends TestCase
{
    public function test_it_validates_instruction_is_required(): void
    {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('Instruction cannot be empty.');

        (new ContentAI)
            ->setPrompt('Test prompt')
            ->generate();
    }

    public function test_it_validates_prompt_is_required(): void
    {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('Prompt cannot be empty.');

        (new ContentAI)
            ->setInstruction('Test instruction')
            ->generate();
    }
}
```

## Common Patterns

### Fluent API Usage
```php
use Bongo\OpenAI\Services\ContentAI;

// Basic usage with defaults from config
$content = (new ContentAI)
    ->setInstruction('Rewrite this text to be more professional')
    ->setPrompt($userInput)
    ->generate();

// Full configuration
$content = (new ContentAI)
    ->setInstruction('Summarise the following text')
    ->setPrompt($longText)
    ->setModel('gpt-4.1-mini')
    ->setTemperature(0.7)
    ->setMaxTokens(200)
    ->setTopP(1)
    ->setFrequencyPenalty(0)
    ->setPresencePenalty(0)
    ->generate();

// Check for null response
if ($content === null) {
    // Handle API error or invalid response
    Log::error('OpenAI API failed to generate content');
}
```

### Config Fallback Pattern
```php
// Always provide config fallback for nullable parameters
public function setModel(?string $model = null): self
{
    $this->model = $model ?? config('openai.model');
    return $this;
}
```

### Response Validation Pattern
```php
protected function hasValidResponse(array $response): bool
{
    return isset($response['choices'][0]['message']['content']);
}

protected function getContentFromResponse(array $response): ?string
{
    // Apply nl2br() for HTML newlines
    return nl2br($response['choices'][0]['message']['content'] ?? '');
}
```

### API Key from Settings
```php
// In service provider boot()
$this->app->booted(function () {
    $config = $this->app->make('config');
    if (empty($config->get('openai.api_key')) && ! empty(setting()->getOpenAIApiKey())) {
        $config->set('openai.api_key', setting()->getOpenAIApiKey());
    }
});
```

## Naming Conventions

- **Service classes**: `{Purpose}AI` (ContentAI, ImageAI, EmbeddingAI)
- **Abstract classes**: `Abstract{Name}` (AbstractAI)
- **Methods**: camelCase with verb prefixes (set, get, has, validate, generate)
- **Properties**: camelCase (protected visibility)
- **Config keys**: snake_case matching env vars (api_key, max_tokens, frequency_penalty)
- **Constants**: SCREAMING_SNAKE_CASE (Package::ACTIVE, Package::HIDDEN)

## Important Rules

1. **Always use strict types**: `declare(strict_types=1);` at top of every PHP file
2. **Always type-hint**: Parameters, return types, and properties
3. **Fluent setters**: Always return `self` for method chaining
4. **Config fallbacks**: All nullable parameters use `?? config('openai.key')`
5. **Validation first**: Validate inputs before making API calls
6. **Null on error**: Return `null` from `generate()` on invalid responses, don't throw
7. **nl2br()**: Apply to text content responses for HTML newline conversion
8. **No routes**: This is a service-only package, no HTTP endpoints
9. **Hidden package**: Register as `Package::HIDDEN` in seeder
10. **Test everything**: Every method should have corresponding test

## Configuration Reference

| Parameter | Default | Description |
|-----------|---------|-------------|
| `api_key` | null | OpenAI API key (required) |
| `model` | 'gpt-4.1-mini' | GPT model to use |
| `max_tokens` | 500 | Maximum tokens in response |
| `temperature` | 1 | Randomness (0=deterministic, 2=very random) |
| `top_p` | 1 | Nucleus sampling parameter |
| `frequency_penalty` | 0 | Reduce repetition of token sequences |
| `presence_penalty` | 0 | Reduce repetition of topics |
| `request_timeout` | 30 | API timeout in seconds |

## Dependencies to Know

- `openai-php/laravel` - Official OpenAI SDK, provides `OpenAI::chat()` facade
- `bongo/framework` - Provides `AbstractServiceProvider` for auto-bootstrapping
- `bongo/package` - Provides `SeedsPackage` trait and `Package` model
- `orchestra/testbench` - Laravel package testing framework

## Extension Examples

### Create ImageAI Service
```php
namespace Bongo\OpenAI\Services;

use OpenAI\Laravel\Facades\OpenAI;

class ImageAI extends AbstractAI
{
    protected string $prompt = '';
    protected string $size = '1024x1024';
    protected string $quality = 'standard';

    public function setPrompt(string $prompt): self
    {
        $this->prompt = $prompt;
        return $this;
    }

    public function setSize(string $size): self
    {
        $this->size = $size;
        return $this;
    }

    public function generate(): ?string
    {
        $response = OpenAI::images()->create([
            'model' => $this->model ?? 'dall-e-3',
            'prompt' => $this->prompt,
            'size' => $this->size,
            'quality' => $this->quality,
        ]);

        return $response->data[0]->url ?? null;
    }
}
```

### Add Streaming to ContentAI
```php
public function stream(): Generator
{
    $this->validateRequest();

    $stream = OpenAI::chat()->createStreamed([
        'model' => $this->model,
        'messages' => [
            ['role' => 'system', 'content' => $this->instruction],
            ['role' => 'user', 'content' => $this->prompt],
        ],
    ]);

    foreach ($stream as $response) {
        yield $response->choices[0]->delta->content ?? '';
    }
}
```

## Common Commands

```bash
# Run tests
composer test

# Code formatting
composer format

# Static analysis
composer analyse

# All quality checks
composer format && composer analyse && composer test
```

## Notes for Copilot

- This package is on the `v3.0` development branch
- No migrations yet (future: store requests/responses)
- No backend UI yet (future: API key management)
- Focus on service layer implementation
- All responses should handle null gracefully
- Settings integration is via `setting()->getOpenAIApiKey()` method
