# Architecture Documentation - Bongo OpenAI Package

## Table of Contents

1. [Overview](#overview)
2. [Directory Structure](#directory-structure)
3. [Class Architecture](#class-architecture)
4. [Service Provider Lifecycle](#service-provider-lifecycle)
5. [API Request Flow](#api-request-flow)
6. [Configuration System](#configuration-system)
7. [Extension Points](#extension-points)
8. [Design Patterns](#design-patterns)
9. [Dependencies](#dependencies)
10. [Testing Architecture](#testing-architecture)

---

## Overview

The Bongo OpenAI package is a Laravel service package that provides a fluent, configurable interface for integrating OpenAI's API into the Bongo CMS monorepo. It follows the Bongo package architecture by extending `AbstractServiceProvider` and implements a template method pattern for extensible AI services.

**Package Characteristics**:
- **Type**: Service-only (no HTTP endpoints, views, or migrations)
- **Status**: Development (v3.0 branch, 4 commits)
- **Lines of Code**: ~264 (excluding tests)
- **Dependencies**: openai-php/laravel ^0.11, bongo/framework ^3.0
- **PHP Version**: 8.2+
- **Laravel Version**: 10-11

---

## Directory Structure

```
/Users/stuart/Packages/bongo/cms/openai/
│
├── src/                                # Source code (264 LOC)
│   ├── Config/
│   │   └── openai.php                  # Configuration with env overrides (42 lines)
│   ├── Services/
│   │   ├── AbstractAI.php              # Abstract base class (109 lines)
│   │   └── ContentAI.php               # Chat-based content generation (85 lines)
│   ├── Seeders/
│   │   └── PackageSeeder.php           # Package registration (26 lines)
│   └── OpenAIServiceProvider.php       # Service provider (47 lines)
│
├── tests/                              # Test suite (Orchestra Testbench)
│   ├── TestCase.php                    # Base test case (27 lines)
│   └── ExampleTest.php                 # Placeholder test (13 lines)
│
├── resources/                          # Empty (future: views, assets)
├── config/                             # Empty (config in src/Config/)
├── database/                           # Empty (future: migrations)
│
├── composer.json                       # Package definition and dependencies
├── phpunit.xml.dist                    # PHPUnit configuration
├── pint.json                           # Laravel Pint code style rules
├── phpstan.neon.dist                   # PHPStan static analysis config
├── README.md                           # Package documentation
├── LICENSE                             # MIT license
└── .gitignore                          # Git ignore rules
```

**Key Directories**:
- `src/Services/` - AI service implementations (abstract base + concrete classes)
- `src/Config/` - Package configuration (auto-loaded by service provider)
- `src/Seeders/` - Database seeders for package registration in Bongo system
- `tests/` - PHPUnit tests with Orchestra Testbench setup

---

## Class Architecture

### Class Hierarchy

```
AbstractServiceProvider (from bongo/framework)
    │
    └── OpenAIServiceProvider
            │
            ├── Registers: src/Config/openai.php
            ├── Auto-sets: API key from settings (boot hook)
            │
            └── Enables:
                    │
                    AbstractAI (abstract class)
                    │   ├── Properties: Model parameters (7 properties)
                    │   ├── Methods: Fluent setters (7 methods)
                    │   └── Abstract: generate(): ?string
                    │
                    └── ContentAI (concrete implementation)
                            ├── Additional Properties: $instruction, $prompt
                            ├── Methods: setInstruction(), setPrompt()
                            ├── Implements: generate() - OpenAI API call
                            └── Helpers: validate, check, extract
```

### Class Diagram (ASCII)

```
┌─────────────────────────────────────────────────────────────┐
│                  AbstractServiceProvider                     │
│                   (from bongo/framework)                     │
└─────────────────────────┬───────────────────────────────────┘
                          │ extends
                          │
┌─────────────────────────▼───────────────────────────────────┐
│              OpenAIServiceProvider                           │
├──────────────────────────────────────────────────────────────┤
│ + string $module = 'openai'                                  │
├──────────────────────────────────────────────────────────────┤
│ + boot(): void                                               │
│   ├─ parent::boot()                                          │
│   └─ app->booted(): Set API key from settings if empty      │
└──────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────┐
│                      AbstractAI                              │
│                   (abstract base class)                      │
├──────────────────────────────────────────────────────────────┤
│ # ?string $model                                             │
│ # ?int $temperature                                          │
│ # ?int $maxTokens                                            │
│ # ?int $topP                                                 │
│ # ?int $frequencyPenalty                                     │
│ # ?int $presencePenalty                                      │
├──────────────────────────────────────────────────────────────┤
│ + setModel(?string): self                                    │
│ + setTemperature(?int): self                                 │
│ + setMaxTokens(?int): self                                   │
│ + setTopP(?int): self                                        │
│ + setFrequencyPenalty(?int): self                            │
│ + setPresencePenalty(?int): self                             │
│ + abstract generate(): ?string                               │
└─────────────────────────┬───────────────────────────────────┘
                          │ extends
                          │
┌─────────────────────────▼───────────────────────────────────┐
│                      ContentAI                               │
│                (chat-based content generation)               │
├──────────────────────────────────────────────────────────────┤
│ # string $instruction = ''                                   │
│ # string $prompt = ''                                        │
├──────────────────────────────────────────────────────────────┤
│ + setInstruction(string): self                               │
│ + setPrompt(string): self                                    │
│ + generate(): ?string                                        │
│ # validateRequest(): void                                    │
│ # hasValidResponse(array): bool                              │
│ # getContentFromResponse(array): ?string                     │
└──────────────────────────────────────────────────────────────┘
```

### Class Reference Table

| Class | Type | Extends | Location | Responsibility |
|-------|------|---------|----------|----------------|
| `OpenAIServiceProvider` | Provider | `AbstractServiceProvider` | `src/OpenAIServiceProvider.php` | Bootstraps package, auto-sets API key from settings |
| `AbstractAI` | Abstract | - | `src/Services/AbstractAI.php` | Base class for AI services with fluent builder pattern |
| `ContentAI` | Concrete | `AbstractAI` | `src/Services/ContentAI.php` | Chat-based content generation via OpenAI API |
| `PackageSeeder` | Seeder | `Illuminate\Database\Seeder` | `src/Seeders/PackageSeeder.php` | Registers package in Bongo package system |
| `TestCase` | Test | `Orchestra\Testbench\TestCase` | `tests/TestCase.php` | Base test case for package tests |

---

## Service Provider Lifecycle

### Bootstrapping Flow

```
┌──────────────────────────────────────────────────────────────────┐
│                    Laravel Application Boot                      │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│             OpenAIServiceProvider::register()                    │
│                    (inherited from parent)                       │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│             OpenAIServiceProvider::boot()                        │
├──────────────────────────────────────────────────────────────────┤
│  1. parent::boot()                                               │
│     ├─ Auto-load: src/Config/openai.php                          │
│     ├─ Auto-register: Routes (none in this package)              │
│     ├─ Auto-register: Views (none in this package)               │
│     ├─ Auto-register: Migrations (none in this package)          │
│     └─ Auto-register: Translations (none in this package)        │
│                                                                  │
│  2. app->booted(callback)                                        │
│     └─ Check: If config('openai.api_key') is empty              │
│        └─ If setting()->getOpenAIApiKey() exists                 │
│           └─ Set: config('openai.api_key', settingValue)         │
└──────────────────────────────────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│                  Package Ready for Use                           │
│     ContentAI and other services can now be instantiated         │
└──────────────────────────────────────────────────────────────────┘
```

### Boot Method Code

```php
public function boot(): void
{
    parent::boot();

    // Custom boot logic: API key from settings fallback
    $this->app->booted(function () {
        $config = $this->app->make('config');
        if (empty($config->get('openai.api_key')) && ! empty(setting()->getOpenAIApiKey())) {
            $config->set('openai.api_key', setting()->getOpenAIApiKey());
        }
    });
}
```

**Key Points**:
1. `parent::boot()` handles all auto-loading (config, routes, views, etc.)
2. Custom `app->booted()` callback runs after Laravel is fully booted
3. API key priority: Environment variable > Bongo settings system
4. Settings integration requires `setting()->getOpenAIApiKey()` method to exist

---

## API Request Flow

### ContentAI Generation Flow

```
┌──────────────────────────────────────────────────────────────────┐
│                        Client Code                               │
│  $content = (new ContentAI)                                      │
│      ->setInstruction('Rewrite as formal English')               │
│      ->setPrompt($userText)                                      │
│      ->setModel('gpt-4.1-mini')                                  │
│      ->setTemperature(1)                                         │
│      ->generate();                                               │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│               ContentAI::generate()                              │
├──────────────────────────────────────────────────────────────────┤
│  Step 1: Validate Request                                        │
│    ├─ validateRequest()                                          │
│    ├─ Check: $instruction not empty                              │
│    ├─ Check: $prompt not empty                                   │
│    └─ Throw: InvalidArgumentException if validation fails        │
└────────────────────────────┬─────────────────────────────────────┘
                             │ (if valid)
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│  Step 2: Prepare API Request                                     │
│    ├─ Model: $this->model (from config or setter)                │
│    ├─ Messages: [                                                │
│    │     ['role' => 'system', 'content' => $instruction],        │
│    │     ['role' => 'user', 'content' => $prompt]                │
│    │   ]                                                          │
│    ├─ Temperature: $this->temperature                            │
│    ├─ Max Tokens: $this->maxTokens                               │
│    ├─ Top P: $this->topP                                         │
│    ├─ Frequency Penalty: $this->frequencyPenalty                 │
│    └─ Presence Penalty: $this->presencePenalty                   │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│  Step 3: Execute API Call                                        │
│    $response = OpenAI::chat()->create([...])                     │
│                                                                  │
│    ┌─────────────────────────────────────────┐                  │
│    │   OpenAI API (openai-php/laravel)       │                  │
│    │   https://api.openai.com/v1/chat/...    │                  │
│    └─────────────────────────────────────────┘                  │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│  Step 4: Validate Response                                       │
│    ├─ Convert: $response->toArray()                              │
│    ├─ Check: hasValidResponse($responseArray)                    │
│    │   └─ isset($response['choices'][0]['message']['content'])   │
│    └─ Branch:                                                    │
│        ├─ Valid: Continue to Step 5                              │
│        └─ Invalid: Return null                                   │
└────────────────────────────┬─────────────────────────────────────┘
                             │ (if valid)
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│  Step 5: Extract Content                                         │
│    ├─ getContentFromResponse($responseArray)                     │
│    ├─ Extract: $response['choices'][0]['message']['content']     │
│    ├─ Apply: nl2br() for HTML newlines                           │
│    └─ Return: Formatted content string                           │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│                       Return to Client                           │
│    string (generated content) OR null (on error/invalid)         │
└──────────────────────────────────────────────────────────────────┘
```

### Error Handling Strategy

| Error Type | Handling | Return Value |
|------------|----------|--------------|
| Empty instruction | Throw `InvalidArgumentException` | N/A (exception) |
| Empty prompt | Throw `InvalidArgumentException` | N/A (exception) |
| Invalid API response | Silent handling | `null` |
| Missing content in response | Silent handling | `null` |
| API network error | Exception from openai-php/laravel | Exception propagates |
| Invalid API key | Exception from openai-php/laravel | Exception propagates |

---

## Configuration System

### Configuration Loading Flow

```
┌──────────────────────────────────────────────────────────────────┐
│                .env File (or environment variables)              │
│  OPENAI_API_KEY=sk-...                                           │
│  OPENAI_MODEL=gpt-4.1-mini                                       │
│  OPENAI_MAX_TOKENS=500                                           │
│  OPENAI_TEMPERATURE=1                                            │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│              src/Config/openai.php                               │
│  return [                                                        │
│      'api_key' => env('OPENAI_API_KEY'),                         │
│      'model' => env('OPENAI_MODEL', 'gpt-4.1-mini'),             │
│      'max_tokens' => env('OPENAI_MAX_TOKENS', 500),              │
│      ...                                                         │
│  ];                                                              │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│        AbstractServiceProvider::boot() (parent class)            │
│  Auto-publishes and merges config:                               │
│    config()->set('openai', require('src/Config/openai.php'))     │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│        OpenAIServiceProvider::boot() (custom hook)               │
│  IF config('openai.api_key') is empty                            │
│  AND setting()->getOpenAIApiKey() exists                         │
│  THEN config()->set('openai.api_key', settingValue)              │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│             AbstractAI Fluent Setters (runtime)                  │
│  ->setModel(?string $model = null)                               │
│    $this->model = $model ?? config('openai.model');              │
│                                                                  │
│  All setters fall back to config if null passed                  │
└────────────────────────────┬─────────────────────────────────────┘
                             │
                             ▼
┌──────────────────────────────────────────────────────────────────┐
│                    Final Runtime Values                          │
│  Used in ContentAI::generate() API call                          │
└──────────────────────────────────────────────────────────────────┘
```

### Configuration Parameters Table

| Parameter | Type | Default | Env Var | Scope | Description |
|-----------|------|---------|---------|-------|-------------|
| `api_key` | string\|null | null | `OPENAI_API_KEY` | Required | OpenAI API authentication key |
| `organization` | string\|null | null | `OPENAI_ORGANIZATION` | Optional | OpenAI organization ID |
| `project` | string\|null | null | `OPENAI_PROJECT` | Optional | OpenAI project ID |
| `base_uri` | string\|null | null | `OPENAI_BASE_URL` | Optional | Custom API base URL |
| `request_timeout` | int | 30 | `OPENAI_REQUEST_TIMEOUT` | Optional | Request timeout (seconds) |
| `model` | string | 'gpt-4.1-mini' | `OPENAI_MODEL` | Required | Default GPT model |
| `max_tokens` | int | 500 | `OPENAI_MAX_TOKENS` | Optional | Max tokens in response |
| `temperature` | int | 1 | `OPENAI_TEMPERATURE` | Optional | Sampling temperature (0-2) |
| `top_p` | int | 1 | `OPENAI_TOP_P` | Optional | Nucleus sampling |
| `frequency_penalty` | int | 0 | `OPENAI_FREQUENCY_PENALTY` | Optional | Token frequency penalty (0-2) |
| `presence_penalty` | int | 0 | `OPENAI_PRESENCE_PENALTY` | Optional | Topic presence penalty (0-2) |

### Configuration Priority

1. **Runtime setter call**: `->setModel('gpt-4')` (highest priority)
2. **Environment variable**: `OPENAI_MODEL=gpt-4`
3. **Bongo settings system**: `setting()->getOpenAIApiKey()` (API key only)
4. **Config default**: `'gpt-4.1-mini'` (lowest priority)

---

## Extension Points

### 1. Adding New AI Service Types

**Pattern**: Extend `AbstractAI` and implement `generate()`

```php
namespace Bongo\OpenAI\Services;

use OpenAI\Laravel\Facades\OpenAI;

class ImageAI extends AbstractAI
{
    protected string $prompt = '';
    protected string $size = '1024x1024';
    protected string $quality = 'standard';

    public function setPrompt(string $prompt): self
    {
        $this->prompt = $prompt;
        return $this;
    }

    public function setSize(string $size): self
    {
        $this->size = $size;
        return $this;
    }

    public function setQuality(string $quality): self
    {
        $this->quality = $quality;
        return $this;
    }

    public function generate(): ?string
    {
        $response = OpenAI::images()->create([
            'model' => $this->model ?? 'dall-e-3',
            'prompt' => $this->prompt,
            'size' => $this->size,
            'quality' => $this->quality,
        ]);

        return $response->data[0]->url ?? null;
    }
}
```

**Steps**:
1. Create new class in `src/Services/`
2. Extend `AbstractAI`
3. Add service-specific properties
4. Add fluent setters
5. Implement `generate()` using appropriate OpenAI facade method
6. Add configuration in `src/Config/openai.php`
7. Add tests in `tests/Services/`

### 2. Adding Configuration Parameters

**Example**: Add streaming support

**Step 1**: Add to `src/Config/openai.php`
```php
'stream' => env('OPENAI_STREAM', false),
```

**Step 2**: Add property to `AbstractAI`
```php
protected ?bool $stream = null;
```

**Step 3**: Add fluent setter
```php
public function setStream(?bool $stream = null): self
{
    $this->stream = $stream ?? config('openai.stream');
    return $this;
}
```

**Step 4**: Use in API call
```php
$response = OpenAI::chat()->create([
    // ... other params
    'stream' => $this->stream,
]);
```

### 3. Customizing Response Processing

**Override** `getContentFromResponse()` in concrete implementations:

```php
protected function getContentFromResponse(array $response): ?string
{
    $content = $response['choices'][0]['message']['content'] ?? '';

    // Custom processing examples:
    // - Parse JSON responses
    // - Convert Markdown to HTML
    // - Extract specific data structures
    // - Apply custom formatting

    return $this->customProcess($content);
}
```

### 4. Adding Middleware or Event Listeners

Add to service provider:

```php
class OpenAIServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'openai';

    protected array $middlewares = [
        'openai.rate-limit' => RateLimitOpenAI::class,
    ];

    protected array $listeners = [
        OpenAIRequestSent::class => [
            LogOpenAIRequest::class,
        ],
    ];
}
```

### 5. Adding Validation Rules

**Example**: Custom request validation

```php
class ContentAI extends AbstractAI
{
    protected function validateRequest(): void
    {
        if (empty($this->instruction)) {
            throw new InvalidArgumentException('Instruction cannot be empty.');
        }

        if (empty($this->prompt)) {
            throw new InvalidArgumentException('Prompt cannot be empty.');
        }

        // Additional custom validations
        if (strlen($this->prompt) > 10000) {
            throw new InvalidArgumentException('Prompt exceeds 10,000 characters.');
        }

        if ($this->temperature < 0 || $this->temperature > 2) {
            throw new InvalidArgumentException('Temperature must be between 0 and 2.');
        }
    }
}
```

---

## Design Patterns

### 1. Template Method Pattern

**Class**: `AbstractAI`

**Structure**:
```
AbstractAI (defines skeleton)
    ├─ Concrete properties: $model, $temperature, etc.
    ├─ Concrete methods: All fluent setters
    └─ Abstract method: generate() (subclasses must implement)

ContentAI (implements template)
    ├─ Additional properties: $instruction, $prompt
    ├─ Additional methods: setInstruction(), setPrompt()
    └─ Implements: generate() with specific OpenAI API call
```

**Benefits**:
- Reusable parameter configuration across all AI services
- Consistent fluent API interface
- Forced implementation of `generate()` in subclasses

### 2. Fluent Builder Pattern

**All setters return** `self` **for method chaining**:

```php
$content = (new ContentAI)
    ->setInstruction('...')   // Returns $this
    ->setPrompt('...')         // Returns $this
    ->setModel('...')          // Returns $this
    ->setTemperature(0.7)     // Returns $this
    ->generate();              // Executes and returns string|null
```

**Benefits**:
- Readable, expressive API
- Easy to configure complex requests
- IDE-friendly auto-completion

### 3. Service Provider Pattern

**Lazy loading and bootstrapping**:

```php
OpenAIServiceProvider extends AbstractServiceProvider
    ├─ register(): Binds services to container (if needed)
    ├─ boot(): Auto-loads config, routes, views, etc.
    └─ Custom boot hooks: API key from settings
```

**Benefits**:
- Automatic package discovery
- Consistent bootstrap process across Bongo packages
- Clean separation of concerns

### 4. Facade Pattern

**Uses** `OpenAI::chat()` **facade from** `openai-php/laravel`:

```php
use OpenAI\Laravel\Facades\OpenAI;

$response = OpenAI::chat()->create([...]);
```

**Benefits**:
- Simple static API access
- No need to manually instantiate clients
- Automatic configuration from Laravel config

### 5. Null Object Pattern (Partial)

**Returns** `null` **instead of throwing exceptions on invalid responses**:

```php
public function generate(): ?string
{
    // ...
    return $this->hasValidResponse($response->toArray())
        ? $this->getContentFromResponse($response->toArray())
        : null;  // Null object instead of exception
}
```

**Benefits**:
- Graceful degradation on API errors
- Client code can handle nulls easily
- Avoids try-catch blocks for expected failures

---

## Dependencies

### Production Dependencies

```json
{
    "php": "^8.2",
    "bongo/framework": "^3.0",
    "illuminate/contracts": "^10.0||^11.0",
    "openai-php/laravel": "^0.11.0"
}
```

**Dependency Graph**:

```
openai package
    │
    ├── php ^8.2
    │   └── Required for strict types, union types, etc.
    │
    ├── bongo/framework ^3.0
    │   ├── Provides: AbstractServiceProvider
    │   ├── Provides: Auto-loading infrastructure
    │   └── Used by: OpenAIServiceProvider
    │
    ├── illuminate/contracts ^10.0||^11.0
    │   ├── Provides: Laravel service container contracts
    │   └── Used by: Service provider registration
    │
    └── openai-php/laravel ^0.11.0
        ├── Provides: OpenAI::chat() facade
        ├── Provides: OpenAI API client
        └── Used by: ContentAI::generate()
```

### Development Dependencies

```json
{
    "laravel/pint": "^1.14",
    "larastan/larastan": "^2.9",
    "nunomaduro/collision": "^8.1.1||^7.10.0",
    "bongo/package": "^3.0",
    "orchestra/testbench": "^9.0.0||^8.22.0",
    "phpstan/extension-installer": "^1.3",
    "phpstan/phpstan-deprecation-rules": "^1.3",
    "phpstan/phpstan-phpunit": "^1.3"
}
```

**Usage**:
- **laravel/pint**: Code formatting (Composer script: `composer format`)
- **larastan**: Static analysis (Composer script: `composer analyse`)
- **orchestra/testbench**: Package testing framework
- **bongo/package**: Package utilities, `SeedsPackage` trait

---

## Testing Architecture

### Test Foundation

**Base Test Case**: `tests/TestCase.php`

```php
class TestCase extends Orchestra
{
    protected function getPackageProviders($app): array
    {
        return [OpenAIServiceProvider::class];
    }

    protected function getEnvironmentSetUp($app): void
    {
        config()->set('database.default', 'testing');
        config()->set('openai.api_key', 'test-api-key');  // Mock key
    }
}
```

**Key Points**:
- Extends Orchestra Testbench (not Laravel TestCase)
- Registers OpenAIServiceProvider automatically
- Sets test database and mock API key

### Test Structure

```
tests/
├── TestCase.php                         # Base test case
├── ExampleTest.php                      # Placeholder
└── (future)
    ├── Services/
    │   ├── AbstractAITest.php           # Test fluent setters
    │   └── ContentAITest.php            # Test generate(), validation
    ├── OpenAIServiceProviderTest.php    # Test boot logic
    └── Integration/
        └── ContentGenerationTest.php    # End-to-end tests
```

### PHPUnit Configuration

**File**: `phpunit.xml.dist`

```xml
<phpunit
    bootstrap="vendor/autoload.php"
    executionOrder="random"
    failOnWarning="true"
    failOnRisky="true">

    <testsuites>
        <testsuite name="default">
            <directory>tests</directory>
        </testsuite>
    </testsuites>

    <source>
        <include>
            <directory>./src</directory>
        </include>
    </source>

    <coverage>
        <report>
            <html outputDirectory="build/coverage"/>
            <clover outputFile="build/logs/clover.xml"/>
        </report>
    </coverage>
</phpunit>
```

**Features**:
- Random test execution order
- Fails on warnings and risky tests
- Code coverage reports (HTML + Clover XML)
- JUnit XML reports for CI/CD

### Composer Test Scripts

```bash
# Basic test run (no coverage)
composer test

# Parallel test execution
composer test:parallel

# Full coverage report
composer test:coverage
```

---

## How to Add New Features

### Example: Add Function Calling Support

**Step 1**: Add configuration

```php
// src/Config/openai.php
'functions' => [
    'enabled' => env('OPENAI_FUNCTIONS_ENABLED', false),
],
```

**Step 2**: Create service class

```php
// src/Services/FunctionAI.php
namespace Bongo\OpenAI\Services;

use OpenAI\Laravel\Facades\OpenAI;

class FunctionAI extends AbstractAI
{
    protected string $prompt = '';
    protected array $functions = [];

    public function setPrompt(string $prompt): self
    {
        $this->prompt = $prompt;
        return $this;
    }

    public function addFunction(array $function): self
    {
        $this->functions[] = $function;
        return $this;
    }

    public function generate(): ?string
    {
        $response = OpenAI::chat()->create([
            'model' => $this->model,
            'messages' => [
                ['role' => 'user', 'content' => $this->prompt],
            ],
            'functions' => $this->functions,
            'function_call' => 'auto',
        ]);

        if (isset($response['choices'][0]['message']['function_call'])) {
            return json_encode($response['choices'][0]['message']['function_call']);
        }

        return $response['choices'][0]['message']['content'] ?? null;
    }
}
```

**Step 3**: Add tests

```php
// tests/Services/FunctionAITest.php
namespace Bongo\OpenAI\Tests\Services;

use Bongo\OpenAI\Services\FunctionAI;
use Bongo\OpenAI\Tests\TestCase;

class FunctionAITest extends TestCase
{
    public function test_it_adds_functions(): void
    {
        $ai = new FunctionAI;

        $ai->addFunction([
            'name' => 'get_weather',
            'description' => 'Get current weather',
            'parameters' => [...],
        ]);

        // Assert function was added
        $this->assertCount(1, $ai->getFunctions());
    }
}
```

**Step 4**: Update documentation

- Add to README.md usage examples
- Add to ARCHITECTURE.md extension points
- Add to .cursorrules common tasks

---

## Summary

This architecture provides:

✅ **Extensible service layer** via abstract base class
✅ **Fluent, chainable API** for easy configuration
✅ **Automatic bootstrapping** via service provider
✅ **Settings integration** for API key fallback
✅ **Type-safe implementation** with strict types enforced
✅ **Testable design** with Orchestra Testbench
✅ **Clean separation** between config, services, and registration
✅ **Future-ready** for additional AI service types (images, embeddings, etc.)

The package follows Laravel and Bongo best practices while maintaining simplicity and extensibility.
