<?php

declare(strict_types=1);

namespace Bongo\OpenAI\Services;

use InvalidArgumentException;
use OpenAI\Laravel\Facades\OpenAI;

class ContentAI extends AbstractAI
{
    protected string $instruction;

    protected string $prompt;

    /**
     * For example, "Rewrite the following content to be standard English and plagiarism free".
     */
    public function setInstruction(string $instruction): self
    {
        $this->instruction = $instruction;

        return $this;
    }

    /**
     * For example, "Write me a blog post about the benefits of using AI in content creation".
     */
    public function setPrompt(string $prompt): self
    {
        $this->prompt = $prompt;

        return $this;
    }

    public function generate(): ?string
    {
        $this->validateRequest();

        $response = OpenAI::chat()->create([
            'model' => $this->model,
            'messages' => [
                ['role' => 'system', 'content' => $this->instruction],
                ['role' => 'user', 'content' => $this->prompt],
            ],
            'temperature' => $this->temperature,
            'max_tokens' => $this->maxTokens,
            'top_p' => $this->topP,
            'frequency_penalty' => $this->frequencyPenalty,
            'presence_penalty' => $this->presencePenalty,
        ]);

        return $this->getContentFromResponse((array) $response);
    }

    protected function validateRequest(): void
    {
        if (empty($this->instruction)) {
            throw new InvalidArgumentException('Instruction must be set before running the request.');
        }
        if (empty($this->prompt)) {
            throw new InvalidArgumentException('Prompt must be set before running the request.');
        }
    }

    protected function hasValidResponse(array $response): bool
    {
        return ! empty($response['choices'])
            && ! empty($response['choices'][0])
            && ! empty($response['choices'][0]['message'])
            && ! empty($response['choices'][0]['message']['content']);
    }

    protected function getContentFromResponse(array $response): ?string
    {
        if ($this->hasValidResponse($response)) {
            return nl2br($response['choices'][0]['message']['content']);
        }

        return null;
    }
}
