# CLAUDE.md - Bongo OpenAI Package

## Overview

Laravel package for OpenAI integration in the Bongo CMS monorepo. Provides fluent, configurable interface for AI-powered content generation using OpenAI's Chat API.

**Package**: `bongo/openai`
**Version**: v3.0 (development branch)
**PHP**: 8.2+
**Laravel**: 10-11
**Status**: Early development (4 commits, ~264 LOC)

**Key Dependency**: `openai-php/laravel` ^0.11.0 (Official OpenAI PHP SDK)

## Documentation

- **ARCHITECTURE.md** - Detailed architecture, class diagrams, flows, extension points
- **.cursorrules** - Cursor AI instructions with coding conventions and common tasks
- **.github/copilot-instructions.md** - GitHub Copilot code templates and patterns
- **README.md** - Package overview, installation, and usage examples

## Quick Start

### Installation

```bash
composer require bongo/openai
```

### Configuration

Add to `.env`:
```bash
OPENAI_API_KEY=sk-your-api-key
OPENAI_MODEL=gpt-4.1-mini
OPENAI_MAX_TOKENS=500
OPENAI_TEMPERATURE=1
```

### Basic Usage

```php
use Bongo\OpenAI\Services\ContentAI;

$content = (new ContentAI)
    ->setInstruction('Rewrite to be standard English and plagiarism free')
    ->setPrompt('Yor orignal content here with typos')
    ->generate();  // Returns formatted string or null

if ($content === null) {
    // Handle API error
}
```

### Advanced Usage

```php
$content = (new ContentAI)
    ->setInstruction('Summarise the following article')
    ->setPrompt($longArticle)
    ->setModel('gpt-4.1-mini')
    ->setTemperature(0.7)
    ->setMaxTokens(200)
    ->setTopP(1)
    ->setFrequencyPenalty(0.5)
    ->setPresencePenalty(0.5)
    ->generate();
```

## Composer Commands

```bash
# Run tests (PHPUnit)
composer test

# Run tests in parallel
composer test:parallel

# Run tests with coverage report
composer test:coverage

# Format code (Laravel Pint)
composer format

# Static analysis (PHPStan)
composer analyse

# Build workbench
composer build

# Start dev server
composer start

# Clear skeleton
composer clear

# Discover packages
composer prepare
```

## Architecture Quick Reference

### Service Provider

Extends `Bongo\Framework\Providers\AbstractServiceProvider` which provides automatic bootstrapping:

- **Config**: Auto-loaded from `src/Config/openai.php`
- **Routes**: None (service-only package)
- **Views**: None
- **Migrations**: None (future)
- **Custom boot hook**: Auto-sets API key from settings if not in env

```php
class OpenAIServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'openai';

    public function boot(): void
    {
        parent::boot();

        // Auto-set API key from settings if env not provided
        $this->app->booted(function () {
            $config = $this->app->make('config');
            if (empty($config->get('openai.api_key')) && ! empty(setting()->getOpenAIApiKey())) {
                $config->set('openai.api_key', setting()->getOpenAIApiKey());
            }
        });
    }
}
```

### Service Layer

**Abstract Base**: `AbstractAI`
- Defines fluent builder pattern for AI parameters
- Properties: `$model`, `$temperature`, `$maxTokens`, `$topP`, penalties
- Methods: All fluent setters with config fallbacks
- Abstract: `generate(): ?string` (must be implemented by subclasses)

**Concrete Implementation**: `ContentAI`
- Extends `AbstractAI`
- Chat-based content generation
- Properties: `$instruction` (system role), `$prompt` (user message)
- Methods: `setInstruction()`, `setPrompt()`, `generate()`
- Uses: `OpenAI::chat()->create()` facade
- Returns: Generated content with `nl2br()` applied, or `null` on error

### Request Flow

```
Client Code
    ↓
ContentAI::generate()
    ↓
validateRequest() - Throws InvalidArgumentException if empty
    ↓
OpenAI::chat()->create([...]) - API call via facade
    ↓
hasValidResponse() - Check response structure
    ↓
getContentFromResponse() - Extract content, apply nl2br()
    ↓
Return string or null
```

## Key Files

| File | Lines | Purpose |
|------|-------|---------|
| `src/OpenAIServiceProvider.php` | 47 | Service provider with API key auto-config |
| `src/Services/AbstractAI.php` | 109 | Abstract base class for AI services |
| `src/Services/ContentAI.php` | 85 | Chat-based content generation |
| `src/Config/openai.php` | 42 | Configuration with env overrides |
| `src/Seeders/PackageSeeder.php` | 26 | Package registration in Bongo system |
| `tests/TestCase.php` | 27 | Orchestra Testbench base test case |

## Configuration

All parameters support environment variable overrides:

| Parameter | Default | Env Var | Description |
|-----------|---------|---------|-------------|
| `api_key` | null | `OPENAI_API_KEY` | OpenAI API key (required) |
| `organization` | null | `OPENAI_ORGANIZATION` | OpenAI organization ID |
| `project` | null | `OPENAI_PROJECT` | OpenAI project ID |
| `base_uri` | null | `OPENAI_BASE_URL` | Custom API base URL |
| `request_timeout` | 30 | `OPENAI_REQUEST_TIMEOUT` | Request timeout (seconds) |
| `model` | 'gpt-4.1-mini' | `OPENAI_MODEL` | Default GPT model |
| `max_tokens` | 500 | `OPENAI_MAX_TOKENS` | Max tokens in response |
| `temperature` | 1 | `OPENAI_TEMPERATURE` | Sampling temperature (0-2) |
| `top_p` | 1 | `OPENAI_TOP_P` | Nucleus sampling |
| `frequency_penalty` | 0 | `OPENAI_FREQUENCY_PENALTY` | Frequency penalty (0-2) |
| `presence_penalty` | 0 | `OPENAI_PRESENCE_PENALTY` | Presence penalty (0-2) |

## Code Style

### Strict Types
All PHP files include:
```php
declare(strict_types=1);
```

### Type Declarations
Always use return types and parameter type hints:
```php
public function generate(): ?string
public function setModel(?string $model = null): self
protected function hasValidResponse(array $response): bool
```

### Fluent Builder
All setters return `self`:
```php
public function setTemperature(?int $temperature = null): self
{
    $this->temperature = $temperature ?? config('openai.temperature');
    return $this;
}
```

### Config Fallbacks
All nullable parameters fall back to config:
```php
$this->model = $model ?? config('openai.model');
$this->maxTokens = $maxTokens ?? config('openai.max_tokens');
```

### Validation
Validate before API calls:
```php
protected function validateRequest(): void
{
    if (empty($this->instruction)) {
        throw new InvalidArgumentException('Instruction cannot be empty.');
    }
    if (empty($this->prompt)) {
        throw new InvalidArgumentException('Prompt cannot be empty.');
    }
}
```

### Response Handling
Check validity before extracting:
```php
return $this->hasValidResponse($response->toArray())
    ? $this->getContentFromResponse($response->toArray())
    : null;
```

## Extension Points

### Add New AI Service Type

1. Create class extending `AbstractAI`:
```php
namespace Bongo\OpenAI\Services;

class ImageAI extends AbstractAI
{
    protected string $prompt = '';
    protected string $size = '1024x1024';

    public function generate(): ?string
    {
        $response = OpenAI::images()->create([...]);
        return $response->data[0]->url ?? null;
    }
}
```

2. Add config section in `src/Config/openai.php`
3. Add tests in `tests/Services/ImageAITest.php`

### Add Configuration Parameter

1. Add to `src/Config/openai.php`:
```php
'stream' => env('OPENAI_STREAM', false),
```

2. Add property and setter to `AbstractAI`:
```php
protected ?bool $stream = null;

public function setStream(?bool $stream = null): self
{
    $this->stream = $stream ?? config('openai.stream');
    return $this;
}
```

3. Use in API call

## Testing

**Framework**: Orchestra Testbench

**Base Test Case**:
```php
class TestCase extends Orchestra
{
    protected function getPackageProviders($app): array
    {
        return [OpenAIServiceProvider::class];
    }

    protected function getEnvironmentSetUp($app): void
    {
        config()->set('database.default', 'testing');
        config()->set('openai.api_key', 'test-api-key');
    }
}
```

**Run Tests**:
```bash
composer test              # Basic
composer test:parallel     # Parallel execution
composer test:coverage     # With coverage report
```

## Package Registration

Registered in Bongo package system via seeder:

```php
$this->package([
    'name' => 'OpenAI',
    'key' => 'openai',
    'route' => 'backend.page',
    'icon' => 'openai',
    'status' => Package::ACTIVE,
    'is_visible' => Package::HIDDEN,  // Not shown in admin UI
]);
```

Run: `php artisan db:seed --class=Bongo\\OpenAI\\Seeders\\PackageSeeder`

## Dependencies

**Production**:
- `openai-php/laravel` ^0.11.0 - Official OpenAI SDK
- `bongo/framework` ^3.0 - AbstractServiceProvider, package system
- `illuminate/contracts` ^10.0||^11.0 - Laravel contracts

**Development**:
- `laravel/pint` ^1.14 - Code formatting
- `larastan/larastan` ^2.9 - Static analysis
- `orchestra/testbench` ^9.0.0||^8.22.0 - Package testing
- `bongo/package` ^3.0 - Package utilities

## Important Notes

- **No HTTP endpoints**: Service-only package with no routes, controllers, or middleware
- **Settings integration**: API key can come from env OR Bongo settings system (`setting()->getOpenAIApiKey()`)
- **Null returns**: `generate()` returns `null` on invalid responses (no exceptions)
- **nl2br()**: Content responses convert newlines to `<br>` tags automatically
- **Hidden package**: Not visible in admin panel UI
- **Extensible**: Abstract base enables multiple AI service types

## Development Status

**Branch**: v3.0 (feature branch)
**Commits**: 4
**Recent Changes**:
- ✅ Service provider with API key auto-config
- ✅ AbstractAI base class with fluent builder
- ✅ ContentAI chat-based generation
- ✅ Configuration system
- ✅ Package registration seeder

**TODO**:
- ⏳ Complete test suite
- ⏳ Add ImageAI service
- ⏳ Add EmbeddingAI service
- ⏳ Add migrations for request/response storage
- ⏳ Add backend UI for API key management
- ⏳ Add streaming support
- ⏳ Add usage tracking/logging

## Working with This Package

This package is part of the Bongo monorepo. Each package is a separate git repository.

### Common Workflows

**Make changes**:
```bash
cd /Users/stuart/Packages/bongo/cms/openai
# Make your changes
vendor/bin/pint              # Format
vendor/bin/phpstan           # Analyse
vendor/bin/phpunit           # Test
git add . && git commit -m "Your message"
```

**Using batch scripts** (from monorepo root):
```bash
# Update dependencies
cd cms && ./_update.sh

# Format all packages
cd cms && ./_pint.sh

# Commit changes
./scripts/commit_packages/commit-packages.sh --dry-run
./scripts/commit_packages/commit-packages.sh -m "Your message"
```

See `/Users/stuart/Packages/Bongo/CLAUDE.md` for full monorepo documentation.

## Contact

**Author**: Stuart Elliott
**License**: MIT
**Repository**: Bitbucket (private)
