# Bongo OpenAI Package

Laravel package for OpenAI integration in the Bongo CMS monorepo. Provides a fluent, configurable interface for AI-powered content generation using OpenAI's Chat API.

[![PHP Version](https://img.shields.io/badge/php-%5E8.2-blue)](https://www.php.net/)
[![Laravel Version](https://img.shields.io/badge/laravel-10%20%7C%2011-orange)](https://laravel.com/)

## Features

- **Fluent API**: Chainable method calls for easy configuration
- **Type-Safe**: Strict type checking with PHP 8.2+
- **Extensible**: Abstract base class for creating new AI service types
- **Settings Integration**: API key can be sourced from environment or Bongo settings
- **Configurable**: All parameters have environment variable overrides with sensible defaults
- **Service-Only**: Lightweight package with no HTTP endpoints or views

## Stability Notice

This package is actively being developed on the `v3.0` branch. We welcome feedback and contributions to improve it. Please submit issues or pull requests via Bitbucket.

## Requirements

- PHP 8.2+
- Laravel 10 or 11
- OpenAI API key ([Get one here](https://platform.openai.com/api-keys))

## Installation

### Composer

Install the package via Composer:

```bash
composer require bongo/openai
```

### Service Provider Registration

The service provider is auto-discovered in Laravel 10+. For manual registration, add to `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\OpenAI\OpenAIServiceProvider::class,
],
```

### Configuration

Publish the configuration file (optional):

```bash
php artisan vendor:publish --provider="Bongo\OpenAI\OpenAIServiceProvider"
```

Add your OpenAI API key to `.env`:

```bash
OPENAI_API_KEY=sk-your-api-key-here
```

#### Optional Environment Variables

```bash
# API Authentication
OPENAI_API_KEY=sk-...
OPENAI_ORGANIZATION=org-...
OPENAI_PROJECT=proj_...

# API Configuration
OPENAI_BASE_URL=https://api.openai.com/v1
OPENAI_REQUEST_TIMEOUT=30

# Model Parameters
OPENAI_MODEL=gpt-4.1-mini
OPENAI_MAX_TOKENS=500
OPENAI_TEMPERATURE=1
OPENAI_TOP_P=1
OPENAI_FREQUENCY_PENALTY=0
OPENAI_PRESENCE_PENALTY=0
```

### Package Registration

Run the package seeder to register in the Bongo package system:

```bash
php artisan db:seed --class=Bongo\\OpenAI\\Seeders\\PackageSeeder
```

## Usage

### Basic Content Generation

```php
use Bongo\OpenAI\Services\ContentAI;

$content = (new ContentAI)
    ->setInstruction('Rewrite to be standard English and plagiarism free')
    ->setPrompt('Yor orignal content here with typos')
    ->generate();

if ($content !== null) {
    echo $content;  // Returns formatted content with nl2br() applied
} else {
    // Handle API error or invalid response
    Log::error('OpenAI API failed to generate content');
}
```

### Advanced Configuration

```php
use Bongo\OpenAI\Services\ContentAI;

$content = (new ContentAI)
    ->setInstruction('Summarise the following article in 3 bullet points')
    ->setPrompt($longArticle)
    ->setModel('gpt-4.1-mini')
    ->setTemperature(0.7)          // Lower = more deterministic
    ->setMaxTokens(200)
    ->setTopP(1)
    ->setFrequencyPenalty(0.5)
    ->setPresencePenalty(0.5)
    ->generate();
```

### Using Config Defaults

All setter methods accept `null` to use config defaults:

```php
$content = (new ContentAI)
    ->setInstruction('Translate to French')
    ->setPrompt('Hello world')
    ->setModel()           // Uses config('openai.model')
    ->setTemperature()     // Uses config('openai.temperature')
    ->generate();
```

## Available Services

### ContentAI

Chat-based content generation using OpenAI's Chat API.

**Methods:**
- `setInstruction(string $instruction): self` - Set system role/instruction
- `setPrompt(string $prompt): self` - Set user message/content to process
- `setModel(?string $model = null): self` - Set GPT model (default: config)
- `setTemperature(?int $temperature = null): self` - Set temperature 0-2 (default: config)
- `setMaxTokens(?int $maxTokens = null): self` - Set max response tokens (default: config)
- `setTopP(?int $topP = null): self` - Set nucleus sampling (default: config)
- `setFrequencyPenalty(?int $frequencyPenalty = null): self` - Set frequency penalty (default: config)
- `setPresencePenalty(?int $presencePenalty = null): self` - Set presence penalty (default: config)
- `generate(): ?string` - Execute API call and return generated content or null

**Returns:** `string|null` - Generated content with newlines converted to `<br>` tags, or `null` on error

**Throws:** `InvalidArgumentException` if instruction or prompt is empty

## Configuration Reference

All configuration parameters are in `config/openai.php`:

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `api_key` | string\|null | null | OpenAI API key (required) |
| `organization` | string\|null | null | OpenAI organization ID |
| `project` | string\|null | null | OpenAI project ID |
| `base_uri` | string\|null | null | Custom API base URL |
| `request_timeout` | int | 30 | Request timeout in seconds |
| `model` | string | 'gpt-4.1-mini' | Default GPT model |
| `max_tokens` | int | 500 | Maximum tokens in response |
| `temperature` | int | 1 | Sampling temperature (0=deterministic, 2=very random) |
| `top_p` | int | 1 | Nucleus sampling parameter |
| `frequency_penalty` | int | 0 | Reduce repetition of token sequences (0-2) |
| `presence_penalty` | int | 0 | Reduce repetition of topics (0-2) |

## Extending the Package

### Create a New AI Service

Extend `AbstractAI` to create new service types:

```php
namespace Bongo\OpenAI\Services;

use OpenAI\Laravel\Facades\OpenAI;

class ImageAI extends AbstractAI
{
    protected string $prompt = '';
    protected string $size = '1024x1024';

    public function setPrompt(string $prompt): self
    {
        $this->prompt = $prompt;
        return $this;
    }

    public function setSize(string $size): self
    {
        $this->size = $size;
        return $this;
    }

    public function generate(): ?string
    {
        $response = OpenAI::images()->create([
            'model' => $this->model ?? 'dall-e-3',
            'prompt' => $this->prompt,
            'size' => $this->size,
        ]);

        return $response->data[0]->url ?? null;
    }
}
```

### Settings Integration

The service provider automatically checks the Bongo settings system if the environment variable is not set:

```php
// In your settings system, implement:
public function getOpenAIApiKey(): ?string
{
    return $this->get('openai_api_key');
}
```

Priority order for API key:
1. Environment variable (`OPENAI_API_KEY`)
2. Bongo settings system (`setting()->getOpenAIApiKey()`)
3. None (will fail)

## Testing

Run the test suite:

```bash
# All tests
composer test

# Parallel execution
composer test:parallel

# With coverage report
composer test:coverage
```

### Writing Tests

Extend the base test case:

```php
namespace Bongo\OpenAI\Tests;

class MyTest extends TestCase
{
    public function test_example(): void
    {
        $content = (new ContentAI)
            ->setInstruction('Test instruction')
            ->setPrompt('Test prompt')
            ->generate();

        $this->assertNotNull($content);
    }
}
```

## Code Quality

Format code with Laravel Pint:

```bash
composer format
```

Run static analysis with PHPStan:

```bash
composer analyse
```

## Architecture

See detailed documentation:

- **ARCHITECTURE.md** - Comprehensive architecture, class diagrams, flows, and extension points
- **CLAUDE.md** - Quick reference guide for AI assistants
- **.cursorrules** - Cursor AI coding conventions and common tasks
- **.github/copilot-instructions.md** - GitHub Copilot code templates

### Quick Architecture Overview

```
OpenAIServiceProvider (extends AbstractServiceProvider)
    │
    ├── Auto-loads: src/Config/openai.php
    ├── Boot hook: Sets API key from settings if env not provided
    │
    └── Enables:
         │
         AbstractAI (abstract base class)
         │   ├── Properties: Model parameters (7 fluent setters)
         │   └── Abstract: generate(): ?string
         │
         └── ContentAI (concrete implementation)
              ├── Properties: $instruction, $prompt
              └── Implements: generate() using OpenAI::chat()
```

## Dependencies

**Production:**
- `openai-php/laravel` ^0.11.0 - Official OpenAI PHP SDK
- `bongo/framework` ^3.0 - Bongo AbstractServiceProvider
- `illuminate/contracts` ^10.0||^11.0 - Laravel contracts

**Development:**
- `laravel/pint` ^1.14 - Code formatting
- `larastan/larastan` ^2.9 - Static analysis
- `orchestra/testbench` ^9.0.0||^8.22.0 - Package testing

## Roadmap

- [ ] Complete test suite (ContentAI, AbstractAI)
- [ ] Add ImageAI service for DALL-E integration
- [ ] Add EmbeddingAI service for embeddings
- [ ] Add migrations for request/response storage
- [ ] Add backend UI for API key management
- [ ] Add streaming support for real-time responses
- [ ] Add usage tracking and logging
- [ ] Add rate limiting

## Contributing

This package is part of the Bongo monorepo. Each package is a separate git repository.

1. Make changes in `/Users/stuart/Packages/bongo/cms/openai`
2. Run code quality checks: `composer format && composer analyse && composer test`
3. Commit: `git add . && git commit -m "Your message"`
4. Push to Bitbucket

See `/Users/stuart/Packages/Bongo/CLAUDE.md` for monorepo batch operation scripts.

## License

MIT License. See [LICENSE](LICENSE) for details.

## Credits

- **Author**: Stuart Elliott
- **OpenAI PHP SDK**: [openai-php/laravel](https://github.com/openai-php/laravel)
- **Bongo Framework**: Internal Bongo package system

## Support

For issues, questions, or feedback:
- Internal Bongo documentation
- Bitbucket repository issues
- Contact: Stuart Elliott
