# Claude Code Guide - bongo/package

## Overview

Laravel package for managing packages within the Bongo framework. Provides a database-driven registry for tracking package status, type, visibility, and dynamic loading with permanent caching.

**Documentation:**
- Architecture details: [ARCHITECTURE.md](ARCHITECTURE.md)
- Cursor AI rules: [.cursorrules](.cursorrules)
- Copilot instructions: [.github/copilot-instructions.md](.github/copilot-instructions.md)

## Quick Reference

### Commands

From `composer.json`:

```bash
# Run tests
vendor/bin/phpunit

# Code style checking
vendor/bin/pint --test

# Code style fixing
vendor/bin/pint

# Static analysis
vendor/bin/phpstan analyse
```

### Architecture Overview

**Core Components:**
- **PackageServiceProvider** - Registers PackageManager singleton and helper functions
- **Package Model** - Represents package records with status/type/visibility
- **PackageManager Service** - Cached singleton for high-performance package queries
- **HasType Trait** - Adds type-based query scopes (standard/extended/custom)
- **SeedsPackage Trait** - Helper for seeding package records

**Framework Integration:**
- Extends `Bongo\Framework\Providers\AbstractServiceProvider`
- Module name: `package` (no config file needed)
- Uses framework traits: HasKey, HasStatus, HasUUID, HasVisible
- Helper function: `package()` returns PackageManager instance

**Package Types:**
- **standard** - Vendor packages in `vendor/bongo/*`
- **extended** - Extended framework packages
- **custom** - Application packages in `app/*`

**Status States:**
- **pending** - Registered but not activated
- **active** - Enabled and loaded
- **inactive** - Disabled

## Key Files

| File | Purpose |
|------|---------|
| `src/PackageServiceProvider.php` | Service provider, registers PackageManager singleton |
| `src/Models/Package.php` | Package model with status/type/visibility, computed attributes |
| `src/Services/PackageManager.php` | Cached singleton service for package queries |
| `src/Traits/HasType.php` | Type-based query scopes (standard/extended/custom) |
| `src/Traits/SeedsPackage.php` | Helper trait for seeding package records |
| `src/helpers.php` | Global `package()` helper function |
| `src/Migrations/2018_01_30_000001_create_packages_table.php` | Creates packages table |

## Code Examples

### Using PackageManager

```php
// Check if package exists
if (package()->has('framework')) {
    // Package exists
}

// Check if package is enabled
if (package()->isEnabled('cms-page')) {
    // Package is active
}

// Get specific package
$package = package()->get('asset');
$url = $package->getUrl();

// Get all active visible packages
foreach (package()->allActive() as $package) {
    echo $package->name;
}
```

### Querying by Type

```php
// Get standard vendor packages
$standardPackages = Package::standard()->get();

// Get custom application packages
$customPackages = Package::custom()->get();

// Combined queries
$activeCustom = Package::custom()
    ->where('status', Package::ACTIVE)
    ->get();
```

### Seeding Package Data

```php
use Bongo\Package\Traits\SeedsPackage;

class PackageSeeder extends Seeder
{
    use SeedsPackage;

    public function run()
    {
        $this->package([
            'name' => 'framework',
            'route' => 'backend.framework',
            'icon' => 'fa-cog',
            'status' => Package::ACTIVE,
            'type' => Package::STANDARD,
            'is_visible' => Package::VISIBLE,
        ]);
    }
}
```

### Package Model Constants

```php
// Status
Package::PENDING
Package::ACTIVE
Package::INACTIVE

// Type
Package::STANDARD   // vendor/bongo/*
Package::EXTENDED
Package::CUSTOM     // app/*

// Visibility
Package::VISIBLE    // 1
Package::HIDDEN     // 0

// Directory constants
Package::VENDOR_NAME    // 'bongo'
Package::VENDOR_FOLDER  // 'vendor'
```

### Computed Attributes

```php
$package = Package::find(1);

// Directory path based on type
$directory = $package->directory;
// standard: base_path('vendor/bongo/')
// custom: app_path()

// Capitalized package name
$namespace = $package->namespace;  // "Framework"

// Service provider class name
$provider = $package->service_provider;  // "FrameworkServiceProvider"

// Full path to service provider
$path = $package->service_provider_file_path;

// Dynamic URL generation
$url = $package->getUrl();  // Returns route URL or '#'
```

## Code Style Summary

**PHP Standards:**
- PHP 8.2+ required
- No strict types declaration in this package
- Type hints on all public methods
- Null coalescing for defaults: `$params['route'] ?? null`

**Laravel Conventions:**
- Laravel 10+ required
- Models extend `AbstractModel` not Eloquent directly
- Services registered as singletons
- Permanent caching with `Cache::rememberForever()`

**Naming:**
- Models: Singular names (Package)
- Services: Manager suffix (PackageManager)
- Traits: Descriptive capabilities (HasType, SeedsPackage)
- Constants: UPPER_SNAKE_CASE
- Methods: camelCase with return types

## Important Notes

### Cache Management

PackageManager permanently caches all packages on construction. After database changes:

```php
// Clear cache manually
Cache::forget('packages');

// Next access rebuilds cache
package()->all();
```

### No Configuration File

This package does not include `src/Config/package.php`. All settings are stored in the database, not configuration files.

### Package Directory Resolution

```php
// Standard packages
base_path('vendor/bongo/{name}')

// Extended/Custom packages
app_path()

// Service provider path
{directory}{namespace}/{ServiceProvider}.php
```

### Route Checking

Package model uses `route_exists()` helper to check if routes are registered:

```php
if ($this->route && route_exists("{$this->route}.index")) {
    return url()->route("{$this->route}.index");
}
```

## Testing

Tests extend `Orchestra\Testbench\TestCase`:

```php
namespace Bongo\Package\Tests;

use Bongo\Package\PackageServiceProvider;
use Orchestra\Testbench\TestCase as Orchestra;

class TestCase extends Orchestra
{
    protected function getPackageProviders($app): array
    {
        return [PackageServiceProvider::class];
    }
}
```

Configuration:
- Uses SQLite in-memory database
- DB_CONNECTION=testing in phpunit.xml
- Test directory: `tests/`

## Common Tasks

### Add New Package Type

1. Add constant to Package model
2. Update migration enum for `type` column
3. Add scope methods to HasType trait
4. Update seeders if needed

### Add Package Observer

```php
class PackageObserver
{
    public function saved(Package $package)
    {
        Cache::forget('packages');
    }
}

// Register in service provider
Package::observe(PackageObserver::class);
```

### Extend PackageManager

```php
class ExtendedPackageManager extends PackageManager
{
    public function getCmsPackages(): Collection
    {
        return $this->items->filter(fn($p) =>
            str_starts_with($p->name, 'cms-')
        );
    }
}

// Register in service provider
$this->app->singleton('package_manager', ExtendedPackageManager::class);
```

## Dependencies

**Required:**
- PHP ≥8.2
- Laravel/Illuminate ^10.0
- bongo/framework ^3.0

**Development:**
- laravel/pint ^1.0
- nunomaduro/collision ^7.9
- nunomaduro/larastan ^2.0
- orchestra/testbench ^8.0
- phpunit/phpunit ^10.0

## Framework Trait Usage

Package model uses these traits from `bongo/framework`:

| Trait | Provides |
|-------|----------|
| HasKey | Unique key attribute, scopes |
| HasStatus | Status management, `isActive()` |
| HasUUID | UUID generation, attribute |
| HasVisible | Visibility flag, `isVisible()` |

Plus package-specific trait:

| Trait | Provides |
|-------|----------|
| HasType | Type scopes, `isStandard()`, `isCustom()` |
