# bongo/package

Provides package management functionality for the Bongo framework. Enables tracking, loading, and managing packages within the system through a database-driven registry.

## Features

- **Database-driven package registry** - Track installed packages with metadata
- **Status management** - Enable/disable packages dynamically (pending/active/inactive)
- **Type classification** - Differentiate between vendor, extended, and custom packages
- **Visibility control** - Show/hide packages in admin interfaces
- **High-performance caching** - Permanent caching for fast package lookups
- **Dynamic URL generation** - Generate package URLs from route configuration
- **Helper functions** - Convenient `package()` helper for package queries

## Requirements

- PHP 8.2 or higher
- Laravel 10 or higher
- Composer
- bongo/framework ^3.0

## Installation

### Composer

```bash
composer require bongo/package
```

### Laravel Auto-Discovery

Laravel will automatically discover and register the `PackageServiceProvider`.

For Laravel versions without auto-discovery, add to `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\Package\PackageServiceProvider::class,
],
```

### Run Migrations

```bash
php artisan migrate
```

This creates the `packages` table.

## Usage

### Package Manager Helper

The `package()` helper function provides access to the PackageManager singleton:

```php
// Check if package exists
if (package()->has('framework')) {
    echo "Framework package exists";
}

// Check if package is enabled
if (package()->isEnabled('cms-page')) {
    echo "CMS Page package is active";
}

// Get specific package
$package = package()->get('asset');
echo $package->name;
echo $package->getUrl();

// Get all active visible packages
foreach (package()->allActive() as $package) {
    echo $package->name;
}
```

### Package Model

The `Package` model represents package records:

```php
use Bongo\Package\Models\Package;

// Query packages
$packages = Package::all();

// Query by status
$active = Package::where('status', Package::ACTIVE)->get();

// Query by type
$standardPackages = Package::standard()->get();
$customPackages = Package::custom()->get();

// Query by visibility
$visible = Package::visible()->get();

// Get package URL
$package = Package::find(1);
$url = $package->getUrl(); // Returns route URL or '#'
```

### Package Types

Packages are classified into three types:

- **standard** - Vendor packages in `vendor/bongo/*` directory
- **extended** - Extended framework packages
- **custom** - Application-specific packages in `app/*` directory

```php
// Check package type
if ($package->isStandard()) {
    echo "This is a standard vendor package";
}

if ($package->isCustom()) {
    echo "This is a custom application package";
}

// Query by type
Package::standard()->get();
Package::extended()->get();
Package::custom()->get();
```

### Package Status

Packages have three status states:

- **pending** - Registered but not activated
- **active** - Enabled and should be loaded
- **inactive** - Disabled

```php
// Update package status
$package->status = Package::ACTIVE;
$package->save();

// Query by status
Package::active()->get();
Package::inactive()->get();
```

### Seeding Packages

Use the `SeedsPackage` trait in your seeders:

```php
use Bongo\Package\Traits\SeedsPackage;
use Bongo\Package\Models\Package;
use Illuminate\Database\Seeder;

class PackageSeeder extends Seeder
{
    use SeedsPackage;

    public function run()
    {
        // Create or update package
        $this->package([
            'name' => 'framework',
            'route' => 'backend.framework',
            'icon' => 'fa-cog',
            'status' => Package::ACTIVE,
            'type' => Package::STANDARD,
            'is_visible' => Package::VISIBLE,
        ]);

        $this->package([
            'name' => 'cms-page',
            'route' => 'backend.cms.page',
            'icon' => 'fa-file',
            'status' => Package::ACTIVE,
            'type' => Package::STANDARD,
            'is_visible' => Package::VISIBLE,
        ]);
    }
}
```

The `package()` method uses `firstOrNew()` to prevent duplicates, so it's safe to run multiple times.

### Computed Attributes

Package model provides several computed attributes:

```php
$package = Package::first();

// Directory path (based on type)
echo $package->directory;
// standard: /path/to/vendor/bongo/
// custom: /path/to/app/

// Capitalized package name
echo $package->namespace;  // "Framework"

// Service provider class name
echo $package->service_provider;  // "FrameworkServiceProvider"

// Full path to service provider file
echo $package->service_provider_file_path;
```

## PackageManager API

The `PackageManager` service provides cached package queries:

| Method | Return Type | Description |
|--------|-------------|-------------|
| `has(string $key)` | `bool` | Check if package exists by key |
| `get(string $key)` | `?Package` | Get package by key |
| `all()` | `Collection` | Get all packages |
| `allActive()` | `array` | Get active + visible packages |
| `isEnabled(string $key)` | `bool` | Check if package is active |

Access via helper function:

```php
package()->has('framework');
package()->isEnabled('cms-page');
package()->get('asset');
package()->all();
package()->allActive();
```

Or via service container:

```php
$manager = app('package_manager');
$manager->isEnabled('cms-page');
```

## Caching

The PackageManager permanently caches all packages for performance. After making database changes to packages, clear the cache:

```php
use Illuminate\Support\Facades\Cache;

// Update package
$package = Package::find(1);
$package->status = Package::ACTIVE;
$package->save();

// Clear cache
Cache::forget('packages');

// Next access rebuilds cache
package()->all();
```

## Constants Reference

### Status Constants

```php
Package::PENDING   // 'pending'
Package::ACTIVE    // 'active'
Package::INACTIVE  // 'inactive'
```

### Type Constants

```php
Package::STANDARD  // 'standard' - vendor/bongo/* packages
Package::EXTENDED  // 'extended' - extended framework packages
Package::CUSTOM    // 'custom' - app/* packages
```

### Visibility Constants

```php
Package::VISIBLE   // 1
Package::HIDDEN    // 0
```

### Directory Constants

```php
Package::VENDOR_NAME    // 'bongo'
Package::VENDOR_FOLDER  // 'vendor'
```

## Testing

Run the test suite:

```bash
vendor/bin/phpunit
```

Tests use Orchestra Testbench with SQLite in-memory database.

## Code Quality

### Laravel Pint

```bash
# Check code style
vendor/bin/pint --test

# Fix code style issues
vendor/bin/pint
```

### Static Analysis

```bash
vendor/bin/phpstan analyse
```

## Database Schema

### packages Table

| Column | Type | Nullable | Default | Description |
|--------|------|----------|---------|-------------|
| `id` | integer | No | Auto | Primary key |
| `uuid` | varchar(36) | No | Auto | UUID identifier (indexed) |
| `name` | varchar(255) | No | - | Package name |
| `key` | varchar(255) | Yes | null | Unique key (indexed) |
| `route` | varchar(255) | Yes | null | Route prefix |
| `icon` | varchar(255) | Yes | null | Icon class |
| `status` | enum | No | pending | Status: pending, active, inactive |
| `type` | enum | No | standard | Type: standard, extended, custom |
| `is_visible` | tinyint | No | 1 | Visibility: 1 (visible), 0 (hidden) |
| `created_at` | timestamp | Yes | null | Creation timestamp |
| `updated_at` | timestamp | Yes | null | Update timestamp |
| `deleted_at` | timestamp | Yes | null | Soft delete timestamp |

## Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Detailed architecture documentation with diagrams and extension examples
- **[CLAUDE.md](CLAUDE.md)** - Claude Code guidance with quick reference
- **[.cursorrules](.cursorrules)** - Cursor AI instructions with coding conventions
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot code patterns

## Contributing

This package follows Laravel and Bongo framework coding standards. Please ensure:

- PHP 8.2+ compatibility
- Type hints on all public methods
- Laravel Pint passes (`vendor/bin/pint`)
- PHPStan analysis passes
- Tests pass (`vendor/bin/phpunit`)

## License

MIT License. See [LICENSE](LICENSE) for details.

## Author

**Stuart Elliott**
- Email: stuart.elliott@bespokeuk.com
- Homepage: https://bespokeuk.com

## Links

- Repository: https://bitbucket.org/designtec/package
- Bongo Framework: https://designtecpackages.co.uk
