# GitHub Copilot Instructions for bongo/package

## Project Overview
Laravel package for managing packages within the Bongo framework. Provides database-driven registry for tracking package status, type, visibility, and dynamic loading.

## Key Classes and Relationships

### Core Classes
- **PackageServiceProvider** (`src/PackageServiceProvider.php`)
  - Extends `Bongo\Framework\Providers\AbstractServiceProvider`
  - Registers `PackageManager` singleton
  - Loads helper functions after boot

- **Package Model** (`src/Models/Package.php`)
  - Extends `Bongo\Framework\Models\AbstractModel`
  - Traits: HasKey, HasStatus, HasUUID, HasVisible, HasType
  - Manages package registry with status/type/visibility

- **PackageManager Service** (`src/Services/PackageManager.php`)
  - Singleton service for package operations
  - Permanently caches all packages
  - Provides query methods: has(), get(), all(), allActive(), isEnabled()

### Traits
- **HasType** (`src/Traits/HasType.php`)
  - Adds type scopes and checkers
  - Types: standard (vendor/bongo), extended, custom (app/)

- **SeedsPackage** (`src/Traits/SeedsPackage.php`)
  - Helper for seeding package records
  - Method: `package(array $params): Package`

## Code Style Templates

### Service Provider Pattern
```php
namespace Bongo\Package;

use Bongo\Framework\Providers\AbstractServiceProvider;
use Bongo\Package\Services\PackageManager;

class PackageServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'package';

    public function register(): void
    {
        parent::register();
        $this->app->singleton('package_manager', PackageManager::class);
    }

    public function boot(): void
    {
        parent::boot();
        $this->app->booted(function () {
            include __DIR__.'/helpers.php';
        });
    }
}
```

### Model with Multiple Traits
```php
namespace Bongo\Package\Models;

use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasKey;
use Bongo\Framework\Traits\HasStatus;
use Bongo\Framework\Traits\HasUUID;
use Bongo\Framework\Traits\HasVisible;
use Bongo\Package\Traits\HasType;

class Package extends AbstractModel
{
    use HasKey, HasStatus, HasType, HasUUID, HasVisible;

    protected $fillable = ['name', 'route', 'icon', 'status', 'type', 'is_visible'];
    protected $table = 'packages';
}
```

### Singleton Service with Caching
```php
namespace Bongo\Package\Services;

use Bongo\Package\Models\Package;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;

class PackageManager
{
    protected Collection $items;

    public function __construct()
    {
        $this->items = Cache::rememberForever("packages", function () {
            return Package::orderBy('key')->get();
        });
    }

    public function has(string $key): bool
    {
        return $this->items->contains('key', $key);
    }

    public function isEnabled(string $key): bool
    {
        return $this->items
            ->where('status', Package::ACTIVE)
            ->contains('key', $key);
    }
}
```

### Type-Based Query Scope
```php
trait HasType
{
    public function scopeStandard($query)
    {
        return $query->where('type', self::STANDARD);
    }

    public function isStandard(): bool
    {
        return $this->type === self::STANDARD;
    }
}
```

### Seeder Helper Trait
```php
trait SeedsPackage
{
    protected function package(array $params): Package
    {
        $package = Package::firstOrNew(['name' => $params['name']]);
        $package->route = $params['route'] ?? null;
        $package->icon = $params['icon'] ?? null;
        $package->status = $params['status'] ?? Package::ACTIVE;
        $package->type = $params['type'] ?? Package::STANDARD;
        $package->is_visible = $params['is_visible'] ?? Package::VISIBLE;
        $package->save();

        return $package;
    }
}
```

### Computed Model Attributes
```php
public function getDirectoryAttribute(): string
{
    if ($this->isStandard()) {
        return base_path().DIRECTORY_SEPARATOR
            .self::VENDOR_FOLDER.DIRECTORY_SEPARATOR
            .self::VENDOR_NAME.DIRECTORY_SEPARATOR;
    }
    return app_path().DIRECTORY_SEPARATOR;
}

public function getUrl(): string
{
    if ($this->route && route_exists("{$this->route}.index")) {
        return url()->route("{$this->route}.index");
    }
    return '#';
}
```

## Common Patterns

### Using PackageManager Helper
```php
// Check if package exists
if (package()->has('framework')) {
    // Package exists
}

// Check if package is enabled
if (package()->isEnabled('cms-page')) {
    // Package is active
}

// Get specific package
$package = package()->get('asset');
$url = $package->getUrl();

// Get all active visible packages
foreach (package()->allActive() as $package) {
    echo $package->name;
}
```

### Querying by Type
```php
// Get standard vendor packages
$standardPackages = Package::standard()->get();

// Get non-standard packages
$customPackages = Package::notStandard()->get();

// Combined queries
$activeCustom = Package::custom()
    ->where('status', Package::ACTIVE)
    ->get();
```

### Database Constants
```php
// Status constants
Package::PENDING
Package::ACTIVE
Package::INACTIVE

// Type constants
Package::STANDARD   // vendor/bongo/*
Package::EXTENDED
Package::CUSTOM     // app/*

// Visibility constants
Package::VISIBLE    // 1
Package::HIDDEN     // 0
```

### Cache Invalidation
```php
// After updating packages, clear cache
Cache::forget('packages');

// PackageManager will rebuild cache on next instantiation
$manager = app('package_manager');
```

## Testing Patterns

### Basic Test Case Setup
```php
namespace Bongo\Package\Tests;

use Bongo\Package\PackageServiceProvider;
use Orchestra\Testbench\TestCase as Orchestra;

class TestCase extends Orchestra
{
    protected function getPackageProviders($app): array
    {
        return [PackageServiceProvider::class];
    }
}
```

## Key Conventions
- Models extend `AbstractModel`, not Eloquent Model directly
- Services registered as singletons in service provider
- Helper functions loaded via `app()->booted()` callback
- Permanent caching for frequently accessed data
- Type hints on all public methods
- Null coalescing for default values
- Strict equality comparisons for constants
