# Architecture Documentation - bongo/package

## Table of Contents
1. [Overview](#overview)
2. [Directory Structure](#directory-structure)
3. [Class Diagram](#class-diagram)
4. [Component Architecture](#component-architecture)
5. [Data Flow](#data-flow)
6. [Traits and Interfaces](#traits-and-interfaces)
7. [Database Schema](#database-schema)
8. [Extension Points](#extension-points)
9. [How to Add Features](#how-to-add-features)

## Overview

The `bongo/package` package provides a database-driven package registry for the Bongo framework. It enables the system to:

- Track installed packages with metadata (name, route, icon, status, type, visibility)
- Determine which packages are active and should be loaded
- Provide a cached, high-performance package lookup service
- Differentiate between vendor packages (standard), extended packages, and custom application packages
- Generate package URLs dynamically based on route configuration

**Key Design Principles:**
- Single source of truth in database
- Permanent caching for performance
- Type-safe package classification
- Framework trait composition for common functionality

## Directory Structure

```
src/
├── PackageServiceProvider.php           # Main service provider, registers PackageManager
├── Models/
│   └── Package.php                      # Package model with status/type/visibility management
├── Services/
│   └── PackageManager.php               # Singleton service for package queries with caching
├── Traits/
│   ├── HasType.php                      # Adds type-based query scopes and checker methods
│   └── SeedsPackage.php                 # Helper trait for seeding package records in seeders
├── Migrations/
│   └── 2018_01_30_000001_create_packages_table.php  # Creates packages table
└── helpers.php                          # Global package() helper function

tests/
└── TestCase.php                         # Orchestra Testbench base test case

composer.json                            # Package dependencies (bongo/framework ^3.0)
phpunit.xml                              # PHPUnit configuration with SQLite testing
README.md                                # Basic package documentation
```

## Class Diagram

```
┌─────────────────────────────────────┐
│   PackageServiceProvider            │
│   (AbstractServiceProvider)         │
├─────────────────────────────────────┤
│ - string $module = 'package'        │
├─────────────────────────────────────┤
│ + register(): void                  │
│ + boot(): void                      │
└──────────────┬──────────────────────┘
               │ registers
               ▼
┌─────────────────────────────────────┐
│      PackageManager                 │
│      (Singleton Service)            │
├─────────────────────────────────────┤
│ - Collection $items                 │
├─────────────────────────────────────┤
│ + __construct()                     │
│ + has(string $key): bool            │
│ + get(string $key): ?Package        │
│ + all(): Collection                 │
│ + allActive(): array                │
│ + isEnabled(string $key): bool      │
└──────────────┬──────────────────────┘
               │ queries
               ▼
┌─────────────────────────────────────┐
│         Package Model               │
│      (AbstractModel)                │
├─────────────────────────────────────┤
│ # string $table = 'packages'        │
│ # array $fillable                   │
├─────────────────────────────────────┤
│ Constants:                          │
│   VENDOR_NAME = 'bongo'             │
│   VENDOR_FOLDER = 'vendor'          │
│   PENDING/ACTIVE/INACTIVE           │
│   VISIBLE/HIDDEN                    │
│   STANDARD/EXTENDED/CUSTOM          │
├─────────────────────────────────────┤
│ Traits:                             │
│   - HasKey                          │
│   - HasStatus                       │
│   - HasType                         │
│   - HasUUID                         │
│   - HasVisible                      │
├─────────────────────────────────────┤
│ + getDirectoryAttribute(): string   │
│ + getNamespaceAttribute(): string   │
│ + getServiceProviderAttribute()     │
│ + getServiceProviderFilePath()      │
│ + getUrl(): string                  │
└─────────────────────────────────────┘

┌─────────────────────────────────────┐
│          HasType Trait              │
├─────────────────────────────────────┤
│ + scopeStandard($query)             │
│ + scopeNotStandard($query)          │
│ + scopeExtended($query)             │
│ + scopeNotExtended($query)          │
│ + scopeCustom($query)               │
│ + scopeNotCustom($query)            │
│ + isStandard(): bool                │
│ + isExtended(): bool                │
│ + isCustom(): bool                  │
└─────────────────────────────────────┘

┌─────────────────────────────────────┐
│       SeedsPackage Trait            │
├─────────────────────────────────────┤
│ # package(array $params): Package   │
└─────────────────────────────────────┘
```

## Component Architecture

### 1. Service Provider (PackageServiceProvider)

**Purpose:** Bootstrap package functionality and register services.

**Responsibilities:**
- Extend `AbstractServiceProvider` for automatic migration/view/translation loading
- Register `PackageManager` as singleton bound to `package_manager`
- Load helper functions after application boot

**Code Flow:**
```
register() → parent::register() → app->singleton('package_manager')
boot() → parent::boot() → app->booted(callback) → include helpers.php
```

**Key Methods:**
```php
public function register(): void
{
    parent::register();
    $this->app->singleton('package_manager', PackageManager::class);
}

public function boot(): void
{
    parent::boot();
    $this->app->booted(function () {
        include __DIR__.'/helpers.php';
    });
}
```

### 2. Package Model (Package)

**Purpose:** Represent package records with rich metadata and computed attributes.

**Responsibilities:**
- Store package configuration (name, route, icon, status, type, visibility)
- Provide type classification (standard/extended/custom)
- Generate computed attributes (directory, namespace, service provider path)
- Generate dynamic URLs from route configuration

**Package Types:**
- **Standard:** Vendor packages in `vendor/bongo/*` directory
- **Extended:** Extended framework packages
- **Custom:** Application-specific packages in `app/*` directory

**Status States:**
- **Pending:** Package registered but not activated
- **Active:** Package is enabled and should be loaded
- **Inactive:** Package is disabled

**Computed Attributes:**

| Attribute | Description | Example |
|-----------|-------------|---------|
| `directory` | Base directory path | `base_path('vendor/bongo/')` or `app_path()` |
| `namespace` | Capitalized package name | `Framework` |
| `service_provider` | Service provider class name | `FrameworkServiceProvider` |
| `service_provider_file_path` | Full path to provider | `vendor/bongo/Framework/FrameworkServiceProvider.php` |

**URL Generation:**
```php
public function getUrl(): string
{
    if ($this->route && route_exists("{$this->route}.index")) {
        return url()->route("{$this->route}.index");
    }
    return '#';
}
```

### 3. PackageManager Service

**Purpose:** High-performance singleton service for package queries.

**Responsibilities:**
- Cache all packages permanently on instantiation
- Provide query methods for package lookup
- Filter active/visible packages
- Check package existence and status

**Caching Strategy:**
- Uses `Cache::rememberForever('packages')` to store entire collection
- Loads once per application lifecycle
- Requires manual cache clearing after database changes

**Query Methods:**

| Method | Return Type | Description |
|--------|-------------|-------------|
| `has(string $key)` | `bool` | Check if package exists |
| `get(string $key)` | `?Package` | Get package by key |
| `all()` | `Collection` | Get all packages |
| `allActive()` | `array` | Get active + visible packages |
| `isEnabled(string $key)` | `bool` | Check if package is active |

**Usage via Helper:**
```php
package()->has('framework');         // Check existence
package()->isEnabled('cms-page');    // Check if active
package()->get('asset');             // Get specific package
package()->allActive();              // Get all active packages
```

### 4. HasType Trait

**Purpose:** Add type-based query scopes and checker methods to Package model.

**Responsibilities:**
- Provide query scopes for filtering by type
- Provide boolean checker methods for type detection

**Query Scopes:**
```php
Package::standard()->get();        // WHERE type = 'standard'
Package::notStandard()->get();     // WHERE type != 'standard'
Package::extended()->get();        // WHERE type = 'extended'
Package::custom()->get();          // WHERE type = 'custom'
```

**Checker Methods:**
```php
$package->isStandard();  // bool
$package->isExtended();  // bool
$package->isCustom();    // bool
```

### 5. SeedsPackage Trait

**Purpose:** Simplify package record creation in database seeders.

**Usage:**
```php
use Bongo\Package\Traits\SeedsPackage;

class PackageSeeder extends Seeder
{
    use SeedsPackage;

    public function run()
    {
        $this->package([
            'name' => 'framework',
            'route' => 'backend.framework',
            'icon' => 'fa-cog',
            'status' => Package::ACTIVE,
            'type' => Package::STANDARD,
            'is_visible' => Package::VISIBLE,
        ]);
    }
}
```

**Behaviour:**
- Uses `firstOrNew(['name' => $name])` to prevent duplicates
- Applies defaults for optional parameters
- Saves record to database
- Returns Package instance

## Data Flow

### Package Query Flow

```
User Code
   │
   ├─ package()->has('framework')
   │     │
   │     └─ app('package_manager')
   │           │
   │           └─ PackageManager::has()
   │                 │
   │                 └─ $this->items->contains('key', $key)
   │                       │
   │                       └─ Return bool
   │
   ├─ package()->isEnabled('cms-page')
   │     │
   │     └─ PackageManager::isEnabled()
   │           │
   │           └─ $this->items->where('status', ACTIVE)->contains()
   │                 │
   │                 └─ Return bool
   │
   └─ package()->allActive()
         │
         └─ PackageManager::allActive()
               │
               └─ $this->items->where('status', ACTIVE)
                                ->where('is_visible', VISIBLE)
                     │
                     └─ Return array
```

### Package Registration Flow

```
Database Seeder
   │
   └─ use SeedsPackage
         │
         └─ $this->package([...])
               │
               ├─ Package::firstOrNew(['name' => $name])
               │     │
               │     └─ Query Database
               │
               ├─ Set attributes (route, icon, status, type, is_visible)
               │
               ├─ $package->save()
               │     │
               │     └─ INSERT or UPDATE packages table
               │
               └─ Return Package instance
```

### Cache Lifecycle

```
Application Boot
   │
   └─ PackageServiceProvider registered
         │
         └─ PackageManager instantiated (singleton)
               │
               ├─ __construct()
               │     │
               │     └─ Cache::rememberForever('packages', callback)
               │           │
               │           ├─ Check cache
               │           │     │
               │           │     ├─ [HIT] Return cached collection
               │           │     │
               │           │     └─ [MISS] Execute callback
               │           │           │
               │           │           └─ Package::orderBy('key')->get()
               │           │                 │
               │           │                 └─ Query Database
               │           │
               │           └─ Store in cache permanently
               │
               └─ Subsequent queries use cached collection
```

**Cache Invalidation:**
```php
// Manual cache clearing required after database changes
Cache::forget('packages');

// Next access rebuilds cache
package()->all(); // Triggers new database query
```

## Traits and Interfaces

### Framework Traits Used by Package Model

| Trait | Source | Purpose | Methods Added |
|-------|--------|---------|---------------|
| `HasKey` | `bongo/framework` | Adds unique key identifier | `key` attribute, scopes |
| `HasStatus` | `bongo/framework` | Adds status management | Status scopes, `isActive()`, etc. |
| `HasUUID` | `bongo/framework` | Adds UUID identifier | `uuid` attribute, auto-generation |
| `HasVisible` | `bongo/framework` | Adds visibility flag | Visibility scopes, `isVisible()` |
| `HasType` | This package | Adds type classification | Type scopes, `isStandard()`, etc. |

### Trait Composition in Package Model

```php
class Package extends AbstractModel
{
    use HasKey;      // Provides: key attribute, scopeKey()
    use HasStatus;   // Provides: scopeActive(), scopeInactive(), isActive()
    use HasType;     // Provides: scopeStandard(), scopeCustom(), isStandard()
    use HasUUID;     // Provides: uuid attribute, automatic generation
    use HasVisible;  // Provides: scopeVisible(), scopeHidden(), isVisible()
}
```

### SeedsPackage Trait Usage

```php
trait SeedsPackage
{
    protected function package(array $params): Package
    {
        // firstOrNew prevents duplicate packages
        $package = Package::firstOrNew(['name' => $params['name']]);

        // Apply parameters with null coalescing defaults
        $package->route = $params['route'] ?? null;
        $package->icon = $params['icon'] ?? null;
        $package->status = $params['status'] ?? Package::ACTIVE;
        $package->type = $params['type'] ?? Package::STANDARD;
        $package->is_visible = $params['is_visible'] ?? Package::VISIBLE;

        $package->save();
        return $package;
    }
}
```

## Database Schema

### packages Table

```sql
CREATE TABLE packages (
    -- Primary Keys
    id INTEGER PRIMARY KEY AUTO_INCREMENT,
    uuid VARCHAR(36) NOT NULL,

    -- Package Identity
    name VARCHAR(255) NOT NULL,
    key VARCHAR(255) NULL,
    route VARCHAR(255) NULL,
    icon VARCHAR(255) NULL,

    -- Status & Classification
    status ENUM('pending', 'active', 'inactive') DEFAULT 'pending',
    type ENUM('standard', 'extended', 'custom') DEFAULT 'standard',
    is_visible TINYINT UNSIGNED DEFAULT 1,

    -- Timestamps
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    deleted_at TIMESTAMP NULL,

    -- Indexes
    INDEX idx_uuid (uuid),
    INDEX idx_key (key)
);
```

### Field Descriptions

| Field | Type | Nullable | Default | Description |
|-------|------|----------|---------|-------------|
| `id` | int | No | Auto | Primary key |
| `uuid` | varchar(36) | No | Auto | UUID identifier (indexed) |
| `name` | varchar(255) | No | - | Package name (e.g., "framework") |
| `key` | varchar(255) | Yes | null | Unique key identifier (indexed) |
| `route` | varchar(255) | Yes | null | Route prefix for package |
| `icon` | varchar(255) | Yes | null | Icon class (e.g., "fa-cog") |
| `status` | enum | No | pending | Status: pending, active, inactive |
| `type` | enum | No | standard | Type: standard, extended, custom |
| `is_visible` | tinyint | No | 1 | Visibility: 1 (visible), 0 (hidden) |
| `created_at` | timestamp | Yes | null | Creation timestamp |
| `updated_at` | timestamp | Yes | null | Last update timestamp |
| `deleted_at` | timestamp | Yes | null | Soft delete timestamp |

## Extension Points

### 1. Adding Custom Package Attributes

**Example: Add "version" field**

```php
// Step 1: Create migration
public function up()
{
    Schema::table('packages', function (Blueprint $table) {
        $table->string('version')->nullable()->after('name');
    });
}

// Step 2: Add to Package model fillable
protected $fillable = [
    'name',
    'version',  // New field
    'route',
    'icon',
    'status',
    'type',
    'is_visible',
];

// Step 3: Update SeedsPackage trait
protected function package(array $params): Package
{
    $package = Package::firstOrNew(['name' => $params['name']]);
    $package->version = $params['version'] ?? null;  // New field
    // ... rest of fields
    $package->save();
    return $package;
}
```

### 2. Adding New Package Types

**Example: Add "premium" type**

```php
// Step 1: Add constant to Package model
class Package extends AbstractModel
{
    public const PREMIUM = 'premium';
}

// Step 2: Create migration to update enum
public function up()
{
    DB::statement("ALTER TABLE packages MODIFY COLUMN type ENUM('standard', 'extended', 'custom', 'premium') DEFAULT 'standard'");
}

// Step 3: Add methods to HasType trait
trait HasType
{
    public function scopePremium($query)
    {
        return $query->where('type', self::PREMIUM);
    }

    public function isPremium(): bool
    {
        return $this->type === self::PREMIUM;
    }
}
```

### 3. Extending PackageManager with Custom Queries

```php
// Create custom manager extending PackageManager
namespace App\Services;

use Bongo\Package\Services\PackageManager;
use Bongo\Package\Models\Package;

class ExtendedPackageManager extends PackageManager
{
    public function getCmsPackages(): Collection
    {
        return $this->items->filter(function ($package) {
            return str_starts_with($package->name, 'cms-');
        });
    }

    public function getByType(string $type): Collection
    {
        return $this->items->where('type', $type);
    }
}

// Register in service provider
$this->app->singleton('package_manager', ExtendedPackageManager::class);
```

### 4. Adding Package Events

```php
// Add to Package model
protected $dispatchesEvents = [
    'created' => PackageCreated::class,
    'updated' => PackageUpdated::class,
];

// Clear cache on package changes
class PackageObserver
{
    public function saved(Package $package)
    {
        Cache::forget('packages');
    }

    public function deleted(Package $package)
    {
        Cache::forget('packages');
    }
}

// Register in service provider
Package::observe(PackageObserver::class);
```

## How to Add Features

### Adding a Package Dependency System

**Goal:** Track which packages depend on other packages.

**Steps:**

1. **Create migration for package_dependencies table:**
```php
Schema::create('package_dependencies', function (Blueprint $table) {
    $table->increments('id');
    $table->unsignedInteger('package_id');
    $table->unsignedInteger('depends_on_package_id');
    $table->timestamps();

    $table->foreign('package_id')
          ->references('id')->on('packages')
          ->onDelete('cascade');
    $table->foreign('depends_on_package_id')
          ->references('id')->on('packages')
          ->onDelete('cascade');
});
```

2. **Add relationship to Package model:**
```php
public function dependencies()
{
    return $this->belongsToMany(
        Package::class,
        'package_dependencies',
        'package_id',
        'depends_on_package_id'
    );
}

public function dependents()
{
    return $this->belongsToMany(
        Package::class,
        'package_dependencies',
        'depends_on_package_id',
        'package_id'
    );
}
```

3. **Add methods to PackageManager:**
```php
public function getDependencies(string $key): Collection
{
    return $this->get($key)?->dependencies ?? collect();
}

public function canDisable(string $key): bool
{
    $package = $this->get($key);
    return $package->dependents()->where('status', Package::ACTIVE)->count() === 0;
}
```

4. **Update seeder trait:**
```php
protected function package(array $params): Package
{
    $package = Package::firstOrNew(['name' => $params['name']]);
    // ... existing fields
    $package->save();

    if (isset($params['dependencies'])) {
        $dependencies = Package::whereIn('name', $params['dependencies'])->pluck('id');
        $package->dependencies()->sync($dependencies);
    }

    return $package;
}
```

### Adding Package Configuration Storage

**Goal:** Store package-specific configuration in JSON field.

**Steps:**

1. **Create migration:**
```php
Schema::table('packages', function (Blueprint $table) {
    $table->json('config')->nullable();
});
```

2. **Update Package model:**
```php
protected $casts = [
    'config' => 'array',
];

protected $fillable = [
    'name', 'route', 'icon', 'status', 'type', 'is_visible', 'config',
];

public function getConfig(string $key, $default = null)
{
    return data_get($this->config, $key, $default);
}

public function setConfig(string $key, $value): void
{
    $config = $this->config ?? [];
    data_set($config, $key, $value);
    $this->config = $config;
}
```

3. **Usage:**
```php
$package = package()->get('framework');
$package->setConfig('cache_enabled', true);
$package->setConfig('cache_ttl', 3600);
$package->save();

if ($package->getConfig('cache_enabled')) {
    // Use caching
}
```

### Adding Package Auto-Discovery

**Goal:** Automatically discover and register packages from composer.json.

**Implementation:**

```php
namespace Bongo\Package\Services;

use Bongo\Package\Models\Package;
use Illuminate\Support\Facades\File;

class PackageDiscovery
{
    public function discover(): array
    {
        $composerPath = base_path('composer.json');
        $composer = json_decode(File::get($composerPath), true);

        $packages = [];
        foreach ($composer['require'] ?? [] as $name => $version) {
            if (str_starts_with($name, 'bongo/')) {
                $packages[] = $this->registerPackage($name);
            }
        }

        return $packages;
    }

    protected function registerPackage(string $name): Package
    {
        $packageName = str_replace('bongo/', '', $name);

        return Package::firstOrCreate(
            ['name' => $packageName],
            [
                'status' => Package::PENDING,
                'type' => Package::STANDARD,
                'is_visible' => Package::VISIBLE,
            ]
        );
    }
}
```

**Register in service provider:**
```php
$this->app->singleton('package_discovery', PackageDiscovery::class);

// Optionally run on boot
if (app()->environment('local')) {
    app('package_discovery')->discover();
}
```
