# GitHub Copilot Instructions - Bongo Page Package

## Project Overview

This is a Laravel package (`bongo/page`) that provides web page management for the Bongo CMS. The package allows creating, managing, and displaying web pages through admin and public interfaces, with support for a visual builder, SEO metadata, image attachments, and custom CSS/JS/schema markup.

**Key Features:**
- CRUD operations for web pages via admin backend
- Public-facing page display with homepage support
- Visual page builder integration (frontend editor)
- Image management and uploads
- SEO metadata (title, description, canonical, index/noindex)
- Custom CSS, JS, and JSON-LD schema per page
- Menu association and navigation
- Page duplication functionality
- Package licensing limits enforcement
- Export to CSV functionality

## Key Classes and Relationships

### Core Model
```php
namespace Bongo\Page\Models;

class Page extends AbstractModel implements Imageable
{
    // Status constants
    const PENDING = 'pending';
    const ACTIVE = 'active';
    const INACTIVE = 'inactive';

    // Traits provide functionality
    use HasContent;        // content field handling
    use HasHeaderClass;    // header CSS classes
    use HasImages;         // polymorphic image relations
    use HasKey;            // unique key generation
    use HasSeo;            // SEO metadata methods
    use HasStatus;         // status management (isActive(), etc.)
    use HasUUID;           // UUID generation
    use SoftDeletes;       // soft deletion

    // Relations
    public function menu(): BelongsTo;  // belongs to Menu

    // Key methods
    public function hasMenu(): bool;
    public function isHomePage(): bool;
    public function duplicate(): self;  // creates a copy with new UUID/slug
    public function scopeNotHomePage($query);
}
```

### Service Provider
```php
namespace Bongo\Page;

class PageServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'page';

    // Auto-registered by AbstractServiceProvider
    protected array $commands = [ExportPagesCommand::class];
    protected array $composers = [...];
    protected array $listeners = [
        PageUpdated::class => [ClearMenuCache::class, UpdateSitemap::class],
        PageDeleted::class => [ClearMenuCache::class, UpdateSitemap::class],
    ];
}
```

### Controllers

**Backend Controllers** (Admin Panel):
```php
namespace Bongo\Page\Http\Controllers\Backend;

class PageController extends AbstractController
{
    public function index(): View;                              // List pages
    public function create(): View;                             // Show create form
    public function store(StorePageRequest $request): RedirectResponse;
    public function show(Page $page): View;                     // Page details
    public function edit(Page $page): View;                     // Edit form
    public function update(UpdatePageRequest $request, Page $page): RedirectResponse;
    public function destroy(Page $page): RedirectResponse;
    public function duplicate(Page $page): RedirectResponse;    // Duplicate page
}

class PageDatatableController extends AbstractController
{
    public function index(): JsonResponse;  // JSON for admin datatables
}

class PageImageController extends AbstractController
{
    public function upload(Request $request, Page $page): JsonResponse;
}
```

**Frontend Controllers** (Public):
```php
namespace Bongo\Page\Http\Controllers\Frontend;

class PageController extends AbstractController
{
    public function index(): View;           // Homepage (is_home_page = 1)
    public function show($slug): View;       // Page by slug
    public function edit($uuid): View;       // Visual builder editor
    public function update(Request $request, $uuid): RedirectResponse;  // Save builder
}

class PageImageController extends AbstractController
{
    public function upload(Request $request, $uuid): JsonResponse;  // Builder uploads
}

class PageBackgroundImageController extends AbstractController
{
    public function upload(Request $request, $uuid): JsonResponse;  // Builder backgrounds
}
```

**API Controller**:
```php
namespace Bongo\Page\Http\Controllers\Api;

class PageController extends AbstractController
{
    public function index(): JsonResponse;  // Returns PageResource collection
}
```

### Form Requests
```php
namespace Bongo\Page\Http\Requests;

class StorePageRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'name' => 'required',
            'slug' => 'required|max:75|unique:pages,slug,NULL,id,deleted_at,NULL',
        ];
    }
}

class UpdatePageRequest extends FormRequest
{
    // Similar validation rules for updates
}
```

### Events
```php
namespace Bongo\Page\Events;

class PageCreated { /* Event fired after page creation */ }
class PageUpdated { /* Event fired after page update */ }
class PageDeleted { /* Event fired after page deletion */ }
```

### Resources
```php
namespace Bongo\Page\Http\Resources;

class PageResource extends JsonResource
{
    public function toArray($request): array
    {
        return [
            'id' => $this->id,
            'uuid' => $this->uuid,
            'menu_id' => $this->menu_id,
            'key' => $this->key,
            'slug' => $this->slug,
            'name' => $this->name,
            'summary' => $this->summary,
            'status' => $this->status,
        ];
    }
}
```

### Commands
```php
namespace Bongo\Page\Commands;

class ExportPagesCommand extends Command
{
    protected $signature = 'page:export';
    protected $description = 'Exports the pages to a csv file.';

    public function handle(): void;  // Exports to storage/pages.csv
}
```

### View Composers
```php
namespace Bongo\Page\Http\ViewComposers;

class PageMenuComposer
{
    public function compose(View $view): void;  // Injects active menus
}

class CreatePageComposer
{
    public function compose(View $view): void;  // Data for create button
}
```

## Code Style Templates

### Creating a Controller Action
```php
namespace Bongo\Page\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Page\Models\Page;
use Illuminate\View\View;

class PageController extends AbstractController
{
    protected Page $page;

    public function __construct(Page $page)
    {
        $this->page = $page;
    }

    public function index(): View
    {
        return view('page::backend.index');
    }

    public function store(StorePageRequest $request): RedirectResponse
    {
        $page = $this->page->create($request->all());
        event(new PageCreated($page));

        return redirect()
            ->route('backend.page.show', $page->id)
            ->success(trans('page::backend.store_success'));
    }
}
```

### Creating a Form Request
```php
namespace Bongo\Page\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StorePageRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'name' => 'required',
            'slug' => 'required|max:75|unique:pages,slug,NULL,id,deleted_at,NULL',
            'content' => 'nullable',
            'status' => 'required|in:pending,active,inactive',
        ];
    }
}
```

### Creating an Event
```php
namespace Bongo\Page\Events;

use Bongo\Page\Models\Page;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class PageCreated
{
    use Dispatchable, SerializesModels;

    public Page $page;

    public function __construct(Page $page)
    {
        $this->page = $page;
    }
}
```

### Registering Event Listeners
```php
// In PageServiceProvider.php
protected array $listeners = [
    PageCreated::class => [],
    PageUpdated::class => [
        ClearMenuCache::class,
        UpdateSitemap::class,
    ],
    PageDeleted::class => [
        ClearMenuCache::class,
        UpdateSitemap::class,
    ],
];
```

### Creating a Migration
```php
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::table('pages', function (Blueprint $table) {
            $table->string('new_field')->nullable()->after('existing_field');
        });
    }

    public function down(): void
    {
        Schema::table('pages', function (Blueprint $table) {
            $table->dropColumn('new_field');
        });
    }
};
```

### Adding Routes
```php
// In src/Routes/backend.php
use Bongo\Page\Http\Controllers\Backend\PageController;
use Illuminate\Support\Facades\Route;

Route::as('page.')
    ->prefix(config('page.prefix'))
    ->group(function () {
        Route::get('/', [PageController::class, 'index'])->name('index');
        Route::post('store', [PageController::class, 'store'])->name('store');

        Route::prefix('{page}')->group(function () {
            Route::get('/', [PageController::class, 'show'])->name('show');
            Route::post('update', [PageController::class, 'update'])->name('update');
        });
    });
```

### Creating Views
```blade
{{-- resources/views/backend/index.blade.php --}}
@extends('framework::backend.layouts.app')

@section('content')
    <div class="page-header">
        <h1>{{ trans('page::backend.pages') }}</h1>
        @include('page::backend.partials.btn_create')
    </div>

    <div class="page-content">
        {{-- Datatable or content here --}}
    </div>
@endsection
```

## Common Patterns

### Package Limit Enforcement
```php
// Check if user has reached their page limit
if (Page::all()->count() >= setting('package::page.number_of_pages')) {
    return redirect()
        ->route('backend.page.index')
        ->warning(trans('page::backend.page_limit_reached'));
}
```

### Homepage Handling
```php
// Get homepage
$page = Page::where('is_home_page', 1)->first();

// Check if page is homepage
if ($page->isHomePage()) {
    // Special handling
}

// Prevent homepage deletion
if ($page->isHomePage()) {
    return redirect()
        ->route('backend.page.index')
        ->error(trans('page::backend.delete_home_page'));
}
```

### Page Duplication
```php
// Duplicate a page
$newPage = $page->duplicate();  // Returns new Page instance with:
// - New UUID
// - Randomized name (original + random 6 chars)
// - New slug based on name
// - New key
// - is_home_page = 0
// - status = PENDING
// - Cloned image relations
```

### Builder Integration
```php
// Process builder HTML and extract images
use Bongo\Image\Services\BuilderService;

$page->content = $request->get('html');
$builderService = new BuilderService($page, '/pages/');
$page->content = $builderService->process();
$page->save();

event(new PageUpdated($page));
```

### Query Scopes
```php
// Get all pages except homepage
$pages = Page::notHomePage()->get();

// Get active pages
$pages = Page::active()->get();  // from HasStatus trait

// Get pages with menu
$pages = Page::whereHas('menu')->get();
```

### Flash Messages
```php
return redirect()
    ->route('backend.page.index')
    ->success(trans('page::backend.store_success'));

return back()->error(trans('page::backend.delete_failed'));

return redirect()
    ->route('backend.page.create')
    ->warning(trans('page::backend.page_limit_reached'));
```

### Translation Keys
```php
trans('page::backend.store_success')
trans('page::backend.update_success')
trans('page::backend.delete_success')
trans('page::backend.delete_failed')
trans('page::backend.page_limit_reached')
trans('page::backend.delete_home_page')
```

## Route Naming Convention

- **API**: `api.page.{action}` (e.g., `api.page.index`)
- **Backend**: `backend.page.{action}` (e.g., `backend.page.show`)
- **Frontend**: `frontend.{resource}.{action}` or `frontend.page.{action}`
- **Special**: `frontend.index` (homepage)

## Middleware Usage

**Backend Routes:**
- `auth` - Requires authentication
- `employee` - Requires employee role

**Frontend Routes:**
- `hasRedirects` - Checks for URL redirects
- `hasShortCodes` - Processes shortcodes in content
- `minifyHtml` - Minifies HTML output

**API Routes:**
- `auth:sanctum` - Sanctum API authentication

## File Path Conventions

- **Controllers**: `src/Http/Controllers/{Section}/{Controller}.php`
- **Models**: `src/Models/{Model}.php`
- **Requests**: `src/Http/Requests/{Action}{Model}Request.php`
- **Events**: `src/Events/{Model}{Action}.php`
- **Views**: `src/Views/{section}/{view}.blade.php`
- **Routes**: `src/Routes/{section}.php`
- **Migrations**: `src/Migrations/{timestamp}_{description}.php`
- **Commands**: `src/Commands/{Action}{Model}Command.php`

## Database Conventions

**Table Name:** `pages`

**Key Columns:**
- `id` - Auto-incrementing primary key
- `uuid` - Public identifier (used in builder URLs)
- `menu_id` - Foreign key to menus table (nullable)
- `key` - Unique internal key
- `name` - Page name/title
- `slug` - URL slug (unique, max 75 chars)
- `content` - HTML content
- `status` - enum: pending, active, inactive
- `is_home_page` - boolean flag (only one page can be homepage)
- `meta_title`, `meta_description`, `meta_canonical`, `meta_index` - SEO fields
- `css`, `js` - Custom code per page
- `schema` - JSON-LD schema markup
- `deleted_at` - Soft delete timestamp

## Testing Patterns

```php
namespace Bongo\Page\Tests;

use Orchestra\Testbench\TestCase;

class PageTest extends TestCase
{
    protected function getPackageProviders($app)
    {
        return [
            \Bongo\Page\PageServiceProvider::class,
        ];
    }

    public function test_can_create_page()
    {
        $page = Page::create([
            'name' => 'Test Page',
            'slug' => 'test-page',
            'status' => Page::ACTIVE,
        ]);

        $this->assertInstanceOf(Page::class, $page);
        $this->assertEquals('test-page', $page->slug);
    }
}
```

## Important Notes for Code Generation

1. **Always extend base classes**: Controllers extend `AbstractController`, Models extend `AbstractModel`
2. **Use type hints**: Parameters and return types should be specified
3. **Fire events**: Create/Update/Delete operations should fire corresponding events
4. **Validate input**: Use FormRequest classes for validation
5. **Check package limits**: Before creating/duplicating pages
6. **Homepage protection**: Prevent deletion and duplicate slugs for homepage
7. **Use traits**: Leverage existing traits from `bongo/framework` for common functionality
8. **Soft deletes**: Remember to consider soft-deleted records in unique validation
9. **Builder uses UUID**: Frontend builder routes use UUID, not ID
10. **Translation namespacing**: Always use `page::` prefix for translations
