# Bongo Page Package

A Laravel package that provides comprehensive web page management for the Bongo CMS, including admin CRUD operations, visual builder integration, SEO metadata, and image management.

[![PHP Version](https://img.shields.io/badge/php-%3E%3D8.2-blue)](https://php.net)
[![Laravel Version](https://img.shields.io/badge/laravel-%5E10.0-red)](https://laravel.com)

## Features

- **Full CRUD Operations** - Create, read, update, and delete web pages via admin interface
- **Visual Page Builder** - Frontend visual editor integration for content creation
- **Homepage Management** - Dedicated homepage support with protection against deletion
- **SEO Metadata** - Title, description, canonical URL, and index/noindex control
- **Custom CSS/JS/Schema** - Per-page custom code and JSON-LD structured data
- **Image Management** - Multiple image attachments with visual builder support
- **Menu Association** - Link pages to navigation menus
- **Page Duplication** - Clone pages with automatic UUID/slug generation
- **Package Licensing** - Configurable page limits enforcement
- **Export Functionality** - Export pages to CSV via Artisan command
- **Soft Deletes** - Recoverable page deletion
- **Event System** - Automatic menu cache clearing and sitemap updates

## Requirements

- PHP 8.2 or higher
- Laravel 10.0 or higher
- Composer

## Installation

### Step 1: Install via Composer

```bash
composer require bongo/page
```

### Step 2: Run Migrations

The package uses auto-discovery, so migrations will be automatically available.

```bash
php artisan migrate
```

### Step 3: Publish Configuration (Optional)

```bash
php artisan vendor:publish --provider="Bongo\Page\PageServiceProvider" --tag="page-config"
```

## Configuration

After publishing, edit `config/page.php`:

```php
return [
    'prefix' => 'pages',  // URL prefix for backend routes
];
```

## Usage

### Creating Pages

**Via Admin Interface:**
Navigate to `/admin/pages/create` to access the page creation form.

**Programmatically:**
```php
use Bongo\Page\Models\Page;

$page = Page::create([
    'name' => 'About Us',
    'slug' => 'about-us',
    'content' => '<h1>About Our Company</h1>',
    'status' => Page::ACTIVE,
    'meta_title' => 'About Us - Company Name',
    'meta_description' => 'Learn more about our company...',
]);
```

### Setting a Homepage

```php
$page = Page::find(1);
$page->update(['is_home_page' => 1]);
// Only one page can be the homepage
```

### Using the Visual Builder

Access the builder for any page via `/edit/{uuid}` (requires authentication and employee role):

```php
$page = Page::where('slug', 'about-us')->first();
$builderUrl = route('frontend.page.edit', $page->uuid);
```

### Querying Pages

```php
// Get active pages
$pages = Page::active()->get();

// Get pages excluding homepage
$pages = Page::notHomePage()->get();

// Get pages with menu
$pages = Page::whereHas('menu')->get();

// Get page with images
$page = Page::with('images')->find(1);
```

### Page Duplication

```php
$originalPage = Page::find(1);
$duplicatedPage = $originalPage->duplicate();
// New page will have:
// - New UUID, slug, and key
// - Randomized name
// - Status set to PENDING
// - is_home_page set to 0
// - Cloned image relations
```

### Exporting Pages

```bash
php artisan page:export
```
Exports all pages to `storage/pages.csv`.

## API Endpoints

### List Pages

```http
GET /api/pages
Authorization: Bearer {token}
```

**Response:**
```json
{
  "data": [
    {
      "id": 1,
      "uuid": "550e8400-e29b-41d4-a716-446655440000",
      "menu_id": 1,
      "key": "about-us",
      "slug": "about-us",
      "name": "About Us",
      "summary": "Company information...",
      "status": "active"
    }
  ]
}
```

## Routes

### Backend (Admin)

| Method | URI | Name | Description |
|--------|-----|------|-------------|
| GET | `/admin/pages` | `backend.page.index` | List all pages |
| GET | `/admin/pages/create` | `backend.page.create` | Show create form |
| POST | `/admin/pages/store` | `backend.page.store` | Store new page |
| GET | `/admin/pages/{page}` | `backend.page.show` | Show page details |
| GET | `/admin/pages/{page}/edit` | `backend.page.edit` | Show edit form |
| POST | `/admin/pages/{page}/update` | `backend.page.update` | Update page |
| DELETE | `/admin/pages/{page}/delete` | `backend.page.destroy` | Delete page |
| GET | `/admin/pages/{page}/duplicate` | `backend.page.duplicate` | Duplicate page |
| POST | `/admin/pages/{page}/image` | `backend.page.image` | Upload image |

### Frontend (Public)

| Method | URI | Name | Description |
|--------|-----|------|-------------|
| GET | `/` | `frontend.index` | Homepage |
| GET | `/{slug}` | `frontend.page.show` | Show page by slug |
| GET | `/edit/{uuid}` | `frontend.page.edit` | Visual builder (auth required) |
| POST | `/update/{uuid}` | `frontend.page.update` | Update via builder (auth required) |

## Events

The package fires the following events:

- **`PageCreated`** - Fired when a page is created
- **`PageUpdated`** - Fired when a page is updated (triggers menu cache clear and sitemap update)
- **`PageDeleted`** - Fired when a page is deleted (triggers menu cache clear and sitemap update)

### Listening to Events

```php
use Bongo\Page\Events\PageCreated;
use Illuminate\Support\Facades\Event;

Event::listen(PageCreated::class, function ($event) {
    $page = $event->page;
    // Your custom logic here
});
```

## Model Reference

### Page Model

**Namespace:** `Bongo\Page\Models\Page`

**Traits:**
- `HasContent` - Content field handling
- `HasHeaderClass` - Header CSS management
- `HasImages` - Polymorphic image relations
- `HasKey` - Unique key generation
- `HasSeo` - SEO metadata methods
- `HasStatus` - Status management
- `HasUUID` - UUID generation
- `SoftDeletes` - Soft deletion

**Constants:**
```php
const PENDING = 'pending';
const ACTIVE = 'active';
const INACTIVE = 'inactive';
const INDEX = 'index';
const NO_INDEX = 'noindex';
```

**Key Methods:**
```php
$page->hasMenu(): bool              // Check if page has menu
$page->isHomePage(): bool           // Check if page is homepage
$page->duplicate(): Page            // Duplicate page
$page->isActive(): bool             // Check if status is active
$page->isPending(): bool            // Check if status is pending
```

**Fillable Attributes:**
```php
'menu_id', 'name', 'slug', 'content', 'status', 'is_home_page',
'meta_title', 'meta_description', 'meta_canonical', 'meta_index',
'css', 'js', 'schema'
```

## Testing

```bash
# Run tests
vendor/bin/phpunit

# Run tests with coverage
vendor/bin/phpunit --coverage-html coverage/

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint
```

## Architecture

The package follows Laravel package development best practices and extends `Bongo\Framework\Providers\AbstractServiceProvider` for automatic bootstrapping.

For detailed architecture information, see [ARCHITECTURE.md](ARCHITECTURE.md).

### Key Components

- **Service Provider** - Auto-registers routes, config, views, migrations
- **Controllers** - Separate backend (admin), frontend (public), and API controllers
- **Form Requests** - Validation via `StorePageRequest` and `UpdatePageRequest`
- **Events** - Domain events for create/update/delete operations
- **View Composers** - Automatic data injection for menus and create buttons
- **Migrations** - 11 migrations for complete database schema

## Dependencies

- **bongo/framework** - Base abstracts, traits, and service provider
- **bongo/image** - Image management and builder service
- **bongo/menu** - Menu association and cache clearing
- **bongo/package** - Package licensing and limits
- **bongo/redirect** - URL redirect middleware
- **bongo/setting** - Settings management

## Security

- Homepage cannot be deleted (enforced in controller)
- Builder routes use UUID instead of ID for security
- Builder requires authentication and employee role
- Soft deletes prevent accidental data loss
- Form request validation on all input
- CSRF protection on all forms

## License

This package is proprietary software licensed for use with the Bongo CMS.

## Support

For issues, questions, or feature requests, please contact the development team or visit the repository at https://bitbucket.org/designtec/page.

## Credits

- **Stuart Elliott** - Developer
- **Bespoke UK** - https://bespokeuk.com

## Changelog

See [CHANGELOG.md](CHANGELOG.md) for version history and changes.
