# Architecture Documentation - Bongo Page Package

## Overview

The Bongo Page package is a Laravel-based content management module that provides comprehensive web page creation, management, and display functionality. It integrates with the broader Bongo CMS ecosystem, leveraging the framework's abstract service provider pattern and extending functionality through traits and interfaces.

**Version:** 3.0
**PHP Requirements:** >= 8.2
**Laravel Requirements:** ^10.0

## Table of Contents

1. [Directory Structure](#directory-structure)
2. [Service Provider Bootstrap](#service-provider-bootstrap)
3. [Model Architecture](#model-architecture)
4. [Controller Architecture](#controller-architecture)
5. [Route Organisation](#route-organisation)
6. [Event System](#event-system)
7. [Request Lifecycle](#request-lifecycle)
8. [Database Schema](#database-schema)
9. [View Architecture](#view-architecture)
10. [Extension Points](#extension-points)

---

## Directory Structure

```
bongo/page/
├── src/
│   ├── Commands/
│   │   └── ExportPagesCommand.php              # CSV export command
│   │
│   ├── Config/
│   │   └── page.php                            # Configuration
│   │
│   ├── Events/
│   │   ├── PageCreated.php                     # Creation event
│   │   ├── PageUpdated.php                     # Update event
│   │   └── PageDeleted.php                     # Deletion event
│   │
│   ├── Exports/
│   │   └── PageExport.php                      # Excel export implementation
│   │
│   ├── Http/
│   │   ├── Controllers/
│   │   │   ├── Api/
│   │   │   │   └── PageController.php          # REST API endpoints
│   │   │   ├── Backend/
│   │   │   │   ├── PageController.php          # Admin CRUD
│   │   │   │   ├── PageDatatableController.php # Datatable JSON
│   │   │   │   └── PageImageController.php     # Admin image uploads
│   │   │   └── Frontend/
│   │   │       ├── PageController.php          # Public display + builder
│   │   │       ├── PageImageController.php     # Builder image uploads
│   │   │       └── PageBackgroundImageController.php
│   │   │
│   │   ├── Requests/
│   │   │   ├── StorePageRequest.php            # Create validation
│   │   │   └── UpdatePageRequest.php           # Update validation
│   │   │
│   │   ├── Resources/
│   │   │   └── PageResource.php                # API transformation
│   │   │
│   │   └── ViewComposers/
│   │       ├── PageMenuComposer.php            # Menu dropdown data
│   │       └── CreatePageComposer.php          # Create button data
│   │
│   ├── Migrations/
│   │   ├── 2019_01_01_000001_create_pages_table.php
│   │   ├── 2021_01_01_000002_add_transparent_header_column_to_pages_table.php
│   │   ├── 2021_01_01_000003_add_sticky_header_column_to_pages_table.php
│   │   ├── 2021_01_01_000004_alter_meta_description_column_on_pages_table.php
│   │   ├── 2021_01_01_000005_create_page_relation_pivot_table.php
│   │   ├── 2021_01_01_000006_add_sort_order_column_to_page_relation_pivot_table.php
│   │   ├── 2021_01_01_000007_drop_page_relation_pivot_table.php
│   │   ├── 2021_01_01_000008_add_menu_id_column_to_pages_table.php
│   │   ├── 2021_01_01_000009_change_page_transparent_header_column_to_enum.php
│   │   ├── 2021_01_01_000010_change_page_sticky_header_column_to_enum.php
│   │   └── 2025_01_01_000011_add_schema_column_to_pages_table.php
│   │
│   ├── Models/
│   │   └── Page.php                            # Core page model
│   │
│   ├── Routes/
│   │   ├── api.php                             # API routes (sanctum auth)
│   │   ├── backend.php                         # Admin routes (auth+employee)
│   │   └── frontend.php                        # Public + builder routes
│   │
│   ├── Seeders/
│   │   ├── DataSeeder.php                      # Demo data
│   │   └── PackageSeeder.php                   # Package configuration
│   │
│   ├── Views/
│   │   ├── backend/                            # Admin templates
│   │   │   ├── index.blade.php                 # Page list
│   │   │   ├── create.blade.php                # Create form
│   │   │   ├── edit.blade.php                  # Edit form
│   │   │   ├── show.blade.php                  # Detail view
│   │   │   └── partials/                       # Shared components
│   │   │
│   │   ├── frontend/                           # Public templates
│   │   │   ├── home.blade.php                  # Homepage
│   │   │   ├── show.blade.php                  # Page display
│   │   │   ├── builder.blade.php               # Visual editor
│   │   │   └── preview.blade.php               # Preview
│   │   │
│   │   ├── exports/                            # Export templates
│   │   │   └── pages.blade.php
│   │   │
│   │   └── stubs/                              # Pre-built page templates
│   │       ├── privacy.blade.php
│   │       ├── cookie_policy.blade.php
│   │       ├── thank_you.blade.php
│   │       └── thank_you_for_your_review.blade.php
│   │
│   └── PageServiceProvider.php                 # Main service provider
│
├── tests/
│   └── TestCase.php                            # Base test class
│
├── composer.json                               # Dependencies
├── phpunit.xml                                 # Test configuration
└── README.md                                   # Package documentation
```

---

## Service Provider Bootstrap

### Inheritance Hierarchy

```
PageServiceProvider
    └── extends Bongo\Framework\Providers\AbstractServiceProvider
            └── extends Illuminate\Support\ServiceProvider
```

### PageServiceProvider Structure

```php
<?php

namespace Bongo\Page;

use Bongo\Framework\Providers\AbstractServiceProvider;

class PageServiceProvider extends AbstractServiceProvider
{
    // Module name - used for config, views, translations
    protected string $module = 'page';

    // Artisan commands to register
    protected array $commands = [
        ExportPagesCommand::class,
    ];

    // View composers (composer class => [views])
    protected array $composers = [
        PageMenuComposer::class => ['page::backend.partials.dropdowns.menu'],
        CreatePageComposer::class => ['page::backend.partials.btn_create'],
    ];

    // Event listeners (event class => [listeners])
    protected array $listeners = [
        PageCreated::class => [],
        PageUpdated::class => [
            ClearMenuCache::class,    // From bongo/menu
            UpdateSitemap::class,     // From bongo/sitemap
        ],
        PageDeleted::class => [
            ClearMenuCache::class,
            UpdateSitemap::class,
        ],
    ];
}
```

### AbstractServiceProvider Auto-Bootstrap

The parent `AbstractServiceProvider` automatically handles:

1. **Config Registration**
   - Loads from `src/Config/{module}.php`
   - Publishes with tag `{module}-config`

2. **Route Registration**
   - `src/Routes/api.php` → Prefix: `api/{prefix}`, Middleware: `auth:sanctum`
   - `src/Routes/backend.php` → Prefix: `backend/{prefix}`, Middleware: `auth`, `employee`
   - `src/Routes/frontend.php` → Named: `frontend.*`
   - `src/Routes/web.php` → Standard web routes
   - `src/Routes/custom.php` → No middleware (webhooks, etc.)

3. **View Registration**
   - Namespace: `{module}::`
   - Directory: `src/Views/{module}/`

4. **Migration Registration**
   - Loads from `src/Migrations/`
   - Publishes with tag `{module}-migrations`

5. **Translation Registration**
   - Namespace: `{module}::`
   - Directory: `src/Translations/`

6. **Component Registration**
   - Registers arrays: `$commands`, `$middlewares`, `$composers`, `$listeners`, `$subscribers`

---

## Model Architecture

### Page Model Class Diagram

```
┌─────────────────────────────────────────────────────────────┐
│                        Page Model                           │
├─────────────────────────────────────────────────────────────┤
│ Extends: AbstractModel                                      │
│ Implements: Imageable                                       │
├─────────────────────────────────────────────────────────────┤
│ Traits:                                                     │
│   • HasContent          (content field handling)            │
│   • HasHeaderClass      (header CSS management)             │
│   • HasImages           (polymorphic image relations)       │
│   • HasKey              (unique key generation)             │
│   • HasSeo              (SEO metadata methods)              │
│   • HasStatus           (status: active/pending/inactive)   │
│   • HasUUID             (UUID generation)                   │
│   • SoftDeletes         (soft deletion)                     │
├─────────────────────────────────────────────────────────────┤
│ Constants:                                                  │
│   STATUS: PENDING, ACTIVE, INACTIVE                         │
│   INDEX: INDEX, NO_INDEX                                    │
│   ENABLED: 1, DISABLED: 0                                   │
├─────────────────────────────────────────────────────────────┤
│ Relations:                                                  │
│   • menu(): BelongsTo → Menu                                │
├─────────────────────────────────────────────────────────────┤
│ Methods:                                                    │
│   • hasMenu(): bool                                         │
│   • isHomePage(): bool                                      │
│   • duplicate(): self                                       │
│   • scopeNotHomePage($query)                                │
├─────────────────────────────────────────────────────────────┤
│ Fillable:                                                   │
│   menu_id, name, slug, content, status, is_home_page,       │
│   meta_title, meta_description, meta_canonical, meta_index, │
│   css, js, schema                                           │
└─────────────────────────────────────────────────────────────┘
```

### Trait Functionality Reference

| Trait | Methods Provided | Database Columns | Purpose |
|-------|------------------|------------------|---------|
| `HasContent` | `getContent()`, `setContent()` | `content` | HTML content management |
| `HasHeaderClass` | `getHeaderClass()`, `hasHeaderClass()` | `header_class`, `sticky_header`, `transparent_header` | Header styling options |
| `HasImages` | `images()`, `getImage()`, `getImages()` | Polymorphic relation | Image attachments via morph |
| `HasKey` | `formatKey()`, `generateKey()` | `key` | Unique internal identifier |
| `HasSeo` | `getMetaTitle()`, `getMetaDescription()`, `getCanonical()`, `shouldIndex()` | `meta_title`, `meta_description`, `meta_canonical`, `meta_index` | SEO metadata |
| `HasStatus` | `isActive()`, `isPending()`, `isInactive()`, `scopeActive()`, `scopePending()` | `status` | Status management |
| `HasUUID` | `generateUuid()`, Boot event to auto-generate | `uuid` | Public identifier generation |
| `SoftDeletes` | `delete()`, `restore()`, `forceDelete()`, `trashed()` | `deleted_at` | Soft deletion support |

### Model Relationships

```
     ┌──────────┐
     │   Menu   │
     └────┬─────┘
          │ 1
          │
          │ belongsTo
          │
          │ n
     ┌────┴─────────┐
     │     Page     │
     └──────┬───────┘
            │
            │ morphMany
            │
            │
     ┌──────┴─────┐
     │   Image    │ (via bongo/image)
     └────────────┘
```

---

## Controller Architecture

### Controller Hierarchy

```
AbstractController (bongo/framework)
    │
    ├── Backend\PageController
    ├── Backend\PageDatatableController
    ├── Backend\PageImageController
    ├── Frontend\PageController
    ├── Frontend\PageImageController
    ├── Frontend\PageBackgroundImageController
    └── Api\PageController
```

### Backend Controller Flow

```
┌─────────────────────────────────────────────────────────────┐
│            Backend\PageController                           │
├─────────────────────────────────────────────────────────────┤
│                                                             │
│  index()                                                    │
│    └─> Returns view: page::backend.index                   │
│                                                             │
│  create()                                                   │
│    ├─> Check package limit                                 │
│    └─> Returns view: page::backend.create                  │
│                                                             │
│  store(StorePageRequest)                                    │
│    ├─> Check package limit                                 │
│    ├─> Create page                                         │
│    ├─> Fire: PageCreated event                             │
│    └─> Redirect to: backend.page.show                      │
│                                                             │
│  show(Page)                                                 │
│    └─> Returns view: page::backend.show                    │
│                                                             │
│  edit(Page)                                                 │
│    └─> Returns view: page::backend.edit                    │
│                                                             │
│  update(UpdatePageRequest, Page)                            │
│    ├─> Update page                                         │
│    ├─> Fire: PageUpdated event                             │
│    │     ├─> Triggers: ClearMenuCache                      │
│    │     └─> Triggers: UpdateSitemap                       │
│    └─> Redirect to: backend.page.show                      │
│                                                             │
│  destroy(Page)                                              │
│    ├─> Check if homepage (prevent deletion)                │
│    ├─> Delete page                                         │
│    ├─> Fire: PageDeleted event                             │
│    │     ├─> Triggers: ClearMenuCache                      │
│    │     └─> Triggers: UpdateSitemap                       │
│    └─> Redirect back or to index                           │
│                                                             │
│  duplicate(Page)                                            │
│    ├─> Check package limit                                 │
│    ├─> Call: $page->duplicate()                            │
│    │     ├─> Replicate with new UUID                       │
│    │     ├─> Randomize name                                │
│    │     ├─> Generate new slug/key                         │
│    │     ├─> Set: is_home_page = 0, status = PENDING       │
│    │     └─> Clone image relations                         │
│    └─> Redirect to: backend.page.edit (new page)           │
│                                                             │
└─────────────────────────────────────────────────────────────┘
```

### Frontend Controller Flow

```
┌─────────────────────────────────────────────────────────────┐
│           Frontend\PageController                           │
├─────────────────────────────────────────────────────────────┤
│                                                             │
│  index()  [Route: GET /]                                    │
│    ├─> Find page where is_home_page = 1                    │
│    ├─> Abort 404 if not found                              │
│    └─> Returns view: page::frontend.home                   │
│                                                             │
│  show($slug)  [Route: GET /{slug}]                          │
│    ├─> Find page by slug                                   │
│    ├─> Abort 404 if not found or inactive                  │
│    ├─> Redirect to / if is_home_page                       │
│    └─> Returns view: page::frontend.show                   │
│                                                             │
│  edit($uuid)  [Route: GET /edit/{uuid}]                     │
│    │   Middleware: auth, employee                          │
│    ├─> Find page by UUID                                   │
│    ├─> Abort 404 if not found                              │
│    └─> Returns view: page::frontend.builder                │
│                                                             │
│  update(Request, $uuid)  [Route: POST /update/{uuid}]       │
│    │   Middleware: auth, employee                          │
│    ├─> Find page by UUID                                   │
│    ├─> Get HTML from request                               │
│    ├─> Process via BuilderService                          │
│    │     └─> Extract and store images                      │
│    ├─> Save page                                           │
│    ├─> Fire: PageUpdated event                             │
│    └─> Redirect to: frontend.page.edit                     │
│                                                             │
└─────────────────────────────────────────────────────────────┘
```

### API Controller Flow

```
┌────────────────────────────────────────┐
│        Api\PageController              │
├────────────────────────────────────────┤
│                                        │
│  index()  [Route: GET /api/pages]      │
│    │   Middleware: auth:sanctum        │
│    ├─> Get all pages                  │
│    └─> Return: PageResource::collection│
│                                        │
└────────────────────────────────────────┘
```

---

## Route Organisation

### Route File Mapping

```
src/Routes/
│
├── api.php
│   └─> Prefix: /api/pages
│   └─> Middleware: auth:sanctum
│   └─> Named: api.page.*
│   └─> Routes:
│         • GET /api/pages → api.page.index
│
├── backend.php
│   └─> Prefix: /admin/pages (config: page.prefix)
│   └─> Middleware: auth, employee
│   └─> Named: backend.page.*
│   └─> Routes:
│         • GET  /admin/pages → backend.page.index
│         • GET  /admin/pages/create → backend.page.create
│         • POST /admin/pages/store → backend.page.store
│         • GET  /admin/pages/datatable → backend.page.datatable
│         • GET  /admin/pages/{page} → backend.page.show
│         • GET  /admin/pages/{page}/edit → backend.page.edit
│         • POST /admin/pages/{page}/update → backend.page.update
│         • DELETE /admin/pages/{page}/delete → backend.page.destroy
│         • GET  /admin/pages/{page}/duplicate → backend.page.duplicate
│         • POST /admin/pages/{page}/image → backend.page.image
│
└── frontend.php
    └─> No prefix (root level)
    └─> Named: frontend.* or frontend.page.*
    └─> Routes:
          Builder (auth + employee middleware):
          • GET  /edit/{uuid} → frontend.page.edit
          • POST /update/{uuid} → frontend.page.update

          Public (hasRedirects, hasShortCodes, minifyHtml middleware):
          • GET  / → frontend.index (homepage)
          • GET  /{slug} → frontend.page.show (catch-all)

          Upload endpoints:
          • POST /image/{uuid} → frontend.page.image
          • POST /background/{uuid} → frontend.page.background_image
```

### Middleware Reference

| Middleware | Purpose | Routes |
|------------|---------|--------|
| `auth` | Laravel authentication | Backend, Builder |
| `employee` | Role-based access (employees only) | Backend, Builder |
| `auth:sanctum` | API token authentication | API |
| `hasRedirects` | Check for URL redirects | Frontend public |
| `hasShortCodes` | Process shortcodes in content | Frontend public |
| `minifyHtml` | Minify HTML output | Frontend public |

---

## Event System

### Event Flow Diagram

```
┌──────────────────┐
│  User Action     │
└────────┬─────────┘
         │
         ▼
┌──────────────────────────────────────────────────────────────┐
│  Controller Action                                           │
│    • store()   → Fire: PageCreated                           │
│    • update()  → Fire: PageUpdated                           │
│    • destroy() → Fire: PageDeleted                           │
└────────┬─────────────────────────────────────────────────────┘
         │
         ▼
┌──────────────────────────────────────────────────────────────┐
│  Event Listeners (registered in PageServiceProvider)         │
├──────────────────────────────────────────────────────────────┤
│                                                              │
│  PageCreated → []  (no listeners)                            │
│                                                              │
│  PageUpdated → [ClearMenuCache, UpdateSitemap]               │
│      │                                                       │
│      ├─> ClearMenuCache::handle()                           │
│      │      └─> Cache::forget('menu.items.*')               │
│      │                                                       │
│      └─> UpdateSitemap::handle()                            │
│             └─> Regenerate sitemap.xml                      │
│                                                              │
│  PageDeleted → [ClearMenuCache, UpdateSitemap]               │
│      │                                                       │
│      ├─> ClearMenuCache::handle()                           │
│      └─> UpdateSitemap::handle()                            │
│                                                              │
└──────────────────────────────────────────────────────────────┘
```

### Event Classes

```php
namespace Bongo\Page\Events;

use Bongo\Page\Models\Page;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class PageCreated
{
    use Dispatchable, SerializesModels;

    public Page $page;

    public function __construct(Page $page)
    {
        $this->page = $page;
    }
}

// PageUpdated and PageDeleted follow same structure
```

---

## Request Lifecycle

### Create Page Lifecycle

```
1. User clicks "Create Page"
   └─> GET /admin/pages/create

2. Backend\PageController::create()
   ├─> Check: Page::all()->count() >= setting('package::page.number_of_pages')
   │     └─> If exceeded: redirect with warning
   └─> Return view: page::backend.create

3. User fills form and submits
   └─> POST /admin/pages/store

4. StorePageRequest validation
   ├─> name: required
   ├─> slug: required|max:75|unique:pages
   └─> If fails: redirect back with errors

5. Backend\PageController::store()
   ├─> Check package limit again
   ├─> Create page: $page = Page::create($request->all())
   ├─> Fire event: event(new PageCreated($page))
   └─> Redirect to: route('backend.page.show', $page->id)
         └─> Flash message: success(trans('page::backend.store_success'))
```

### Update Page via Builder Lifecycle

```
1. Authenticated employee accesses builder
   └─> GET /edit/{uuid}

2. Frontend\PageController::edit()
   ├─> Find page by UUID
   ├─> Abort 404 if not found
   └─> Return view: page::frontend.builder
         └─> Loads visual editor with page content

3. User edits content in builder and saves
   └─> POST /update/{uuid}
         └─> Payload: { html: "<div>...</div>" }

4. Frontend\PageController::update()
   ├─> Find page by UUID
   ├─> Extract HTML from request
   ├─> Process images via BuilderService
   │     ├─> Find img tags in HTML
   │     ├─> Download/process external images
   │     ├─> Store in bongo/image package
   │     ├─> Update HTML with new image paths
   │     └─> Return processed HTML
   ├─> Update page: $page->content = $processedHtml
   ├─> Save page
   ├─> Fire event: event(new PageUpdated($page))
   │     ├─> ClearMenuCache::handle()
   │     │     └─> Clear cached menu items
   │     └─> UpdateSitemap::handle()
   │           └─> Regenerate sitemap.xml
   └─> Redirect to: route('frontend.page.edit', $page->uuid)
         └─> Flash message: success(trans('page::backend.update_success'))
```

### Display Page Lifecycle

```
1. User visits page URL
   └─> GET /{slug}
         └─> Middleware: hasRedirects → hasShortCodes → minifyHtml

2. hasRedirects middleware
   └─> Check for URL redirects in bongo/redirect package
         └─> If redirect exists: redirect

3. Frontend\PageController::show()
   ├─> Find page: Page::where('slug', $slug)->first()
   ├─> Check: !$page || (!$page->isActive() && !auth()->user())
   │     └─> If true: abort(404)
   ├─> Check: $page->is_home_page
   │     └─> If true: redirect('/', 301)
   └─> Return view: page::frontend.show
         └─> Compact: 'page'

4. hasShortCodes middleware
   └─> Process shortcodes in rendered HTML
         └─> Replace [shortcode] with actual content

5. minifyHtml middleware
   └─> Minify HTML output
         └─> Remove unnecessary whitespace

6. Response sent to user
```

---

## Database Schema

### Pages Table Structure

```sql
CREATE TABLE pages (
    -- Primary key
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,

    -- Foreign keys
    menu_id BIGINT UNSIGNED NULL,  -- FK to menus.id

    -- Identifiers
    uuid VARCHAR(36) UNIQUE NOT NULL,
    key VARCHAR(255) UNIQUE NOT NULL,

    -- Core fields
    name VARCHAR(255) NOT NULL,
    slug VARCHAR(75) UNIQUE NOT NULL,
    content LONGTEXT NULL,
    status ENUM('pending', 'active', 'inactive') DEFAULT 'pending',
    is_home_page BOOLEAN DEFAULT 0,

    -- SEO fields
    meta_title VARCHAR(255) NULL,
    meta_description TEXT NULL,
    meta_canonical VARCHAR(255) NULL,
    meta_index ENUM('index', 'noindex') DEFAULT 'index',

    -- Custom code
    css LONGTEXT NULL,
    js LONGTEXT NULL,
    schema LONGTEXT NULL,  -- JSON-LD structured data

    -- Header options
    sticky_header ENUM('default', 'enabled', 'disabled') NULL,
    transparent_header ENUM('default', 'enabled', 'disabled') NULL,
    header_class VARCHAR(255) NULL,

    -- Timestamps
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    deleted_at TIMESTAMP NULL,  -- Soft deletes

    -- Indexes
    INDEX idx_slug (slug),
    INDEX idx_status (status),
    INDEX idx_is_home_page (is_home_page),
    INDEX idx_deleted_at (deleted_at),
    UNIQUE KEY unique_slug_not_deleted (slug, deleted_at)
);
```

### Images Relationship (Polymorphic)

```sql
-- From bongo/image package
CREATE TABLE images (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    imageable_type VARCHAR(255) NOT NULL,  -- 'Bongo\Page\Models\Page'
    imageable_id BIGINT UNSIGNED NOT NULL,  -- page.id
    filename VARCHAR(255) NOT NULL,
    path VARCHAR(255) NOT NULL,
    -- ... other image fields

    INDEX idx_imageable (imageable_type, imageable_id)
);
```

### Migration Timeline

| Migration | Description |
|-----------|-------------|
| `2019_01_01_000001` | Create pages table (initial) |
| `2021_01_01_000002` | Add transparent_header column |
| `2021_01_01_000003` | Add sticky_header column |
| `2021_01_01_000004` | Alter meta_description column type |
| `2021_01_01_000005` | Create page_relation pivot table |
| `2021_01_01_000006` | Add sort_order to page_relation |
| `2021_01_01_000007` | Drop page_relation pivot table |
| `2021_01_01_000008` | Add menu_id column |
| `2021_01_01_000009` | Change transparent_header to enum |
| `2021_01_01_000010` | Change sticky_header to enum |
| `2025_01_01_000011` | Add schema column (JSON-LD) |

---

## View Architecture

### View Namespace

All views use the `page::` namespace, mapping to `src/Views/page/`.

### View Hierarchy

```
page::
├── backend/
│   ├── index.blade.php              # Page list (datatable)
│   ├── create.blade.php             # Create form
│   ├── edit.blade.php               # Edit form
│   ├── show.blade.php               # Detail view
│   └── partials/
│       ├── form/
│       │   └── details.blade.php    # Form fields
│       ├── dropdowns/
│       │   └── menu.blade.php       # Menu selector (uses PageMenuComposer)
│       ├── btn_create.blade.php     # Create button (uses CreatePageComposer)
│       └── ...
│
├── frontend/
│   ├── home.blade.php               # Homepage template
│   ├── show.blade.php               # Page display template
│   ├── builder.blade.php            # Visual editor interface
│   └── preview.blade.php            # Preview template
│
├── exports/
│   └── pages.blade.php              # CSV export template
│
└── stubs/
    ├── privacy.blade.php            # Pre-built privacy policy template
    ├── cookie_policy.blade.php      # Pre-built cookie policy template
    ├── thank_you.blade.php          # Pre-built thank you page template
    └── thank_you_for_your_review.blade.php
```

### View Composers

```
PageMenuComposer
    └─> Attached to: page::backend.partials.dropdowns.menu
    └─> Purpose: Inject active menus into dropdown
    └─> Data: $menus (plucked 'name' => 'id')

CreatePageComposer
    └─> Attached to: page::backend.partials.btn_create
    └─> Purpose: Provide data for create button partial
    └─> Data: (specific to implementation)
```

---

## Extension Points

### Adding New Page Attributes

**Step 1:** Create migration
```bash
php artisan make:migration add_custom_field_to_pages_table
```

```php
// In migration
public function up(): void
{
    Schema::table('pages', function (Blueprint $table) {
        $table->string('custom_field')->nullable()->after('content');
    });
}
```

**Step 2:** Add to fillable in Page model
```php
// src/Models/Page.php
protected $fillable = [
    // ... existing fields
    'custom_field',
];
```

**Step 3:** Update form requests
```php
// src/Http/Requests/StorePageRequest.php
public function rules(): array
{
    return [
        // ... existing rules
        'custom_field' => 'nullable|string|max:255',
    ];
}
```

**Step 4:** Update views
```blade
{{-- In create.blade.php / edit.blade.php --}}
<div class="form-group">
    <label for="custom_field">Custom Field</label>
    <input type="text" name="custom_field" value="{{ old('custom_field', $page->custom_field ?? '') }}">
</div>
```

### Creating New Events

**Step 1:** Create event class
```php
// src/Events/PagePublished.php
namespace Bongo\Page\Events;

use Bongo\Page\Models\Page;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class PagePublished
{
    use Dispatchable, SerializesModels;

    public Page $page;

    public function __construct(Page $page)
    {
        $this->page = $page;
    }
}
```

**Step 2:** Fire event in controller
```php
// In controller method
if ($page->status === Page::ACTIVE) {
    event(new PagePublished($page));
}
```

**Step 3:** Register listeners
```php
// In PageServiceProvider.php
protected array $listeners = [
    // ... existing events
    PagePublished::class => [
        SendPublishNotification::class,
        UpdateSearchIndex::class,
    ],
];
```

### Adding Controller Actions

**Step 1:** Add route
```php
// src/Routes/backend.php
Route::get('{page}/preview', [PageController::class, 'preview'])->name('preview');
```

**Step 2:** Add controller method
```php
// src/Http/Controllers/Backend/PageController.php
public function preview(Page $page): View
{
    return view('page::backend.preview', compact('page'));
}
```

**Step 3:** Create view
```blade
{{-- src/Views/backend/preview.blade.php --}}
@extends('framework::backend.layouts.app')

@section('content')
    <div class="page-preview">
        {!! $page->content !!}
    </div>
@endsection
```

### Extending with Traits

To add reusable functionality across multiple models:

**Step 1:** Create trait in bongo/framework or locally
```php
// In bongo/framework or locally
namespace Bongo\Framework\Traits;

trait HasVersioning
{
    public function versions()
    {
        return $this->morphMany(Version::class, 'versionable');
    }

    public function createVersion(): void
    {
        $this->versions()->create([
            'content' => $this->content,
            'created_by' => auth()->id(),
        ]);
    }
}
```

**Step 2:** Use trait in Page model
```php
// src/Models/Page.php
use Bongo\Framework\Traits\HasVersioning;

class Page extends AbstractModel
{
    use HasVersioning;
    // ... other traits
}
```

**Step 3:** Use in controller
```php
// Before saving updated content
$page->createVersion();  // Save current version
$page->update($request->all());
```

### Custom Middleware

**Step 1:** Create middleware class
```php
// src/Http/Middleware/CheckPageAccess.php
namespace Bongo\Page\Http\Middleware;

use Closure;
use Illuminate\Http\Request;

class CheckPageAccess
{
    public function handle(Request $request, Closure $next)
    {
        // Custom logic
        return $next($request);
    }
}
```

**Step 2:** Register in service provider
```php
// In PageServiceProvider.php
protected array $middlewares = [
    'checkPageAccess' => CheckPageAccess::class,
];
```

**Step 3:** Use in routes
```php
// In routes file
Route::get('{page}', [PageController::class, 'show'])
    ->middleware('checkPageAccess');
```

### API Resource Customisation

**Step 1:** Extend PageResource
```php
// src/Http/Resources/PageDetailResource.php
namespace Bongo\Page\Http\Resources;

class PageDetailResource extends PageResource
{
    public function toArray($request): array
    {
        return array_merge(parent::toArray($request), [
            'content' => $this->content,
            'meta' => [
                'title' => $this->meta_title,
                'description' => $this->meta_description,
            ],
            'images' => ImageResource::collection($this->whenLoaded('images')),
        ]);
    }
}
```

**Step 2:** Use in controller
```php
// In Api\PageController
public function show(Page $page): JsonResponse
{
    $page->load('images');
    return PageDetailResource::make($page);
}
```

---

## How to Add New Features

### Example: Add Page Scheduling

**1. Database Migration**
```php
Schema::table('pages', function (Blueprint $table) {
    $table->timestamp('publish_at')->nullable()->after('status');
    $table->timestamp('unpublish_at')->nullable()->after('publish_at');
});
```

**2. Update Model**
```php
// Add to fillable
protected $fillable = [
    // ...
    'publish_at',
    'unpublish_at',
];

// Add cast
protected $casts = [
    // ...
    'publish_at' => 'datetime',
    'unpublish_at' => 'datetime',
];

// Add scope
public function scopeScheduled($query)
{
    return $query->where('publish_at', '<=', now())
        ->where(function ($q) {
            $q->whereNull('unpublish_at')
              ->orWhere('unpublish_at', '>', now());
        });
}
```

**3. Create Command**
```php
// src/Commands/PublishScheduledPagesCommand.php
class PublishScheduledPagesCommand extends Command
{
    protected $signature = 'page:publish-scheduled';

    public function handle(): void
    {
        Page::where('status', Page::PENDING)
            ->where('publish_at', '<=', now())
            ->each(function ($page) {
                $page->update(['status' => Page::ACTIVE]);
                event(new PagePublished($page));
            });
    }
}
```

**4. Register Command**
```php
// In PageServiceProvider
protected array $commands = [
    ExportPagesCommand::class,
    PublishScheduledPagesCommand::class,
];
```

**5. Schedule Command**
```php
// In app/Console/Kernel.php
$schedule->command('page:publish-scheduled')->everyMinute();
```

**6. Update Views**
```blade
{{-- Add to form --}}
<input type="datetime-local" name="publish_at" value="{{ $page->publish_at?->format('Y-m-d\TH:i') }}">
<input type="datetime-local" name="unpublish_at" value="{{ $page->unpublish_at?->format('Y-m-d\TH:i') }}">
```

---

## Dependencies and Integration

### Package Dependencies

```
bongo/page
├── bongo/framework (AbstractServiceProvider, traits, abstracts)
├── bongo/image (Image management, BuilderService, Imageable)
├── bongo/menu (Menu association, ClearMenuCache listener)
├── bongo/package (Package limits, settings)
├── bongo/redirect (hasRedirects middleware)
└── bongo/setting (setting() helper function)

Optional integrations:
├── bongo/sitemap (UpdateSitemap listener)
└── bongo/builder (Visual page editor)
```

### Cross-Package Communication

```
Page Package
    ├─> Fires: PageUpdated, PageDeleted
    │
    ├─> Listened by:
    │   ├─> bongo/menu → ClearMenuCache
    │   └─> bongo/sitemap → UpdateSitemap
    │
    ├─> Uses:
    │   ├─> bongo/image → BuilderService, Imageable interface
    │   ├─> bongo/menu → Menu model relationship
    │   └─> bongo/setting → Package limit checks
    │
    └─> Provides:
        ├─> Routes for other packages to link to
        ├─> Page model for relationships
        └─> Views for frontend display
```

---

## Testing Strategy

### Test Structure
```
tests/
├── Unit/
│   ├── Models/
│   │   └── PageTest.php           # Model methods, scopes, relations
│   └── Services/
│       └── ...
│
├── Feature/
│   ├── Backend/
│   │   └── PageControllerTest.php # CRUD operations
│   ├── Frontend/
│   │   └── PageDisplayTest.php    # Public display, builder
│   └── Api/
│       └── PageApiTest.php        # API endpoints
│
└── TestCase.php                   # Base test class
```

### Example Unit Test
```php
namespace Bongo\Page\Tests\Unit\Models;

use Bongo\Page\Models\Page;
use Bongo\Page\Tests\TestCase;

class PageTest extends TestCase
{
    public function test_page_can_be_duplicated()
    {
        $page = Page::factory()->create(['name' => 'Original Page']);

        $duplicate = $page->duplicate();

        $this->assertNotEquals($page->id, $duplicate->id);
        $this->assertNotEquals($page->uuid, $duplicate->uuid);
        $this->assertNotEquals($page->slug, $duplicate->slug);
        $this->assertEquals(Page::PENDING, $duplicate->status);
        $this->assertEquals(0, $duplicate->is_home_page);
    }

    public function test_homepage_cannot_be_deleted()
    {
        $homepage = Page::factory()->create(['is_home_page' => 1]);

        // Test in controller that it returns error
    }
}
```

---

This architecture documentation provides a comprehensive overview of the Bongo Page package structure, patterns, and extension points. For specific implementation details, refer to the source code and inline comments.
