# CLAUDE.md - Bongo Page Package

## Overview

The Bongo Page package provides web page management functionality for the Bongo CMS, allowing creation, editing, and display of web pages through admin and public interfaces with visual builder support.

**Package:** `bongo/page` v3.0
**Namespace:** `Bongo\Page`
**PHP Requirements:** >= 8.2
**Laravel Requirements:** ^10.0

### Quick Links

- [ARCHITECTURE.md](ARCHITECTURE.md) - Detailed architecture documentation with diagrams
- [.cursorrules](.cursorrules) - Cursor AI development guidelines
- [.github/copilot-instructions.md](.github/copilot-instructions.md) - GitHub Copilot patterns

---

## Commands

### Development

```bash
# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint

# Static analysis
vendor/bin/phpstan analyse

# Install dependencies
composer install

# Update dependencies
composer update -W
```

### Artisan Commands

```bash
# Export all pages to CSV
php artisan page:export
# Exports to: storage/pages.csv
```

---

## Quick Architecture Reference

### Package Structure

```
src/
├── Commands/          # Artisan commands (ExportPagesCommand)
├── Config/            # Configuration (page.php)
├── Events/            # Domain events (PageCreated, PageUpdated, PageDeleted)
├── Exports/           # Excel exports (PageExport)
├── Http/
│   ├── Controllers/
│   │   ├── Api/          # API endpoints (sanctum auth)
│   │   ├── Backend/      # Admin CRUD operations
│   │   └── Frontend/     # Public display + visual builder
│   ├── Requests/         # Form validation
│   ├── Resources/        # API transformations
│   └── ViewComposers/    # View data injection
├── Migrations/        # Database schema (11 migrations)
├── Models/            # Page model with traits
├── Routes/            # api.php, backend.php, frontend.php
├── Seeders/           # Demo data and package configuration
└── Views/             # Blade templates (backend, frontend, stubs)
```

### Service Provider (Extends AbstractServiceProvider)

The `PageServiceProvider` extends `Bongo\Framework\Providers\AbstractServiceProvider`, which automatically:

- **Loads config** from `src/Config/page.php`
- **Registers routes** from `src/Routes/` with middleware:
  - `api.php` → `auth:sanctum` middleware
  - `backend.php` → `auth` + `employee` middleware
  - `frontend.php` → public routes with `hasRedirects`, `hasShortCodes`, `minifyHtml`
- **Loads views** from namespace `page::`
- **Runs migrations** from `src/Migrations/`
- **Registers commands, composers, listeners** via protected arrays

```php
// PageServiceProvider.php
protected string $module = 'page';

protected array $commands = [ExportPagesCommand::class];
protected array $composers = [...];
protected array $listeners = [
    PageUpdated::class => [ClearMenuCache::class, UpdateSitemap::class],
    PageDeleted::class => [ClearMenuCache::class, UpdateSitemap::class],
];
```

### Page Model

**Extends:** `Bongo\Framework\Models\AbstractModel`
**Implements:** `Bongo\Image\Interfaces\Imageable`

**Traits:**
- `HasContent` - Content field handling
- `HasHeaderClass` - Header CSS management
- `HasImages` - Polymorphic image relations
- `HasKey` - Unique key generation
- `HasSeo` - SEO metadata methods
- `HasStatus` - Status management (active/pending/inactive)
- `HasUUID` - UUID generation
- `SoftDeletes` - Soft deletion support

**Key Methods:**
- `hasMenu(): bool` - Check if page has associated menu
- `isHomePage(): bool` - Check if page is homepage
- `duplicate(): self` - Duplicate page with new UUID/slug
- `scopeNotHomePage($query)` - Exclude homepage from query

**Relations:**
- `menu(): BelongsTo` - Belongs to Menu model

---

## Key Files Reference

| File | Purpose | Key Points |
|------|---------|------------|
| `src/PageServiceProvider.php` | Service provider | Registers commands, composers, listeners |
| `src/Models/Page.php` | Core model | Uses 8 traits, implements Imageable, has duplication logic |
| `src/Config/page.php` | Configuration | Sets route prefix ('pages') |
| `src/Routes/api.php` | API routes | Prefix: `/api/pages`, middleware: `auth:sanctum` |
| `src/Routes/backend.php` | Admin routes | Prefix: `/admin/pages`, middleware: `auth`, `employee` |
| `src/Routes/frontend.php` | Public routes | Homepage (`/`), page display (`/{slug}`), builder routes |
| `src/Http/Controllers/Backend/PageController.php` | Admin CRUD | Index, create, store, show, edit, update, destroy, duplicate |
| `src/Http/Controllers/Frontend/PageController.php` | Public display | Homepage, show by slug, builder edit/update |
| `src/Http/Controllers/Api/PageController.php` | API | Returns PageResource collection |
| `src/Http/Requests/StorePageRequest.php` | Create validation | name (required), slug (unique, max:75) |
| `src/Http/Requests/UpdatePageRequest.php` | Update validation | Similar to store validation |
| `src/Http/Resources/PageResource.php` | API transformation | Maps model to JSON response |
| `src/Events/PageCreated.php` | Creation event | No listeners registered |
| `src/Events/PageUpdated.php` | Update event | Triggers menu cache clear, sitemap update |
| `src/Events/PageDeleted.php` | Deletion event | Triggers menu cache clear, sitemap update |
| `src/Commands/ExportPagesCommand.php` | Export command | Exports pages to CSV via `page:export` |
| `src/Views/backend/index.blade.php` | Admin list | Datatable view |
| `src/Views/backend/create.blade.php` | Create form | Page creation form |
| `src/Views/backend/edit.blade.php` | Edit form | Page editing form |
| `src/Views/frontend/home.blade.php` | Homepage template | For `is_home_page = 1` |
| `src/Views/frontend/show.blade.php` | Page display | Public page template |
| `src/Views/frontend/builder.blade.php` | Visual editor | Builder interface |
| `src/Migrations/2019_01_01_000001_create_pages_table.php` | Initial schema | Creates pages table |
| `src/Migrations/2025_01_01_000011_add_schema_column_to_pages_table.php` | Latest migration | Adds JSON-LD schema support |

---

## Routes Quick Reference

### API Routes
```
GET /api/pages → api.page.index
```
**Middleware:** `auth:sanctum`

### Backend Routes
```
GET    /admin/pages                  → backend.page.index
GET    /admin/pages/create           → backend.page.create
POST   /admin/pages/store            → backend.page.store
GET    /admin/pages/datatable        → backend.page.datatable
GET    /admin/pages/{page}           → backend.page.show
GET    /admin/pages/{page}/edit      → backend.page.edit
POST   /admin/pages/{page}/update    → backend.page.update
DELETE /admin/pages/{page}/delete    → backend.page.destroy
GET    /admin/pages/{page}/duplicate → backend.page.duplicate
POST   /admin/pages/{page}/image     → backend.page.image
```
**Middleware:** `auth`, `employee`

### Frontend Routes
```
# Public pages
GET  /                → frontend.index (homepage)
GET  /{slug}          → frontend.page.show (any page)

# Builder (employee only)
GET  /edit/{uuid}     → frontend.page.edit
POST /update/{uuid}   → frontend.page.update

# Image uploads
POST /image/{uuid}           → frontend.page.image
POST /background/{uuid}      → frontend.page.background_image
```
**Public middleware:** `hasRedirects`, `hasShortCodes`, `minifyHtml`
**Builder middleware:** `auth`, `employee`

---

## Database Schema

### Pages Table

**Key Columns:**
- `id` - Primary key
- `uuid` - Public identifier (used in builder URLs)
- `menu_id` - Foreign key to menus (nullable)
- `key` - Unique internal key
- `name` - Page title
- `slug` - URL slug (unique, max 75 chars)
- `content` - HTML content (LONGTEXT)
- `status` - enum: pending, active, inactive
- `is_home_page` - boolean (only one allowed)
- `meta_title`, `meta_description`, `meta_canonical`, `meta_index` - SEO fields
- `css`, `js` - Custom CSS/JS per page
- `schema` - JSON-LD structured data
- `sticky_header`, `transparent_header` - Header options
- `deleted_at` - Soft delete timestamp

**Indexes:**
- Unique on `slug` (excluding soft deleted)
- Index on `status`, `is_home_page`, `deleted_at`

---

## Code Style Summary

### Conventions

- **PHP 8.2+** with strict types: `declare(strict_types=1);`
- **Type hints** on parameters and return types
- **PSR-12** via Laravel Pint
- **Constants** for magic values (status, booleans)
- **Events** fired for create/update/delete operations
- **Flash messages**: `->success()`, `->error()`, `->warning()`
- **Translations**: `trans('page::backend.{key}')`

### Controller Pattern

```php
namespace Bongo\Page\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Page\Models\Page;
use Illuminate\View\View;

class PageController extends AbstractController
{
    protected Page $page;

    public function __construct(Page $page)
    {
        $this->page = $page;
    }

    public function index(): View
    {
        return view('page::backend.index');
    }
}
```

### Form Request Pattern

```php
namespace Bongo\Page\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StorePageRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'name' => 'required',
            'slug' => 'required|max:75|unique:pages,slug,NULL,id,deleted_at,NULL',
        ];
    }
}
```

### Event Pattern

```php
// Fire in controller
event(new PageCreated($page));
event(new PageUpdated($page));
event(new PageDeleted($page));

// Register in PageServiceProvider
protected array $listeners = [
    PageUpdated::class => [ClearMenuCache::class, UpdateSitemap::class],
];
```

---

## Common Patterns

### Package Limit Enforcement

```php
// Check before create/duplicate
if (Page::all()->count() >= setting('package::page.number_of_pages')) {
    return redirect()
        ->route('backend.page.index')
        ->warning(trans('page::backend.page_limit_reached'));
}
```

### Homepage Protection

```php
// Get homepage
$page = Page::where('is_home_page', 1)->first();

// Check if homepage
if ($page->isHomePage()) {
    // Cannot delete homepage
}

// Redirect homepage slug to /
if ($page->is_home_page) {
    return redirect()->to('/', 301);
}
```

### Page Duplication

```php
$newPage = $page->duplicate();
// Returns new Page with:
// - New UUID, slug, key
// - Randomized name (original + 6 random chars)
// - is_home_page = 0
// - status = PENDING
// - Cloned image relations
```

### Builder Integration

```php
use Bongo\Image\Services\BuilderService;

// Process builder HTML
$page->content = $request->get('html');
$builderService = new BuilderService($page, '/pages/');
$page->content = $builderService->process(); // Extracts and stores images
$page->save();

event(new PageUpdated($page));
```

### Query Scopes

```php
// Get all pages except homepage
$pages = Page::notHomePage()->get();

// Get active pages (from HasStatus trait)
$pages = Page::active()->get();

// Get pages with menu
$pages = Page::whereHas('menu')->get();
```

---

## Important Business Logic

### Homepage Management
- Only **one page** can have `is_home_page = 1`
- Homepage **cannot be deleted** (validation in `destroy()` method)
- Accessing homepage via slug redirects to `/` with 301

### Package Limits
- Pages are limited by `setting('package::page.number_of_pages')`
- Checked in: `create()`, `store()`, `duplicate()` methods
- Prevents creating pages beyond licensed amount

### Builder Security
- Builder routes use **UUID** (not ID) for security
- Builder requires **auth** + **employee** middleware
- Images processed via `BuilderService` to extract and store

### Event Side Effects
- `PageUpdated` → Clears menu cache, updates sitemap
- `PageDeleted` → Clears menu cache, updates sitemap
- `PageCreated` → No automatic listeners (extend as needed)

### Status and Visibility
- `status = 'active'` → Visible to public
- `status = 'pending'` → Hidden from public (unless authenticated)
- `status = 'inactive'` → Hidden from public (unless authenticated)
- Soft deleted pages excluded via `deleted_at IS NULL`

---

## Dependencies

### Required Packages
- **bongo/framework** - Base abstracts, traits, service provider
- **bongo/image** - Image handling, `BuilderService`, `Imageable` interface
- **bongo/menu** - Menu association, `ClearMenuCache` listener
- **bongo/package** - Package settings/licensing
- **bongo/redirect** - Redirect middleware
- **bongo/setting** - Settings helper (`setting()`)

### Optional Integrations
- **bongo/sitemap** - `UpdateSitemap` listener (auto sitemap generation)
- **bongo/builder** - Visual page editor (frontend builder)

---

## Testing

### Running Tests

```bash
# Run all tests
vendor/bin/phpunit

# Run specific test file
vendor/bin/phpunit tests/Unit/Models/PageTest.php

# Run with coverage
vendor/bin/phpunit --coverage-html coverage/
```

### Test Configuration
- **Bootstrap:** `vendor/autoload.php`
- **Test suite:** `tests/` directory
- **Package:** Orchestra Testbench for Laravel package testing
- **Database:** `testing` connection (in-memory SQLite)

---

## Extension Points

### Adding New Page Attributes
1. Create migration for new column
2. Add to `$fillable` in `Page` model
3. Update validation in `StorePageRequest`/`UpdatePageRequest`
4. Update views (`create.blade.php`, `edit.blade.php`)
5. Update `PageResource` if field should be in API

### Creating New Events
1. Create event class in `src/Events/`
2. Fire event in controller: `event(new YourEvent($page))`
3. Register listeners in `PageServiceProvider::$listeners`

### Adding Controller Actions
1. Add route in appropriate file (`backend.php`, `frontend.php`, `api.php`)
2. Create method in controller with return type
3. Create/update request class if validation needed
4. Create view if returning HTML

### Adding View Composers
1. Create composer class in `src/Http/ViewComposers/`
2. Implement `compose(View $view)` method
3. Register in `PageServiceProvider::$composers` array

---

## Troubleshooting

### Common Issues

**Issue:** Pages not appearing on frontend
**Solution:** Check `status = 'active'` and `deleted_at IS NULL`

**Issue:** Homepage shows 404
**Solution:** Ensure one page has `is_home_page = 1` and `status = 'active'`

**Issue:** Cannot delete page
**Solution:** Check if it's the homepage (homepage cannot be deleted)

**Issue:** Package limit reached
**Solution:** Increase `setting('package::page.number_of_pages')` or delete pages

**Issue:** Builder images not saving
**Solution:** Check `BuilderService` is processing HTML, verify storage permissions

**Issue:** Menu cache not clearing
**Solution:** Verify `ClearMenuCache` listener is registered for `PageUpdated`/`PageDeleted`

---

## Additional Resources

- **Laravel Documentation:** https://laravel.com/docs/10.x
- **PSR-12 Coding Style:** https://www.php-fig.org/psr/psr-12/
- **Laravel Pint:** https://laravel.com/docs/10.x/pint
- **Orchestra Testbench:** https://packages.tools/testbench

---

## Development Workflow

### Making Changes

1. **Read existing code** - Use the Read tool to understand current implementation
2. **Update code** - Use Edit tool to modify existing files
3. **Add tests** - Create/update tests in `tests/` directory
4. **Check style** - Run `vendor/bin/pint --test`
5. **Fix style** - Run `vendor/bin/pint`
6. **Run tests** - Run `vendor/bin/phpunit`
7. **Commit changes** - Use descriptive commit messages

### Adding Features

1. **Plan the change** - Consider impact on existing functionality
2. **Update database** - Create migration if needed
3. **Update model** - Add fields, methods, relationships
4. **Update controllers** - Add/modify actions
5. **Update routes** - Register new routes
6. **Update views** - Create/modify templates
7. **Fire events** - For significant actions
8. **Update validation** - Add rules for new fields
9. **Update API** - Modify resources if applicable
10. **Test thoroughly** - Unit and feature tests

---

## Notes for AI Assistants

- This package follows **Laravel 10 conventions**
- Extends `AbstractServiceProvider` from `bongo/framework` - **do not manually register routes, config, views, migrations**
- Always **fire events** for create/update/delete operations
- Check **package limits** before creating/duplicating pages
- Protect **homepage** from deletion
- Use **UUID** for builder routes, **ID** for backend routes
- **Soft deletes** must be considered in unique validations
- **Type hints** are mandatory (PHP 8.2+)
- Follow **PSR-12** via Laravel Pint
- Use existing **traits** from `bongo/framework` for common functionality
