<?php

namespace Bongo\Page\Models;

use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasContent;
use Bongo\Framework\Traits\HasHeaderClass;
use Bongo\Framework\Traits\HasKey;
use Bongo\Framework\Traits\HasSeo;
use Bongo\Framework\Traits\HasStatus;
use Bongo\Framework\Traits\HasUUID;
use Bongo\Image\Interfaces\Imageable;
use Bongo\Image\Traits\HasImages;
use Bongo\Menu\Models\Menu;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Page extends AbstractModel implements Imageable
{
    // status
    public const PENDING = 'pending';
    public const ACTIVE = 'active';
    public const INACTIVE = 'inactive';

    public const INDEX = 'index';
    public const NO_INDEX = 'noindex';

    // booleans
    public const ENABLED = 1;
    public const DISABLED = 0;

    // relations
    public const PARENT = 'parent';
    public const CHILD = 'child';
    public const SIBLING = 'sibling';

    use HasContent;
    use HasHeaderClass;
    use HasImages;
    use HasKey;
    use HasSeo;
    use HasStatus;
    use HasUUID;
    use SoftDeletes;

    /** @var array */
    protected $fillable = [
        'menu_id',
        'name',
        'slug',
        'content',
        'status',
        'is_home_page',
        'meta_title',
        'meta_description',
        'meta_canonical',
        'meta_index',
        'css',
        'js',
        'schema',
    ];

    public function menu(): BelongsTo
    {
        return $this->belongsTo(Menu::class, 'menu_id');
    }

    public function scopeNotHomePage($query)
    {
        return $query->where('is_home_page', '!=', 1);
    }

    public function hasMenu(): bool
    {
        $this->loadMissing('menu.items');

        return ! empty($this->menu_id) && ! is_null($this->menu);
    }

    public function isHomePage(): bool
    {
        return $this->is_home_page === 1;
    }

    public function duplicate(): self
    {
        // Copy attributes and update unique fields
        $newItem = $this->replicate();
        $newItem->uuid = (string) Str::uuid();
        $newItem->name = $this->name.' '.Str::random(6);
        $newItem->slug = Str::slug(Str::limit($newItem->name, 75));
        $newItem->key = $newItem->formatKey($newItem->name);
        $newItem->is_home_page = 0;
        $newItem->status = self::PENDING;
        $newItem->save();

        // Reset & re-load the relations
        $this->relations = [];
        $this->load('images');

        // Clone image relations
        if (isset($this->images) && count($this->images) > 0) {
            foreach ($this->images as $image) {
                $newItem->images()->sync($image->id);
            }
        }

        return $newItem;
    }
}
