<?php

namespace Bongo\Page\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Page\Events\PageCreated;
use Bongo\Page\Events\PageDeleted;
use Bongo\Page\Events\PageUpdated;
use Bongo\Page\Http\Requests\StorePageRequest;
use Bongo\Page\Http\Requests\UpdatePageRequest;
use Bongo\Page\Models\Page;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class PageController extends AbstractController
{
    /** @var Page */
    protected Page $page;

    public function __construct(Page $page)
    {
        $this->page = $page;
    }

    public function index(): View
    {
        return view('page::backend.index');
    }

    /**
     * @throws BindingResolutionException
     */
    public function create(): View
    {
        if (Page::all()->count() >= setting('package::page.number_of_pages')) {
            return redirect()
                ->route('backend.page.index')
                ->warning(trans('page::backend.page_limit_reached'));
        }

        $page = $this->page;

        return view('page::backend.create', compact('page'));
    }

    /**
     * @throws BindingResolutionException
     */
    public function store(StorePageRequest $request): RedirectResponse
    {
        if (Page::all()->count() >= setting('package::page.number_of_pages')) {
            return redirect()
                ->route('backend.page.index')
                ->warning(trans('page::backend.page_limit_reached'));
        }

        $page = $this->page->create($request->all());
        event(new PageCreated($page));

        return redirect()
            ->route('backend.page.show', $page->id)
            ->success(trans('page::backend.store_success'));
    }

    public function show(Page $page): View
    {
        return view('page::backend.show', compact('page'));
    }

    public function edit(Page $page): View
    {
        return view('page::backend.edit', compact('page'));
    }

    public function update(UpdatePageRequest $request, Page $page): RedirectResponse
    {
        $page->update($request->all());
        event(new PageUpdated($page));

        return redirect()
            ->route('backend.page.show', $page->id)
            ->success(trans('page::backend.update_success'));
    }

    /**
     * @throws Exception
     */
    public function destroy(Page $page)
    {
        if ($page->isHomePage()) {
            return redirect()
                ->route('backend.page.index')
                ->error(trans('page::backend.delete_home_page'));
        }

        if ($page->delete()) {
            event(new PageDeleted($page));
            $redirect = back();

            // If the page was deleted from their detail view,
            // we'll redirect back to the page list view.
            if (route('backend.page.show', $page) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.page.index');
            }

            return $redirect->success(trans('page::backend.delete_success'));
        }

        return back()->error(trans('page::backend.delete_failed'));
    }

    /**
     * @throws BindingResolutionException
     */
    public function duplicate(Page $page)
    {
        if (Page::all()->count() >= setting('package::page.number_of_pages')) {
            return redirect()
                ->route('backend.page.index')
                ->warning(trans('page::backend.page_limit_reached'));
        }

        $newPage = $page->duplicate();

        return redirect()
            ->route('backend.page.edit', $newPage->id)
            ->success(trans('page::backend.update_success'));
    }
}
