# Bongo Post Package - Architecture Documentation

## Overview
The `bongo/post` package provides comprehensive blog post and post category management for the Bongo CMS platform. It implements a full-featured content management system with SEO support, image attachments, event-driven cache invalidation, and multi-channel content delivery (frontend, backend, API).

**Core Capabilities:**
- Blog posts with publication dates and user attribution
- Hierarchical categorisation (many-to-many relationships)
- Image attachment system via `bongo/image`
- SEO metadata management (title, description, canonical, indexing)
- Custom CSS/JS overrides per post/category
- Soft deletion with audit trail
- Post duplication with relationship cloning
- Related posts by shared categories
- Event-driven sitemap and menu cache updates
- Frontend builder integration for in-place editing

## Directory Structure

```
cms/post/
├── src/
│   ├── Config/
│   │   └── post.php                           # Configuration (URL prefixes)
│   ├── Events/
│   │   ├── PostCreated.php                    # Fired after post creation
│   │   ├── PostUpdated.php                    # Fired after post update
│   │   ├── PostDeleted.php                    # Fired after post deletion
│   │   ├── PostCategoryCreated.php            # Fired after category creation
│   │   ├── PostCategoryUpdated.php            # Fired after category update
│   │   └── PostCategoryDeleted.php            # Fired after category deletion
│   ├── Http/
│   │   ├── Controllers/
│   │   │   ├── Api/
│   │   │   │   ├── PostController.php         # API endpoint for posts (index)
│   │   │   │   └── PostCategoryController.php # API endpoint for categories (index)
│   │   │   ├── Backend/
│   │   │   │   ├── PostController.php         # Admin CRUD + duplicate
│   │   │   │   ├── PostDatatableController.php    # DataTables server-side processing
│   │   │   │   ├── PostImageController.php        # Admin image uploads
│   │   │   │   ├── PostCategoryController.php     # Category admin CRUD
│   │   │   │   ├── PostCategoryDatatableController.php
│   │   │   │   └── PostCategoryImageController.php
│   │   │   └── Frontend/
│   │   │       ├── PostController.php         # Public index/show + builder edit
│   │   │       ├── PostImageController.php    # Builder image upload
│   │   │       ├── PostBackgroundImageController.php # Builder background upload
│   │   │       └── PostCategoryController.php # Category public index/show
│   │   ├── Requests/
│   │   │   ├── StorePostRequest.php           # Validates unique slug, max 75 chars
│   │   │   ├── UpdatePostRequest.php          # Update validation rules
│   │   │   ├── StorePostCategoryRequest.php   # Category store validation
│   │   │   └── UpdatePostCategoryRequest.php  # Category update validation
│   │   ├── Resources/
│   │   │   ├── PostResource.php               # API resource transformation
│   │   │   └── PostCategoryResource.php       # Category API transformation
│   │   └── ViewComposers/
│   │       ├── PostComposer.php               # Provides post dropdown data
│   │       ├── PostCategoryComposer.php       # Provides category dropdown data
│   │       └── RecentPostsComposer.php        # Provides recent posts (limit 2)
│   ├── Listeners/
│   │   ├── SetPostDate.php                    # Auto-sets date on creation (queued)
│   │   └── SetPostUser.php                    # Auto-sets user_id on creation (queued)
│   ├── Migrations/                            # 15 migration files
│   │   ├── 2019_01_01_000001_create_posts_table.php
│   │   ├── 2019_01_01_000002_create_post_categories_table.php
│   │   ├── 2019_01_01_000003_create_post_categories_pivot_table.php
│   │   ├── 2021_01_01_000004_add_transparent_header_column_to_posts_table.php
│   │   ├── 2021_01_01_000005_add_sticky_header_column_to_posts_table.php
│   │   ├── 2021_01_01_000006_add_transparent_header_column_to_post_categories_table.php
│   │   ├── 2021_01_01_000007_add_sticky_header_column_to_post_categories_table.php
│   │   ├── 2021_01_01_000008_alter_meta_description_column_on_posts_table.php
│   │   ├── 2021_01_01_000009_alter_meta_description_column_on_post_categories_table.php
│   │   ├── 2021_01_01_000010_add_summary_column_to_posts_table.php
│   │   ├── 2021_01_01_000011_change_post_transparent_header_column_to_enum.php
│   │   ├── 2021_01_01_000012_change_post_sticky_header_column_to_enum.php
│   │   ├── 2021_01_01_000013_change_post_category_transparent_header_column_to_enum.php
│   │   ├── 2021_01_01_000014_change_post_category_sticky_header_column_to_enum.php
│   │   └── 2025_01_01_000015_add_schema_column_to_posts_table.php
│   ├── Models/
│   │   ├── Post.php                           # Main post model with Imageable
│   │   └── PostCategory.php                   # Category model with Imageable
│   ├── Routes/
│   │   ├── api.php                            # API routes with auth:sanctum
│   │   ├── backend.php                        # Admin routes with auth + employee
│   │   └── frontend.php                       # Public routes with SEO middleware
│   ├── Seeders/
│   │   ├── PackageSeeder.php                  # Seeds package metadata
│   │   └── DataSeeder.php                     # Seeds sample data
│   ├── Traits/
│   │   └── HasRelated.php                     # Related posts functionality
│   ├── Translations/                          # Translation files (namespace: post::)
│   ├── Views/
│   │   ├── backend/                           # Admin Blade templates
│   │   └── frontend/                          # Public Blade templates
│   └── PostServiceProvider.php                # Service provider extending AbstractServiceProvider
├── tests/
│   ├── Feature/                               # Feature tests
│   ├── Unit/                                  # Unit tests
│   └── TestCase.php                           # Base test case
├── .editorconfig
├── .gitignore
├── .styleci.yml
├── auth.json                                  # Composer authentication
├── composer.json                              # Package definition
├── LICENSE
├── phpunit.xml                                # PHPUnit configuration
└── README.md
```

## Class Diagrams

### Model Hierarchy (ASCII)

```
AbstractModel (bongo/framework)
       ↑
       |
       |-- Post (implements Imageable)
       |   |
       |   |-- Uses: HasUUID, HasKey, HasStatus, HasContent
       |   |-- Uses: HasSeo, HasHeaderClass, HasImages, HasRelated
       |   |-- Uses: SoftDeletes
       |   |
       |   |-- Relationships:
       |   |   |-- categories(): BelongsToMany → PostCategory
       |   |   |-- images(): MorphToMany (via HasImages)
       |   |
       |   |-- Custom Methods:
       |       |-- duplicate(): self
       |       |-- getSummaryAttribute($dbValue): ?string
       |       |-- getRelated() (from HasRelated)
       |       |-- getRelatedByRandom(int $limit = 4): Collection
       |       |-- getPrimaryCategory(): ?PostCategory
       |       |-- getPrevious(): ?Post
       |       |-- getNext(): ?Post
       |
       |-- PostCategory (implements Imageable)
           |
           |-- Uses: HasUUID, HasKey, HasStatus, HasContent
           |-- Uses: HasSeo, HasHeaderClass, HasImages
           |-- Uses: SoftDeletes
           |
           |-- Relationships:
               |-- posts(): BelongsToMany → Post
               |-- images(): MorphToMany (via HasImages)
```

### Controller Architecture

```
AbstractController (bongo/framework)
       ↑
       |
       |-- Backend\PostController
       |   |-- index(): View (datatable page)
       |   |-- create(): View
       |   |-- store(StorePostRequest): RedirectResponse → fires PostCreated
       |   |-- show(Post): View
       |   |-- edit(Post): View
       |   |-- update(UpdatePostRequest, Post): RedirectResponse → fires PostUpdated
       |   |-- destroy(Post): RedirectResponse → fires PostDeleted
       |   |-- duplicate(Post): RedirectResponse
       |   |-- syncPostCategories($post): void (private)
       |
       |-- Backend\PostCategoryController
       |   |-- Similar structure to PostController
       |   |-- Fires: PostCategoryCreated, PostCategoryUpdated, PostCategoryDeleted
       |
       |-- Backend\PostDatatableController
       |   |-- index(): JsonResponse (DataTables server-side)
       |
       |-- Backend\PostImageController
       |   |-- upload(Post): JsonResponse
       |
       |-- Frontend\PostController
       |   |-- index(): View (public post listing)
       |   |-- show(string $slug): View (public post detail)
       |   |-- edit(string $uuid): View (builder integration)
       |   |-- update(Request, string $uuid): RedirectResponse (builder)
       |
       |-- Frontend\PostImageController
       |   |-- upload(string $uuid): JsonResponse (builder)
       |
       |-- Frontend\PostBackgroundImageController
       |   |-- upload(string $uuid): JsonResponse (builder)
       |
       |-- Api\PostController
       |   |-- index(): JsonResponse (PostResource collection)
       |
       |-- Api\PostCategoryController
           |-- index(): JsonResponse (PostCategoryResource collection)
```

### Event Flow Diagram

```
Controller Action
       |
       v
   [Create/Update/Delete Post]
       |
       v
   Fire Event (PostCreated/PostUpdated/PostDeleted)
       |
       |---> SetPostDate (PostCreated only)
       |         |-- Sets date if empty
       |         |-- Queued (3 tries)
       |
       |---> SetPostUser (PostCreated only)
       |         |-- Sets user_id if empty
       |         |-- Queued (3 tries)
       |
       |---> ClearMenuCache (Update/Delete)
       |         |-- Clears menu cache (bongo/menu)
       |
       |---> UpdateSitemap (Update/Delete)
                 |-- Regenerates sitemap (bongo/sitemap)

[Same flow for PostCategory events]
```

### Service Provider Bootstrap Flow

```
PostServiceProvider::boot()
       |
       v
AbstractServiceProvider::boot()
       |
       |---> registerConfig()
       |        |-- Loads: src/Config/post.php
       |        |-- Publishes to: config/post.php
       |
       |---> registerRoutes()
       |        |-- Loads: src/Routes/api.php (prefix: api.*, middleware: auth:sanctum)
       |        |-- Loads: src/Routes/backend.php (prefix: backend.*, middleware: auth, employee)
       |        |-- Loads: src/Routes/frontend.php (named: frontend.*)
       |
       |---> registerViews()
       |        |-- Namespace: 'post'
       |        |-- Path: src/Views/post/
       |
       |---> registerMigrations()
       |        |-- Loads from: src/Migrations/
       |
       |---> registerTranslations()
       |        |-- Namespace: 'post'
       |        |-- Path: src/Translations/
       |
       |---> registerViewComposers()
       |        |-- PostComposer → post::backend.partials.dropdowns.post
       |        |-- PostCategoryComposer → post::backend.category.partials.dropdowns.category
       |
       |---> registerListeners()
                |-- PostCreated → [SetPostDate, SetPostUser]
                |-- PostUpdated → [ClearMenuCache, UpdateSitemap]
                |-- PostDeleted → [ClearMenuCache, UpdateSitemap]
                |-- PostCategoryUpdated → [ClearMenuCache, UpdateSitemap]
                |-- PostCategoryDeleted → [ClearMenuCache, UpdateSitemap]

PostServiceProvider::boot() (custom)
       |
       v
AliasLoader::alias('PostCategory', PostCategory::class)
```

## Trait Usage Reference

### Post Model Traits

| Trait | Source Package | Purpose | Key Methods/Properties |
|-------|---------------|---------|----------------------|
| `HasUUID` | bongo/framework | Generates and indexes UUID | `uuid` field, auto-generation on create |
| `HasKey` | bongo/framework | URL-friendly key generation | `key` field, `formatKey($name)` |
| `HasStatus` | bongo/framework | Status management | `status` field, `active()`, `pending()`, `inactive()` scopes |
| `HasContent` | bongo/framework | Content field helpers | `content` field, `getSummary()` method |
| `HasSeo` | bongo/framework | SEO metadata | `meta_title`, `meta_description`, `meta_canonical`, `meta_index` fields |
| `HasHeaderClass` | bongo/framework | Header styling options | `transparent_header`, `sticky_header` enum fields |
| `HasImages` | bongo/image | Image attachment system | `images()` relationship, `hasImages()`, `getPrimaryImage()` |
| `HasRelated` | bongo/post | Related posts logic | `getRelated()`, `getRelatedByRandom()`, `getPrevious()`, `getNext()` |
| `SoftDeletes` | Laravel | Soft deletion | `deleted_at` field, `deleted_by` audit, `withTrashed()` scope |

### PostCategory Model Traits

| Trait | Source Package | Purpose | Key Methods/Properties |
|-------|---------------|---------|----------------------|
| `HasUUID` | bongo/framework | Generates and indexes UUID | `uuid` field, auto-generation on create |
| `HasKey` | bongo/framework | URL-friendly key generation | `key` field, `formatKey($name)` |
| `HasStatus` | bongo/framework | Status management | `status` field, `active()`, `pending()`, `inactive()` scopes |
| `HasContent` | bongo/framework | Content field helpers | `content` field, `getSummary()` method |
| `HasSeo` | bongo/framework | SEO metadata | `meta_title`, `meta_description`, `meta_canonical`, `meta_index` fields |
| `HasHeaderClass` | bongo/framework | Header styling options | `transparent_header`, `sticky_header` enum fields |
| `HasImages` | bongo/image | Image attachment system | `images()` relationship, `hasImages()`, `getPrimaryImage()` |
| `SoftDeletes` | Laravel | Soft deletion | `deleted_at` field, `deleted_by` audit, `withTrashed()` scope |

## Database Schema

### `posts` Table

| Column | Type | Nullable | Indexed | Description |
|--------|------|----------|---------|-------------|
| `id` | UNSIGNED INT | No | Primary | Auto-increment primary key |
| `uuid` | UUID | No | Yes | Unique identifier for external references |
| `user_id` | UNSIGNED INT | Yes | Yes | Foreign key to users table (author) |
| `date` | TIMESTAMP | Yes | No | Publication date/time |
| `name` | VARCHAR(255) | No | No | Post title (max 75 for slug generation) |
| `key` | VARCHAR(255) | Yes | Yes | URL-friendly identifier |
| `slug` | VARCHAR(255) | No | Yes | URL slug (auto-generated from name) |
| `content` | LONGTEXT | Yes | No | Post content (supports HTML/shortcodes) |
| `summary` | TEXT | Yes | No | Short description (auto-generates if empty) |
| `status` | ENUM | No | No | pending/active/inactive |
| `meta_title` | VARCHAR(255) | Yes | No | SEO meta title |
| `meta_description` | TEXT | Yes | No | SEO meta description |
| `meta_canonical` | VARCHAR(255) | Yes | No | Canonical URL |
| `meta_index` | ENUM | No | No | index/noindex |
| `css` | TEXT | Yes | No | Custom CSS overrides |
| `js` | TEXT | Yes | No | Custom JavaScript |
| `schema` | TEXT | Yes | No | Structured data (JSON-LD) |
| `transparent_header` | ENUM | No | No | Header transparency setting |
| `sticky_header` | ENUM | No | No | Sticky header setting |
| `created_by` | UNSIGNED INT | Yes | Yes | User ID who created |
| `updated_by` | UNSIGNED INT | Yes | Yes | User ID who last updated |
| `deleted_by` | UNSIGNED INT | Yes | Yes | User ID who deleted |
| `created_at` | TIMESTAMP | No | No | Creation timestamp |
| `updated_at` | TIMESTAMP | No | No | Last update timestamp |
| `deleted_at` | TIMESTAMP | Yes | No | Soft deletion timestamp |

**Unique Constraints:**
- `slug` (considering `deleted_at` null)

### `post_categories` Table

| Column | Type | Nullable | Indexed | Description |
|--------|------|----------|---------|-------------|
| `id` | UNSIGNED INT | No | Primary | Auto-increment primary key |
| `uuid` | UUID | No | Yes | Unique identifier |
| `name` | VARCHAR(255) | No | No | Category name |
| `key` | VARCHAR(255) | Yes | Yes | URL-friendly identifier |
| `slug` | VARCHAR(255) | No | Yes | URL slug |
| `content` | TEXT | Yes | No | Category description |
| `status` | ENUM | No | No | pending/active/inactive |
| `meta_title` | VARCHAR(255) | Yes | No | SEO meta title |
| `meta_description` | TEXT | Yes | No | SEO meta description |
| `meta_canonical` | VARCHAR(255) | Yes | No | Canonical URL |
| `meta_index` | ENUM | No | No | index/noindex |
| `css` | TEXT | Yes | No | Custom CSS overrides |
| `js` | TEXT | Yes | No | Custom JavaScript |
| `transparent_header` | ENUM | No | No | Header transparency setting |
| `sticky_header` | ENUM | No | No | Sticky header setting |
| `created_by` | UNSIGNED INT | Yes | Yes | User ID who created |
| `updated_by` | UNSIGNED INT | Yes | Yes | User ID who last updated |
| `deleted_by` | UNSIGNED INT | Yes | Yes | User ID who deleted |
| `created_at` | TIMESTAMP | No | No | Creation timestamp |
| `updated_at` | TIMESTAMP | No | No | Last update timestamp |
| `deleted_at` | TIMESTAMP | Yes | No | Soft deletion timestamp |

### `post_categories_pivot` Table

| Column | Type | Nullable | Indexed | Description |
|--------|------|----------|---------|-------------|
| `post_id` | UNSIGNED INT | No | Yes | Foreign key to posts.id |
| `post_category_id` | UNSIGNED INT | No | Yes | Foreign key to post_categories.id |

**Composite Key:** (`post_id`, `post_category_id`)

## Lifecycle Diagrams

### Post Creation Lifecycle

```
1. User submits form
       ↓
2. StorePostRequest validates
       |-- Checks unique slug
       |-- Max 75 characters for name
       ↓
3. PostController::store()
       |-- Creates post record
       |-- Syncs categories (syncPostCategories)
       ↓
4. PostCreated event fired
       ↓
5. Queued Listeners execute (async)
       |-- SetPostDate: Sets date if empty
       |-- SetPostUser: Sets user_id if empty
       ↓
6. Redirect to backend.post.show
```

### Post Update Lifecycle

```
1. User submits edit form
       ↓
2. UpdatePostRequest validates
       ↓
3. PostController::update()
       |-- Updates post record
       |-- Syncs categories (syncPostCategories)
       ↓
4. PostUpdated event fired
       ↓
5. Synchronous Listeners execute
       |-- ClearMenuCache: Invalidates menu cache
       |-- UpdateSitemap: Regenerates sitemap.xml
       ↓
6. Redirect to backend.post.show
```

### Post Duplication Lifecycle

```
1. User clicks duplicate button
       ↓
2. PostController::duplicate()
       ↓
3. Post::duplicate() method
       |-- Replicates all attributes
       |-- Generates new UUID (Str::uuid())
       |-- Sets current user as author
       |-- Sets current timestamp as date
       |-- Appends random 6-char string to name
       |-- Generates new slug from name
       |-- Generates new key from name
       |-- Sets status to PENDING
       |-- Saves new record
       ↓
4. Relationship cloning
       |-- Loads original: categories, images
       |-- Syncs categories to new post
       |-- Syncs images to new post
       ↓
5. Returns new Post instance
       ↓
6. Redirect to backend.post.edit (new post)
```

### Related Posts Retrieval

```
1. Call: $post->getRelatedByRandom(4)
       ↓
2. Load missing categories relationship
       ↓
3. Query: posts in same categories
       |-- Join: post_categories_pivot
       |-- Where: post_id != current
       |-- Where: category_id IN (current post categories)
       |-- Order: Random
       |-- Limit: 4
       ↓
4. Pluck post IDs
       ↓
5. Find posts with images eager loaded
       ↓
6. Return Collection
```

## Extension Points

### Adding New Post Fields

**Step 1:** Create migration
```php
// src/Migrations/YYYY_MM_DD_HHMMSS_add_field_to_posts_table.php
public function up()
{
    Schema::table('posts', function (Blueprint $table) {
        $table->string('your_field')->nullable()->after('content');
    });
}
```

**Step 2:** Add to fillable array
```php
// src/Models/Post.php
protected $fillable = [
    // ... existing fields
    'your_field',
];
```

**Step 3:** Add validation (optional)
```php
// src/Http/Requests/StorePostRequest.php or UpdatePostRequest.php
public function rules(): array
{
    return [
        // ... existing rules
        'your_field' => 'nullable|string|max:255',
    ];
}
```

**Step 4:** Update views
- Add form field in `src/Views/backend/edit.blade.php`
- Display in `src/Views/frontend/show.blade.php`

**Step 5:** Update API resource (if exposing via API)
```php
// src/Http/Resources/PostResource.php
public function toArray(Request $request): array
{
    return [
        // ... existing fields
        'your_field' => $this->your_field,
    ];
}
```

### Adding New Event Listeners

**Step 1:** Create listener class
```php
// src/Listeners/YourListener.php
namespace Bongo\Post\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;

class YourListener implements ShouldQueue
{
    public int $tries = 3;

    public function handle($event): void
    {
        // Your logic here
        $post = $event->post;
    }
}
```

**Step 2:** Register in service provider
```php
// src/PostServiceProvider.php
protected array $listeners = [
    PostCreated::class => [
        SetPostDate::class,
        SetPostUser::class,
        YourListener::class, // Add here
    ],
];
```

### Adding Custom Routes

**Backend Route:**
```php
// src/Routes/backend.php
Route::prefix('{post}')->group(function () {
    Route::get('your-action', [PostController::class, 'yourAction'])
        ->name('your_action');
});
```

**Frontend Route:**
```php
// src/Routes/frontend.php
Route::get('your-endpoint/{slug}', [PostController::class, 'yourEndpoint'])
    ->middleware(['hasRedirects', 'hasShortCodes', 'minifyHtml'])
    ->name('your_endpoint');
```

**API Route:**
```php
// src/Routes/api.php
Route::get('{post}', [PostController::class, 'show'])
    ->name('show');
```

### Adding Custom View Composers

**Step 1:** Create composer
```php
// src/Http/ViewComposers/YourComposer.php
namespace Bongo\Post\Http\ViewComposers;

use Illuminate\View\View;

class YourComposer
{
    public function compose(View $view): void
    {
        $data = // ... fetch data
        $view->with(compact('data'));
    }
}
```

**Step 2:** Register in service provider
```php
// src/PostServiceProvider.php
protected array $composers = [
    YourComposer::class => [
        'post::your.view.path',
    ],
];
```

### Extending Post Model with Custom Methods

```php
// src/Models/Post.php
class Post extends AbstractModel implements Imageable
{
    // ... existing code

    public function yourCustomMethod(): YourReturnType
    {
        // Your logic here
    }

    // Custom scope
    public function scopeYourScope($query, $param)
    {
        return $query->where('field', $param);
    }
}
```

### Creating Custom Events

**Step 1:** Create event class
```php
// src/Events/YourCustomEvent.php
namespace Bongo\Post\Events;

use Bongo\Post\Models\Post;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class YourCustomEvent
{
    use Dispatchable, SerializesModels;

    public Post $post;

    public function __construct(Post $post)
    {
        $this->post = $post;
    }
}
```

**Step 2:** Fire event in controller
```php
// src/Http/Controllers/Backend/PostController.php
use Bongo\Post\Events\YourCustomEvent;

public function yourAction(Post $post)
{
    // ... your logic
    event(new YourCustomEvent($post));
}
```

**Step 3:** Register listeners (if any)
```php
// src/PostServiceProvider.php
protected array $listeners = [
    YourCustomEvent::class => [
        YourEventListener::class,
    ],
];
```

## How to Add New Features

### Adding a New Controller Action

1. **Define route** in appropriate route file
2. **Create controller method** with return type
3. **Create form request** if accepting input
4. **Fire events** if state changes
5. **Create view** if rendering HTML
6. **Add translations** for user messages
7. **Update tests** for new functionality

### Adding a New Model Relationship

1. **Create migration** for pivot table or foreign key
2. **Add relationship method** to model
3. **Update controllers** to load relationship
4. **Update views** to display related data
5. **Consider API resources** for API exposure

### Integrating with New Packages

1. **Add dependency** to `composer.json`
2. **Create event listener** for integration points
3. **Register listener** in service provider
4. **Update documentation** with integration details

### Adding Configuration Options

1. **Add config key** to `src/Config/post.php`
2. **Use config value** via `config('post.your_key')`
3. **Publish config** if users need to override
4. **Document** the new configuration option

## Testing Strategy

### Unit Tests
- Model methods (e.g., `duplicate()`, `getRelatedByRandom()`)
- Trait functionality
- Validation rules

### Feature Tests
- Controller actions (CRUD operations)
- Event firing and listener execution
- Route accessibility
- Middleware application

### Integration Tests
- Full post creation workflow
- Category synchronisation
- Image attachment
- Related posts retrieval

## Performance Considerations

### Query Optimisation
- Always eager load relationships: `Post::with('categories', 'images')`
- Use `select()` to limit columns when not all are needed
- Consider pagination for large datasets

### Caching Strategy
- Menu cache cleared on post updates (via ClearMenuCache listener)
- Consider caching related posts queries
- Sitemap regenerated asynchronously

### Queue Configuration
- `SetPostDate` and `SetPostUser` are queued (3 retries)
- Configure queue workers for production
- Monitor failed jobs

## Security Considerations

### Authorization
- Backend routes require `auth` + `employee` middleware
- Builder edit routes require `auth` + `employee`
- API routes require `auth:sanctum`
- Frontend public routes have no auth (intended)

### Validation
- Unique slug validation includes soft-deleted records
- Max 75 characters for name (slug generation limit)
- Enum validation for status fields
- XSS protection via Laravel's Blade escaping

### Mass Assignment Protection
- Use `$fillable` array in models
- Never use `$guarded = []`
- Validate all user input via form requests

## Dependencies and Integration

### Required Bongo Packages
- `bongo/framework` - AbstractServiceProvider, traits, base classes
- `bongo/image` - Image attachment system
- `bongo/menu` - Menu cache integration
- `bongo/package` - Package metadata management
- `bongo/setting` - Settings system

### Optional Bongo Packages
- `bongo/sitemap` - Sitemap regeneration on content changes

### Laravel Dependencies
- Laravel 10+
- PHP 8.2+
- Sanctum for API authentication
