# Bongo Post Package

A comprehensive Laravel package providing blog post and post category management for the Bongo CMS platform.

[![Latest Version](https://img.shields.io/packagist/v/bongo/post.svg?style=flat-square)](https://packagist.org/packages/bongo/post)
[![MIT License](https://img.shields.io/badge/license-MIT-brightgreen.svg?style=flat-square)](LICENSE)

## Features

- **Blog Posts** - Full-featured blog post management with publication dates and user attribution
- **Categorisation** - Many-to-many relationships between posts and categories
- **Image Attachments** - Image management via `bongo/image` package with thumbnail support
- **SEO Optimisation** - Meta title, description, canonical URLs, and indexing control
- **Custom Styling** - Per-post/category CSS and JavaScript overrides
- **Event-Driven** - Automatic sitemap updates and menu cache invalidation
- **API Support** - RESTful API endpoints with Sanctum authentication
- **Builder Integration** - Frontend in-place editing capabilities
- **Post Duplication** - Clone posts with all relationships preserved
- **Related Posts** - Automatic related post suggestions by category
- **Soft Deletion** - Recoverable deletion with full audit trail
- **Frontend & Admin** - Separate public and admin interfaces

## Requirements

- PHP 8.2 or higher
- Laravel 10.0 or higher

## Installation

Install via Composer:

```bash
composer require bongo/post
```

The service provider will be automatically registered via Laravel's package auto-discovery.

### Publish Configuration (Optional)

```bash
php artisan vendor:publish --provider="Bongo\Post\PostServiceProvider" --tag="config"
```

This creates `config/post.php` where you can customise URL prefixes.

### Run Migrations

```bash
php artisan migrate
```

This creates the following tables:
- `posts` - Blog posts with SEO and content fields
- `post_categories` - Post categories
- `post_categories_pivot` - Many-to-many relationship

## Usage

### Creating Posts

**Via Admin Panel:**
Navigate to `/admin/posts` to access the post management interface.

**Programmatically:**
```php
use Bongo\Post\Models\Post;

$post = Post::create([
    'name' => 'My First Blog Post',
    'content' => '<p>Post content here...</p>',
    'status' => Post::ACTIVE,
]);

// Attach categories
$post->categories()->sync([1, 2, 3]);

// Attach images
$post->images()->sync([4, 5]);
```

### Working with Categories

```php
use Bongo\Post\Models\PostCategory;

$category = PostCategory::create([
    'name' => 'Technology',
    'content' => 'Tech-related posts',
    'status' => PostCategory::ACTIVE,
]);

// Get all posts in category
$posts = $category->posts()->active()->get();
```

### Related Posts

```php
// Get random related posts (same categories)
$relatedPosts = $post->getRelatedByRandom(4);

// Get primary category
$primaryCategory = $post->getPrimaryCategory();

// Navigate posts within category
$previousPost = $post->getPrevious();
$nextPost = $post->getNext();
```

### Working with Images

```php
// Check if post has images
if ($post->hasImages()) {
    // Get primary image with preset
    $thumbnail = $post->getPrimaryImage(['preset' => 'thumb']);
}

// In Blade templates
@if($post->hasImages())
    <img src="{{ $post->getPrimaryImage(['preset' => 'thumb']) }}" alt="{{ $post->name }}">
@endif
```

### Querying Posts

```php
// Active posts only
$activePosts = Post::active()->get();

// Pending posts
$pendingPosts = Post::pending()->get();

// With eager loading
$posts = Post::active()
    ->with('categories', 'images')
    ->latest('date')
    ->paginate(10);
```

### Duplicating Posts

```php
$originalPost = Post::find(1);
$duplicatedPost = $originalPost->duplicate();
// Creates copy with new UUID, slug, pending status, and cloned relationships
```

### API Usage

**Authentication:** API endpoints require Sanctum authentication.

**List Posts:**
```bash
GET /api/posts
Authorization: Bearer {token}
```

**Response:**
```json
{
    "data": [
        {
            "id": 1,
            "uuid": "550e8400-e29b-41d4-a716-446655440000",
            "key": "my-first-blog-post",
            "slug": "my-first-blog-post",
            "name": "My First Blog Post",
            "summary": "Post summary...",
            "thumbnail": "/storage/images/thumb-post-1.jpg",
            "date": "2025-01-19T12:00:00.000000Z",
            "status": "active"
        }
    ]
}
```

## Configuration

After publishing the config file, you can customise URL prefixes:

```php
// config/post.php
return [
    'prefix' => 'posts',              // Frontend post URL prefix
    'category_prefix' => 'categories', // Frontend category URL prefix
];
```

## Routes

### Frontend Routes (Public)
- `GET /posts` - Post listing
- `GET /posts/{slug}` - Individual post
- `GET /categories` - Category listing
- `GET /categories/{slug}` - Posts in category

### Backend Routes (Admin)
- `GET /admin/posts` - Post management dashboard
- `GET /admin/posts/create` - Create new post
- `POST /admin/posts/store` - Store post
- `GET /admin/posts/{post}/edit` - Edit post
- `POST /admin/posts/{post}/update` - Update post
- `DELETE /admin/posts/{post}/delete` - Delete post
- `GET /admin/posts/{post}/duplicate` - Duplicate post
- Similar routes for categories at `/admin/categories`

### API Routes
- `GET /api/posts` - List posts (auth:sanctum)
- `GET /api/post-categories` - List categories (auth:sanctum)

## Events

The package fires the following events:

- `Bongo\Post\Events\PostCreated` - After post creation
- `Bongo\Post\Events\PostUpdated` - After post update
- `Bongo\Post\Events\PostDeleted` - After post deletion
- `Bongo\Post\Events\PostCategoryCreated` - After category creation
- `Bongo\Post\Events\PostCategoryUpdated` - After category update
- `Bongo\Post\Events\PostCategoryDeleted` - After category deletion

These events trigger:
- Automatic date/user assignment (on creation)
- Menu cache invalidation (on updates/deletion)
- Sitemap regeneration (on updates/deletion)

## Model Traits

Both `Post` and `PostCategory` models use these framework traits:

- `HasUUID` - Automatic UUID generation and indexing
- `HasKey` - URL-friendly key generation from name
- `HasStatus` - Status management (pending/active/inactive)
- `HasContent` - Content field helpers
- `HasSeo` - SEO metadata fields
- `HasHeaderClass` - Header styling options (sticky/transparent)
- `HasImages` - Image attachment system
- `SoftDeletes` - Laravel soft deletion

Additionally, `Post` uses:
- `HasRelated` - Related posts functionality

## Status Constants

Both models define status constants:

```php
Post::PENDING   // 'pending'
Post::ACTIVE    // 'active'
Post::INACTIVE  // 'inactive'
Post::INDEX     // 'index'
Post::NO_INDEX  // 'noindex'
```

## Testing

Run the test suite:

```bash
vendor/bin/phpunit
```

## Code Style

Fix code style issues:

```bash
vendor/bin/pint
```

Check code style without fixing:

```bash
vendor/bin/pint --test
```

## Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Comprehensive architecture documentation with class diagrams, database schema, and extension points
- **[CLAUDE.md](CLAUDE.md)** - Quick reference guide for AI assistants
- **[.cursorrules](.cursorrules)** - Cursor AI coding guidelines
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot instructions

## Dependencies

### Required Packages
- `bongo/framework` (^3.0) - Base framework and traits
- `bongo/image` (^3.0) - Image attachment system
- `bongo/menu` (^3.0) - Menu integration
- `bongo/package` (^3.0) - Package management
- `bongo/setting` (^3.0) - Settings system

### Optional Integrations
- `bongo/sitemap` - Automatic sitemap updates on content changes

## License

The MIT License (MIT). Please see [LICENSE](LICENSE) for more information.

## Author

**Stuart Elliott**
Email: stuart.elliott@bespokeuk.com
Website: [https://bespokeuk.com](https://bespokeuk.com)

## Support

For issues, questions, or contributions, please visit the [Bitbucket repository](https://bitbucket.org/designtec/post).
