# GitHub Copilot Instructions - Bongo Post Package

## Project Overview
Laravel package providing blog post and post category management with image attachments, SEO, event-driven cache updates, and API endpoints.

**Key Features:**
- Blog posts with categories (many-to-many relationship)
- Image attachments via `bongo/image` package
- SEO fields (meta title, description, canonical, index)
- Custom CSS/JS per post/category
- Soft deletes with audit trail
- Event-driven sitemap and menu cache updates
- Related posts by category
- Post duplication
- Frontend builder integration

## Key Classes and Relationships

### Models
**Post** (`src/Models/Post.php`)
```php
namespace Bongo\Post\Models;

class Post extends AbstractModel implements Imageable
{
    // Traits: HasUUID, HasKey, HasStatus, HasContent, HasSeo,
    //         HasHeaderClass, HasImages, HasRelated, SoftDeletes

    // Constants
    const PENDING = 'pending';
    const ACTIVE = 'active';
    const INACTIVE = 'inactive';

    // Key Methods
    public function categories(): BelongsToMany
    public function duplicate(): self
    public function getSummaryAttribute($dbValue): ?string

    // From HasRelated trait
    public function getRelated()
    public function getRelatedByRandom(int $limit = 4): Collection
    public function getPrimaryCategory(): ?PostCategory
    public function getPrevious(): ?Post
    public function getNext(): ?Post
}
```

**PostCategory** (`src/Models/PostCategory.php`)
```php
namespace Bongo\Post\Models;

class PostCategory extends AbstractModel implements Imageable
{
    // Traits: HasUUID, HasKey, HasStatus, HasContent, HasSeo,
    //         HasHeaderClass, HasImages, SoftDeletes

    public function posts(): BelongsToMany
}
```

### Controllers
**Backend\PostController** (`src/Http/Controllers/Backend/PostController.php`)
- Standard CRUD operations
- `duplicate(Post $post)` - Creates copy of post
- `syncPostCategories($post)` - Syncs many-to-many relationships
- Fires events: `PostCreated`, `PostUpdated`, `PostDeleted`

**Frontend\PostController** (`src/Http/Controllers/Frontend/PostController.php`)
- Public post display with SEO middleware
- Builder integration (edit/update with auth)
- Routes: `frontend.post.index`, `frontend.post.show`

**Api\PostController** (`src/Http/Controllers/Api/PostController.php`)
- API endpoint with Sanctum authentication
- Returns `PostResource` transformed data

### Events & Listeners
**Events:**
- `PostCreated`, `PostUpdated`, `PostDeleted`
- `PostCategoryCreated`, `PostCategoryUpdated`, `PostCategoryDeleted`

**Listeners:**
- `SetPostDate` - Auto-sets date on creation (queued)
- `SetPostUser` - Auto-sets user_id on creation (queued)
- `ClearMenuCache` (bongo/menu) - Clears menu cache on updates
- `UpdateSitemap` (bongo/sitemap) - Updates sitemap on changes

### Service Provider
**PostServiceProvider** (`src/PostServiceProvider.php`)
```php
class PostServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'post';

    protected array $composers = [
        PostComposer::class => ['post::backend.partials.dropdowns.post'],
        PostCategoryComposer::class => ['post::backend.category.partials.dropdowns.category'],
    ];

    protected array $listeners = [
        PostCreated::class => [SetPostDate::class, SetPostUser::class],
        PostUpdated::class => [ClearMenuCache::class, UpdateSitemap::class],
        // ...
    ];
}
```

## Code Style Templates

### Controller Method
```php
public function store(StorePostRequest $request): RedirectResponse
{
    $post = $this->post->create($request->all());
    $this->syncPostCategories($post);
    event(new PostCreated($post));

    return redirect()
        ->route('backend.post.show', $post->id)
        ->success(trans('post::backend.store_success'));
}
```

### Event Listener
```php
namespace Bongo\Post\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;

class YourListener implements ShouldQueue
{
    public int $tries = 3;

    public function handle($event): void
    {
        // Handle event
    }
}
```

### Form Request
```php
namespace Bongo\Post\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StorePostRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'name' => "required|unique:posts,slug,NULL,id,deleted_at,NULL|max:75",
            'status' => 'required|in:pending,active,inactive',
        ];
    }
}
```

### API Resource
```php
namespace Bongo\Post\Http\Resources;

use Illuminate\Http\Resources\Json\JsonResource;

class PostResource extends JsonResource
{
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'uuid' => $this->uuid,
            'slug' => $this->slug,
            'name' => $this->name,
            'thumbnail' => $this->hasImages()
                ? $this->getPrimaryImage(['preset' => 'thumb'])
                : '/images/default_image.png',
        ];
    }
}
```

### View Composer
```php
namespace Bongo\Post\Http\ViewComposers;

use Illuminate\View\View;

class PostComposer
{
    public function compose(View $view): void
    {
        $posts = Post::active()->orderBy('name')->pluck('name', 'id');

        $view->with(compact('posts'));
    }
}
```

## Common Patterns

### Query Scopes (from Traits)
```php
// From HasStatus trait
Post::active()->get();           // status = 'active'
Post::pending()->get();          // status = 'pending'
Post::inactive()->get();         // status = 'inactive'

// Combined with relationships
Post::active()->with('categories', 'images')->get();
```

### Related Posts
```php
// Random related posts in same categories
$related = $post->getRelatedByRandom(4);

// Navigation within category context
$previous = $post->getPrevious();
$next = $post->getNext();

// Primary category
$category = $post->getPrimaryCategory();
```

### Image Handling
```php
// Check if has images
if ($post->hasImages()) {
    $thumbnail = $post->getPrimaryImage(['preset' => 'thumb']);
}

// Upload endpoint
Route::post('image/{uuid}', [PostImageController::class, 'upload']);
```

### Duplication
```php
// Duplicate post with new identifiers
$newPost = $post->duplicate();
// Returns new Post with:
// - Fresh UUID, slug, key
// - Current user as author
// - Pending status
// - Cloned categories and images
```

### Event Firing
```php
// Always fire events after persistence
$post = $this->post->create($request->all());
event(new PostCreated($post));

$post->update($request->all());
event(new PostUpdated($post));

if ($post->delete()) {
    event(new PostDeleted($post));
}
```

### Category Sync
```php
// Sync many-to-many relationship
private function syncPostCategories($post)
{
    $categoryIds = request('category_ids');
    if (isset($categoryIds)) {
        $post->categories()->sync($categoryIds);
    }
}
```

## Route Patterns

### Backend Routes (admin panel)
```php
// Prefix: config('post.prefix') - default 'posts'
// Named: backend.post.*
// Middleware: auth + employee

Route::as('post.')
    ->prefix(config('post.prefix'))
    ->group(function () {
        Route::get('/', [PostController::class, 'index'])->name('index');
        Route::get('create', [PostController::class, 'create'])->name('create');
        Route::post('store', [PostController::class, 'store'])->name('store');
        Route::get('datatable', [PostDatatableController::class, 'index'])->name('datatable');

        Route::prefix('{post}')->group(function () {
            Route::get('/', [PostController::class, 'show'])->name('show');
            Route::get('edit', [PostController::class, 'edit'])->name('edit');
            Route::post('update', [PostController::class, 'update'])->name('update');
            Route::delete('delete', [PostController::class, 'destroy'])->name('destroy');
            Route::get('duplicate', [PostController::class, 'duplicate'])->name('duplicate');
        });
    });
```

### Frontend Routes (public)
```php
// Middleware: hasRedirects, hasShortCodes, minifyHtml

Route::as('post.')
    ->prefix(config('post.prefix'))
    ->group(function () {
        Route::get('{slug}', [PostController::class, 'show'])->name('show');
        Route::get('/', [PostController::class, 'index'])->name('index');
    });
```

### API Routes
```php
// Prefix: /api/posts
// Middleware: auth:sanctum

Route::prefix('posts')
    ->as('post.')
    ->group(function () {
        Route::get('/', [PostController::class, 'index'])->name('index');
    });
```

## Database Schema

### Posts Table
```
id, uuid, user_id, date, name, key, slug, content, summary, status,
meta_title, meta_description, meta_canonical, meta_index,
css, js, schema,
transparent_header, sticky_header,
created_by, updated_by, deleted_by,
created_at, updated_at, deleted_at
```

### Post Categories Table
```
id, uuid, name, key, slug, content, status,
meta_title, meta_description, meta_canonical, meta_index,
css, js,
transparent_header, sticky_header,
created_by, updated_by, deleted_by,
created_at, updated_at, deleted_at
```

### Pivot Table
```
post_categories_pivot: post_id, post_category_id
```

## Configuration
File: `src/Config/post.php`
```php
return [
    'prefix' => 'posts',              // URL prefix for routes
    'category_prefix' => 'categories', // URL prefix for category routes
];
```

## Translation Namespace
- Namespace: `post::`
- Example: `trans('post::backend.store_success')`
- Files in: `src/Translations/`

## Testing
- Test suite: `vendor/bin/phpunit`
- Environment: SQLite (testing connection)
- Structure: `tests/Feature/`, `tests/Unit/`

## Dependencies
- `bongo/framework` - Base classes, traits, AbstractServiceProvider
- `bongo/image` - Image attachment system
- `bongo/menu` - Menu cache integration
- `bongo/package` - Package management
- `bongo/setting` - Settings system
