# CLAUDE.md - Bongo Post Package

## Overview
This Laravel package provides blog post and post category management for the Bongo CMS. It supports image attachments, SEO metadata, event-driven cache updates, and multi-channel content delivery.

**Key Features:**
- Blog posts with publication dates and categorisation
- Many-to-many relationships between posts and categories
- Image attachment system via `bongo/image`
- SEO metadata (title, description, canonical, indexing)
- Custom CSS/JS per post/category
- Soft deletion with audit trail
- Post duplication with relationship cloning
- Related posts by shared categories
- Event-driven sitemap and menu cache invalidation
- Frontend builder integration for in-place editing

## Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Detailed architecture documentation with class diagrams, database schema, lifecycle flows, and extension points
- **[.cursorrules](.cursorrules)** - Cursor AI instructions with coding conventions, common tasks, and architecture patterns
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot instructions with code templates and patterns
- **[README.md](README.md)** - General package information and installation instructions

## Quick Reference

### Requirements
- PHP 8.2+
- Laravel 10+
- Bongo framework packages (framework, image, menu, package, setting)

### Installation
```bash
composer require bongo/post
```

Service provider auto-discovery will register `Bongo\Post\PostServiceProvider`.

### Commands

```bash
# Run tests
vendor/bin/phpunit

# Fix code style
vendor/bin/pint

# Check code style without fixing
vendor/bin/pint --test
```

## Architecture Quick Reference

### Extends Bongo Framework
This package extends `Bongo\Framework\Providers\AbstractServiceProvider` which provides:

**Automatic Bootstrapping:**
- Config from `src/Config/post.php`
- Routes from `src/Routes/` with middleware:
  - `api.php` → 'api.*' prefix, `auth:sanctum` middleware
  - `backend.php` → 'backend.*' prefix, `auth` + `employee` middleware
  - `frontend.php` → 'frontend.*' named routes
- Views from `src/Views/post/`
- Migrations from `src/Migrations/`
- Translations from `src/Translations/`
- Event listeners via `$listeners` array
- View composers via `$composers` array

**Service Provider Configuration:**
```php
class PostServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'post';

    protected array $composers = [
        PostComposer::class => ['post::backend.partials.dropdowns.post'],
        PostCategoryComposer::class => ['post::backend.category.partials.dropdowns.category'],
    ];

    protected array $listeners = [
        PostCreated::class => [SetPostDate::class, SetPostUser::class],
        PostUpdated::class => [ClearMenuCache::class, UpdateSitemap::class],
        PostDeleted::class => [ClearMenuCache::class, UpdateSitemap::class],
        PostCategoryUpdated::class => [ClearMenuCache::class, UpdateSitemap::class],
        PostCategoryDeleted::class => [ClearMenuCache::class, UpdateSitemap::class],
    ];
}
```

### Models

**Post** (`src/Models/Post.php`)
- Extends: `Bongo\Framework\Models\AbstractModel`
- Implements: `Bongo\Image\Interfaces\Imageable`
- Traits: `HasUUID`, `HasKey`, `HasStatus`, `HasContent`, `HasSeo`, `HasHeaderClass`, `HasImages`, `HasRelated`, `SoftDeletes`
- Key methods:
  - `categories(): BelongsToMany`
  - `duplicate(): self` - Clones post with new identifiers
  - `getSummaryAttribute(): ?string` - Auto-generates summary if empty
  - `getRelated()` - Query builder for related posts
  - `getRelatedByRandom(int $limit = 4): Collection`
  - `getPrimaryCategory(): ?PostCategory`
  - `getPrevious(): ?Post`, `getNext(): ?Post`

**PostCategory** (`src/Models/PostCategory.php`)
- Extends: `Bongo\Framework\Models\AbstractModel`
- Implements: `Bongo\Image\Interfaces\Imageable`
- Traits: `HasUUID`, `HasKey`, `HasStatus`, `HasContent`, `HasSeo`, `HasHeaderClass`, `HasImages`, `SoftDeletes`
- Key methods:
  - `posts(): BelongsToMany`

### Routes

**Frontend Routes** (prefix: configurable, default `posts`)
- `frontend.post.index` - GET `/posts`
- `frontend.post.show` - GET `/posts/{slug}`
- `frontend.post.edit` - GET `/posts/edit/{uuid}` (auth + employee)
- `frontend.post.update` - POST `/posts/update/{uuid}` (auth + employee)
- `frontend.post.image` - POST `/posts/image/{uuid}`
- `frontend.post.background_image` - POST `/posts/background/{uuid}`

**Backend Routes** (prefix: configurable, default `posts`)
- `backend.post.index`, `create`, `store`, `show`, `edit`, `update`, `destroy`
- `backend.post.duplicate` - GET `/admin/posts/{post}/duplicate`
- `backend.post.datatable` - GET `/admin/posts/datatable`

**API Routes**
- `api.post.index` - GET `/api/posts` (auth:sanctum)
- `api.post_category.index` - GET `/api/post-categories` (auth:sanctum)

### Events & Listeners

**Events:**
- `PostCreated`, `PostUpdated`, `PostDeleted`
- `PostCategoryCreated`, `PostCategoryUpdated`, `PostCategoryDeleted`

**Listeners:**
- `SetPostDate` - Auto-sets date on creation (queued, 3 tries)
- `SetPostUser` - Auto-sets user_id on creation (queued, 3 tries)
- `ClearMenuCache` - Clears menu cache on updates (bongo/menu)
- `UpdateSitemap` - Regenerates sitemap on updates (bongo/sitemap)

### Controllers

**Backend:**
- `PostController` - CRUD operations, duplicate, fires events
- `PostDatatableController` - Server-side DataTables processing
- `PostImageController` - Admin image uploads
- `PostCategoryController` - Category CRUD
- `PostCategoryDatatableController` - Category DataTables
- `PostCategoryImageController` - Category image uploads

**Frontend:**
- `PostController` - Public index/show, builder edit/update
- `PostImageController` - Builder image upload
- `PostBackgroundImageController` - Builder background upload
- `PostCategoryController` - Category public index/show

**API:**
- `PostController` - API index with `PostResource`
- `PostCategoryController` - Category API index with `PostCategoryResource`

## Key Files

| File | Purpose |
|------|---------|
| `src/PostServiceProvider.php` | Service provider extending AbstractServiceProvider |
| `src/Models/Post.php` | Post model with Imageable, 9 traits, duplicate method |
| `src/Models/PostCategory.php` | Category model with Imageable, 7 traits |
| `src/Traits/HasRelated.php` | Related posts functionality (getRelated, getPrevious, getNext) |
| `src/Config/post.php` | URL prefix configuration |
| `src/Routes/frontend.php` | Public routes with SEO middleware |
| `src/Routes/backend.php` | Admin routes with auth + employee middleware |
| `src/Routes/api.php` | API routes with auth:sanctum middleware |
| `src/Http/Controllers/Backend/PostController.php` | Main admin controller with CRUD + duplicate |
| `src/Http/Controllers/Frontend/PostController.php` | Public controller + builder integration |
| `src/Http/Requests/StorePostRequest.php` | Validates unique slug, max 75 chars |
| `src/Events/PostCreated.php` | Fired after post creation |
| `src/Listeners/SetPostDate.php` | Auto-sets date on creation (queued) |
| `src/Listeners/SetPostUser.php` | Auto-sets user_id on creation (queued) |
| `src/Http/Resources/PostResource.php` | API resource transformation |
| `src/Http/ViewComposers/PostCategoryComposer.php` | Provides categories for dropdowns |

## Code Style Summary

### Naming Conventions
- Controllers: Singular + Controller (e.g., `PostController`)
- Models: Singular PascalCase (e.g., `Post`, `PostCategory`)
- Tables: Plural snake_case (e.g., `posts`, `post_categories`)
- Routes: Dot notation (e.g., `backend.post.show`)
- Events: Past tense verb + model (e.g., `PostCreated`)
- Listeners: Descriptive action (e.g., `SetPostDate`)

### Return Types
Always declare return types on public methods:
- Controllers: `View`, `RedirectResponse`, `JsonResponse`
- Models: Specific types (`?PostCategory`, `Collection`, `self`)

### Status Constants
Use model constants, never magic strings:
```php
Post::PENDING   // 'pending'
Post::ACTIVE    // 'active'
Post::INACTIVE  // 'inactive'
Post::INDEX     // 'index'
Post::NO_INDEX  // 'noindex'
```

### Controller Patterns
```php
// Store/Update pattern
public function store(StorePostRequest $request): RedirectResponse
{
    $post = $this->post->create($request->all());
    $this->syncPostCategories($post);
    event(new PostCreated($post));

    return redirect()
        ->route('backend.post.show', $post->id)
        ->success(trans('post::backend.store_success'));
}

// Destroy pattern
public function destroy(Post $post)
{
    if ($post->delete()) {
        event(new PostDeleted($post));
        return redirect()->success(trans('post::backend.delete_success'));
    }
    return back()->error(trans('post::backend.delete_failed'));
}
```

### Query Scopes (from Traits)
```php
Post::active()->get();              // Only active posts
Post::pending()->get();             // Only pending posts
Post::inactive()->get();            // Only inactive posts

// With eager loading
Post::active()->with('categories', 'images')->get();
```

### Related Posts
```php
$related = $post->getRelatedByRandom(4);  // Random related posts
$previous = $post->getPrevious();          // Previous in category
$next = $post->getNext();                  // Next in category
$category = $post->getPrimaryCategory();   // First category
```

### Image Handling
```php
if ($post->hasImages()) {
    $thumbnail = $post->getPrimaryImage(['preset' => 'thumb']);
}
```

### Duplication
```php
$newPost = $post->duplicate();  // Clones with new UUID, slug, status=pending
```

## Configuration

**File:** `src/Config/post.php`
```php
return [
    'prefix' => 'posts',              // URL prefix for post routes
    'category_prefix' => 'categories', // URL prefix for category routes
];
```

## Database Schema

**Posts Table:** `id`, `uuid`, `user_id`, `date`, `name`, `slug`, `key`, `content`, `summary`, `status`, `meta_title`, `meta_description`, `meta_canonical`, `meta_index`, `css`, `js`, `schema`, `transparent_header`, `sticky_header`, audit fields, timestamps, soft deletes

**Post Categories Table:** Similar to posts but without `date`, `summary`, `schema`

**Pivot Table:** `post_categories_pivot` (post_id, post_category_id)

## Common Tasks

### Add New Post Field
1. Create migration in `src/Migrations/`
2. Add to `Post::$fillable`
3. Update form request validation
4. Update views (backend edit, frontend show)
5. Add to `PostResource::toArray()` if exposing via API

### Create New Event Listener
1. Create class in `src/Listeners/`
2. Implement `ShouldQueue` for async, add `public int $tries = 3;`
3. Register in `PostServiceProvider::$listeners`

### Add Custom Route
1. Add to appropriate file in `src/Routes/`
2. Follow naming convention: `{area}.post.{action}`
3. Routes auto-registered by AbstractServiceProvider

### Extend Post Queries
Use framework trait scopes:
- `Post::active()`, `Post::pending()`, `Post::inactive()`
- Combine with eager loading: `Post::active()->with('categories', 'images')->get()`

## Translation Namespace
- Namespace: `post::`
- Example: `trans('post::backend.store_success')`
- Files in: `src/Translations/`

## Testing
```bash
vendor/bin/phpunit  # Run test suite (SQLite DB_CONNECTION=testing)
```

## Dependencies
- `bongo/framework` (^3.0) - Base framework, AbstractServiceProvider, traits
- `bongo/image` (^3.0) - Image attachment system
- `bongo/menu` (^3.0) - Menu cache clearing
- `bongo/package` (^3.0) - Package management
- `bongo/setting` (^3.0) - Settings system
