<?php

namespace Bongo\Post\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Post\Events\PostCategoryCreated;
use Bongo\Post\Events\PostCategoryDeleted;
use Bongo\Post\Events\PostCategoryUpdated;
use Bongo\Post\Http\Requests\StorePostCategoryRequest;
use Bongo\Post\Http\Requests\UpdatePostCategoryRequest;
use Bongo\Post\Models\PostCategory;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class PostCategoryController extends AbstractController
{
    protected PostCategory $postCategory;

    public function __construct(PostCategory $postCategory)
    {
        $this->postCategory = $postCategory;
    }

    public function index(): View
    {
        return view('post::backend.category.index');
    }

    public function create(): View
    {
        $postCategory = $this->postCategory;

        return view('post::backend.category.create', compact('postCategory'));
    }

    public function store(StorePostCategoryRequest $request): RedirectResponse
    {
        $postCategory = $this->postCategory->create($request->all());
        $this->syncPosts($postCategory);
        event(new PostCategoryCreated($postCategory));

        return redirect()
            ->route('backend.post_category.show', $postCategory->id)
            ->success(trans('post::backend.category_store_success'));
    }

    public function show(PostCategory $postCategory): View
    {
        return view('post::backend.category.show', compact('postCategory'));
    }

    public function edit(PostCategory $postCategory): View
    {
        return view('post::backend.category.edit', compact('postCategory'));
    }

    public function update(UpdatePostCategoryRequest $request, PostCategory $postCategory): RedirectResponse
    {
        $postCategory->update($request->all());
        $this->syncPosts($postCategory);
        event(new PostCategoryUpdated($postCategory));

        return redirect()
            ->route('backend.post_category.show', $postCategory->id)
            ->success(trans('post::backend.category_update_success'));
    }

    public function destroy(PostCategory $postCategory)
    {
        if ($postCategory->delete()) {
            event(new PostCategoryDeleted($postCategory));
            $redirect = back();

            // If the post was deleted from their detail view,
            // we'll redirect back to the post list view.
            if (route('backend.post_category.show', $postCategory) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.post_category.index');
            }

            return $redirect
                ->success(trans('post::backend.category_delete_success'));
        }

        return back()
            ->error(trans('post::backend.category_delete_failed'));
    }

    private function syncPosts($postCategory)
    {
        $postIds = request('post_ids');
        if (isset($postIds)) {
            $postCategory->posts()->sync($postIds);
        }
    }
}
