<?php

namespace Bongo\Post\Models;

use Bongo\Framework\Casts\Date;
use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasContent;
use Bongo\Framework\Traits\HasHeaderClass;
use Bongo\Framework\Traits\HasKey;
use Bongo\Framework\Traits\HasSeo;
use Bongo\Framework\Traits\HasStatus;
use Bongo\Framework\Traits\HasUUID;
use Bongo\Image\Interfaces\Imageable;
use Bongo\Image\Traits\HasImages;
use Bongo\Post\Traits\HasRelated;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Post extends AbstractModel implements Imageable
{
    // status
    public const PENDING = 'pending';
    public const ACTIVE = 'active';
    public const INACTIVE = 'inactive';

    // index
    public const INDEX = 'index';
    public const NO_INDEX = 'noindex';

    // booleans
    public const ENABLED = 1;
    public const DISABLED = 0;

    // Traits
    use HasContent;
    use HasHeaderClass;
    use HasImages;
    use HasKey;
    use HasRelated;
    use HasSeo;
    use HasStatus;
    use HasUUID;
    use SoftDeletes;

    protected $casts = [
        'date' => Date::class,
    ];

    protected $fillable = [
        'date',
        'name',
        'slug',
        'content',
        'summary',
        'status',
        'meta_title',
        'meta_description',
        'meta_canonical',
        'meta_index',
        'css',
        'js',
        'schema',
    ];

    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(PostCategory::class, 'post_categories_pivot');
    }

    public function getSummaryAttribute($dbValue): ?string
    {
        if (! empty($dbValue)) {
            return $dbValue;
        }

        return $this->getSummary();
    }

    public function duplicate(): self
    {
        // Copy attributes and update unique fields
        $newItem = $this->replicate();
        $newItem->uuid = (string) Str::uuid();
        $newItem->user_id = user()->id;
        $newItem->date = Carbon::now()->format('Y-m-d H:i:s');
        $newItem->name = $this->name.' '.Str::random(6);
        $newItem->slug = Str::slug(Str::limit($newItem->name, 75));
        $newItem->key = $newItem->formatKey($newItem->name);
        $newItem->status = self::PENDING;
        $newItem->save();

        // Reset & re-load the relations
        $this->relations = [];
        $this->load('categories', 'images');

        // Clone category relations
        if (isset($this->categories) && count($this->categories) > 0) {
            foreach ($this->categories as $category) {
                $newItem->categories()->sync($category->id);
            }
        }

        // Clone image relations
        if (isset($this->images) && count($this->images) > 0) {
            foreach ($this->images as $image) {
                $newItem->images()->sync($image->id);
            }
        }

        return $newItem;
    }
}
