<?php

namespace Bongo\Post\Traits;

use Bongo\Post\Models\Post;
use Bongo\Post\Models\PostCategory;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Session;

trait HasRelated
{
    public function getRelated()
    {
        $this->loadMissing('categories');

        return Post::query()
            ->active()
            ->join('post_categories_pivot', 'posts.id', '=', 'post_categories_pivot.post_id')
            ->where('posts.id', '!=', $this->id)
            ->whereIn('post_category_id', $this->categories->pluck('id')->toArray());
    }

    public function getRelatedByRandom(int $limit = 4): Collection
    {
        $this->loadMissing('categories');

        // Get a list of active post ids, where they are within the same category as this post
        $relatedPostIds = $this->getRelated()
            ->take($limit)
            ->inRandomOrder()
            ->pluck('posts.id');

        return Post::with('images')->find($relatedPostIds);
    }

    public function getPrimaryCategory(): ?PostCategory
    {
        return $this->categories()->first();
    }

    public function getPrevious(): ?Post
    {
        $this->loadMissing('categories');

        $selectedCategory = Session::get('selected_post_category', $this->getPrimaryCategory());

        if ($selectedCategory) {
            return $selectedCategory
                ->posts()
                ->active()
                ->with('images')
                ->where('posts.id', '<', $this->id)
                ->orderBy('posts.id', 'DESC')
                ->first();
        }

        return null;
    }

    public function getNext(): ?Post
    {
        $this->loadMissing('categories');

        $selectedCategory = Session::get('selected_post_category', $this->getPrimaryCategory());

        if ($selectedCategory) {
            return $selectedCategory
                ->posts()
                ->active()
                ->with('images')
                ->where('posts.id', '>', $this->id)
                ->orderBy('posts.id', 'ASC')
                ->first();
        }

        return null;
    }
}
