<?php

namespace Bongo\Post\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Post\Events\PostCreated;
use Bongo\Post\Events\PostDeleted;
use Bongo\Post\Events\PostUpdated;
use Bongo\Post\Http\Requests\StorePostRequest;
use Bongo\Post\Http\Requests\UpdatePostRequest;
use Bongo\Post\Models\Post;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class PostController extends AbstractController
{
    protected Post $post;

    public function __construct(Post $post)
    {
        $this->post = $post;
    }

    public function index(): View
    {
        return view('post::backend.index');
    }

    public function create(): View
    {
        $post = $this->post;

        return view('post::backend.create', compact('post'));
    }

    public function store(StorePostRequest $request): RedirectResponse
    {
        $post = $this->post->create($request->all());
        $this->syncPostCategories($post);
        event(new PostCreated($post));

        return redirect()
            ->route('backend.post.show', $post->id)
            ->success(trans('post::backend.store_success'));
    }

    public function show(Post $post): View
    {
        return view('post::backend.show', compact('post'));
    }

    public function edit(Post $post): View
    {
        return view('post::backend.edit', compact('post'));
    }

    public function update(UpdatePostRequest $request, Post $post): RedirectResponse
    {
        $post->update($request->all());
        $this->syncPostCategories($post);
        event(new PostUpdated($post));

        return redirect()
            ->route('backend.post.show', $post->id)
            ->success(trans('post::backend.update_success'));
    }

    public function destroy(Post $post)
    {
        if ($post->delete()) {
            event(new PostDeleted($post));
            $redirect = back();

            // If the post was deleted from their detail view,
            // we'll redirect back to the post list view.
            if (route('backend.post.show', $post) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.post.index');
            }

            return $redirect->success(trans('post::backend.delete_success'));
        }

        return back()->error(trans('post::backend.delete_failed'));
    }

    public function duplicate(Post $post)
    {
        $newPost = $post->duplicate();

        return redirect()
            ->route('backend.post.edit', $newPost->id)
            ->success(trans('post::backend.update_success'));
    }

    private function syncPostCategories($post)
    {
        $categoryIds = request('category_ids');
        if (isset($categoryIds)) {
            $post->categories()->sync($categoryIds);
        }
    }
}
